# (c) Copyright 2010-2015. CodeWeavers, Inc.

import os
import signal
import subprocess
import sys
import traceback

CHECK_SIGNALS = 1
CHECK_VTE = 2

OK = 0
MISSING_GIR = 1
BROKEN = 2
NO_DISPLAY = 3
MISSING_PYGI = 4
MISSING_VTE = 5

init_check_ret = None

def init_check():
    # pylint: disable=W0603
    global init_check_ret

    # Make sure Gtk.init_check() is only called once.
    if init_check_ret is not None:
        return init_check_ret

    try:
        from gi.repository import Gtk
        init_check_ret, _argv = Gtk.init_check()
    except ImportError:
        # We didn't run Gtk.init_check(), so we can try again later
        pass

    return init_check_ret

def handle_interrupt():
    from gi.repository import Gtk
    if not Gtk.Application.get_default():
        Gtk.main_quit()

def install_interrupt_handler():
    from gi.repository import GLib
    try:
        GLib.unix_signal_add(GLib.PRIORITY_HIGH, signal.SIGINT, handle_interrupt)
    except AttributeError:
        import cxlog
        cxlog.warn("Error setting interrupt handler: %s\n" % traceback.format_exc())

def has_api(name, version):
    import cxlog

    log_name = name
    if version:
        log_name = "%s %s" % (name, version)

    try:
        import gi
        if version:
            gi.require_version(name, version)
        else:
            gi.require_foreign(name)
        return True
    except ValueError:
        cxlog.log("%s is missing %s support" % (sys.executable, log_name))
        return False
    except ImportError:
        cxlog.log("%s is missing %s support" % (sys.executable, log_name))
        return False
    except Exception: # pylint: disable=W0703
        cxlog.log("%s failed to require %s:\n%s" % (sys.executable, log_name, traceback.format_exc()))
        traceback.print_exc()
        return False

def status(extras=CHECK_SIGNALS):
    import cxlog
    import warnings
    warnings.filterwarnings("ignore", "", Warning)
    try:
        try:
            # pylint: disable=F0401,W0611,W0612
            import gi
        except ImportError:
            cxlog.log("%s is missing GObject introspection support" % sys.executable)
            return MISSING_PYGI

        # This has a desired side-effect of making sure we don't get Gtk2
        # or something.
        for (name, version) in (('Gtk', '3.0'),
                                ('Gdk', '3.0'),
                                ('GdkPixbuf', '2.0'),
                                ('GObject', '2.0'),
                                ('Pango', '1.0'),
                                ('GLib', '2.0'),
                                ('cairo', None)):
            if not has_api(name, version):
                return MISSING_GIR

        from gi.repository import Gtk

        if extras & CHECK_SIGNALS:
            try:
                # Passing an object to connect_signals() fails if using the
                # native GTK implementation through the GNOME introspection
                # code instead of the Python-specific implementation provided
                # in a separate package.
                builder = Gtk.Builder()
                builder.connect_signals(builder)
            except ValueError:
                return MISSING_GIR

        if (extras & CHECK_VTE) and not has_api('Vte', '2.91'):
            return MISSING_VTE

    finally:
        warnings.resetwarnings()

    # Disable a warning when editing entry text
    # Warning: g_value_get_int: assertion 'G_VALUE_HOLDS_INT (value)' failed
    # Upstream bug: https://gitlab.gnome.org/GNOME/pygobject/issues/12
    warnings.filterwarnings('ignore', '.*g_value_get_int.*')

    try:
        if not init_check():
            # Assume if this much works, it's a display issue
            return NO_DISPLAY
    except RuntimeError as run_error:
        if str(run_error) == "could not open display":
            cxlog.log("could not open display")
            return NO_DISPLAY
        traceback.print_exc()
        cxlog.log("Error opening display: %s\n" % traceback.format_exc())
        return BROKEN
    except Exception: # pylint: disable=W0703
        traceback.print_exc()
        cxlog.log("Error opening display: %s\n" % traceback.format_exc())
        return BROKEN

    from gi.repository import GLib
    # Set up a signal handler only if someone starts a GLib mainloop
    GLib.idle_add(install_interrupt_handler, priority=GLib.PRIORITY_HIGH)

    return OK

def ensure_present_gtk(extras=CHECK_SIGNALS):
    gtk_status = status(extras)
    if gtk_status in (MISSING_PYGI, MISSING_GIR, BROKEN):
        # GTK+ is missing or broken so try with another version of Python,
        # hoping to have better luck. Make sure to try the system Python
        # binaries in case there is a broken / incomplete Python installation
        # in the path (e.g. Anaconda).
        for candidate in ('python3', '/usr/bin/python3'):
            try:
                # Run this module with other Python versions to see if they
                # pass the test.
                popen = subprocess.Popen([candidate, __file__]) # pylint: disable=R1732
                if popen.wait() in (OK, NO_DISPLAY):
                    # Note that it's better to issue a $DISPLAY error than
                    # send the user on a wild goose chase for a gtk module.
                    os.execvp(candidate, [candidate] + sys.argv)
            except Exception: # pylint: disable=W0703
                # Let the caller deal with the fallout
                pass
    return gtk_status

def check_gtk(extras=CHECK_SIGNALS, warn_gtk=True, warn_display=True):
    gtk_status = ensure_present_gtk(extras)
    if warn_gtk and (gtk_status in (MISSING_PYGI, MISSING_GIR, BROKEN)):
        import cxfixes
        if gtk_status != MISSING_GIR:
            cxfixes.add_error('pygi', level='required')
        cxfixes.add_error('gir-data', level='required')
        cxfixes.fix_errors()
        gtk_status = status(extras)
        if gtk_status in (MISSING_PYGI, MISSING_GIR, BROKEN):
            if gtk_status != MISSING_GIR:
                cxfixes.add_error('pygi', level='required')
            cxfixes.add_error('gir-data', level='required')
            cxfixes.report_errors()
    if warn_display and gtk_status == NO_DISPLAY:
        import cxlog
        cxlog.err("Verify $DISPLAY: it is not set, invalid or you are not allowed to access it")
    return gtk_status

if __name__ == '__main__':
    sys.exit(status(CHECK_SIGNALS | CHECK_VTE))
