/* SPDX-License-Identifier: GPL-2.0 */
#ifndef _BCACHEFS_MOVE_H
#define _BCACHEFS_MOVE_H

#include "bcachefs_ioctl.h"
#include "alloc/buckets.h"
#include "btree/bbpos.h"
#include "btree/iter.h"
#include "data/update.h"
#include "move_types.h"

struct bch_read_bio;

struct moving_context {
	struct btree_trans	*trans;
	struct list_head	list;
	void			*fn;

	struct bch_ratelimit	*rate;
	struct bch_move_stats	*stats;
	struct write_point_specifier wp;
	bool			wait_on_copygc;

	/* For waiting on outstanding reads and writes: */
	struct closure		cl;

	struct mutex		lock;
	struct list_head	reads;
	struct list_head	ios;

	/* in flight sectors: */
	atomic_t		read_sectors;
	atomic_t		write_sectors;
	atomic_t		read_ios;
	atomic_t		write_ios;

	wait_queue_head_t	wait;
};

#define move_ctxt_wait_event_timeout(_ctxt, _cond, _timeout)			\
({										\
	int _ret = 0;								\
	while (true) {								\
		bool cond_finished = false;					\
		bch2_moving_ctxt_do_pending_writes(_ctxt);			\
										\
		if (_cond)							\
			break;							\
		bch2_trans_unlock_long((_ctxt)->trans);				\
		_ret = __wait_event_timeout((_ctxt)->wait,			\
			     bch2_moving_ctxt_next_pending_write(_ctxt) ||	\
			     (cond_finished = (_cond)), _timeout);		\
		if (_ret || ( cond_finished))					\
			break;							\
	}									\
	_ret;									\
})

#define move_ctxt_wait_event(_ctxt, _cond)				\
do {									\
	bool cond_finished = false;					\
	bch2_moving_ctxt_do_pending_writes(_ctxt);			\
									\
	if (_cond)							\
		break;							\
	bch2_trans_unlock_long((_ctxt)->trans);				\
	__wait_event((_ctxt)->wait,					\
		     bch2_moving_ctxt_next_pending_write(_ctxt) ||	\
		     (cond_finished = (_cond)));			\
	if (cond_finished)						\
		break;							\
} while (1)

typedef int (*move_pred_fn)(struct btree_trans *, void *, enum btree_id, struct bkey_s_c,
			    struct bch_inode_opts *, struct data_update_opts *);

extern const char * const bch2_data_ops_strs[];

void bch2_moving_ctxt_exit(struct moving_context *);
void bch2_moving_ctxt_init(struct moving_context *, struct bch_fs *,
			   struct bch_ratelimit *, struct bch_move_stats *,
			   struct write_point_specifier, bool);
struct data_update *bch2_moving_ctxt_next_pending_write(struct moving_context *);
void bch2_moving_ctxt_do_pending_writes(struct moving_context *);
void bch2_moving_ctxt_flush_all(struct moving_context *);
void bch2_move_ctxt_wait_for_io(struct moving_context *);
int bch2_move_ratelimit(struct moving_context *);

int bch2_move_extent(struct moving_context *,
		     struct move_bucket *,
		     struct bch_inode_opts *,
		     struct data_update_opts *,
		     struct btree_iter *, unsigned, struct bkey_s_c);

int bch2_move_data_btree(struct moving_context *, struct bpos, struct bpos,
			 move_pred_fn, void *, enum btree_id, unsigned);

int bch2_move_data_phys(struct bch_fs *, unsigned, u64, u64, unsigned,
			struct bch_ratelimit *, struct bch_move_stats *,
			struct write_point_specifier, bool,
			move_pred_fn, void *);

int bch2_evacuate_data(struct moving_context *, unsigned, u64, u64);

int bch2_evacuate_bucket(struct moving_context *,
			   struct move_bucket *,
			   struct bpos, int,
			   struct data_update_opts);
int bch2_data_job(struct bch_fs *,
		  struct bch_move_stats *,
		  struct bch_ioctl_data *);

void bch2_move_stats_to_text(struct printbuf *, struct bch_move_stats *);
void bch2_move_stats_exit(struct bch_move_stats *, struct bch_fs *);
void bch2_move_stats_init(struct bch_move_stats *, const char *);

void bch2_fs_moving_ctxts_to_text(struct printbuf *, struct bch_fs *);

void bch2_fs_move_init(struct bch_fs *);

#endif /* _BCACHEFS_MOVE_H */
