# frozen_string_literal: true

require File.expand_path("../../test_helper", File.dirname(__FILE__))

class ReportHTMLTest < Minitest::Test
  def setup
    super
    @store = Coverband.configuration.store
    Coverband.configure do |config|
      config.store = @store
      config.root = fixtures_root
      config.ignore = ["notsomething.rb", "lib/*"]
    end
    mock_file_hash
  end

  test "generate dynamic content hosted html report" do
    @store.send(:save_report, basic_coverage)

    html = Coverband::Reporters::HTMLReport.new(@store,
      open_report: false).report
    assert_match "Generated by", html
  end

  test "files with no Coverage but in project are shown in reports" do
    @store.send(:save_report, basic_source_fixture_coverage)

    html = Coverband::Reporters::HTMLReport.new(@store,
      open_report: false).report
    assert_match "sample.rb", html
    # in project, but not in coverage data
    assert_match "app/models/user.rb", html
  end

  test "files with no Coverage but in project details page list warning" do
    @store.send(:save_report, basic_coverage_full_path)

    basic_coverage_file_full_path
    base_path = Dir.pwd
    # in project, but not in coverage data
    html = Coverband::Reporters::HTMLReport.new(Coverband.configuration.store,
      filename: "#{Dir.pwd}/test/fixtures/app/models/user.rb",
      base_path: base_path,
      open_report: false).file_details
    assert_match "This file was never loaded", html
  end

  test "generate dynamic content detailed file report" do
    @store.send(:save_report, basic_coverage_full_path)

    filename = basic_coverage_file_full_path
    base_path = "/coverage"
    html = Coverband::Reporters::HTMLReport.new(Coverband.configuration.store,
      filename: filename,
      base_path: base_path,
      open_report: false).file_details
    assert_match "Coverage first seen", html
  end

  test "generate dynamic content detailed file report handles missing file" do
    @store.send(:save_report, basic_coverage_full_path)

    filename = "missing_path"
    base_path = "/coverage"
    html = Coverband::Reporters::HTMLReport.new(Coverband.configuration.store,
      filename: filename,
      base_path: base_path,
      open_report: false).file_details
    assert_match "File No Longer Available", html
  end

  test "generate dynamic content detailed file report does not allow loading real non project files" do
    @store.send(:save_report, basic_coverage_full_path)

    filename = "#{test_root}/test_helper.rb"
    base_path = "/coverage"
    html = Coverband::Reporters::HTMLReport.new(Coverband.configuration.store,
      filename: filename,
      base_path: base_path,
      open_report: false).file_details
    assert_match "File No Longer Available", html
  end
end
