# frozen_string_literal: true

require 'time'

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      class OperationalReadinessCheck
        include Concerns::IssueConcern
        include Concerns::WorkItemConcern

        BASE_LABELS_FOR_SEARCH = ["feature::addition"].freeze
        OPERATIONAL_READINESS_ISSUE_TEMPLATE_PATH = '.gitlab/issue_templates/Operational Readiness.md'

        def initialize(token:, project: nil, group: nil, limit_to_minutes: nil, search_labels: [], issue_is_blocking: false, dry_run: false)
          @token = token
          @project = "#{group}/#{project}"
          @group = group
          @limit_to_minutes = limit_to_minutes
          @search_labels = search_labels
          @issue_is_blocking = issue_is_blocking
          @dry_run = dry_run
        end

        def invoke!
          created_after = utc_time_minus_mins(limit_to_minutes)

          epics = work_items_client.paginated_call(:group_work_items,
            labels: search_labels.concat(BASE_LABELS_FOR_SEARCH).uniq, state: 'opened', created_after: created_after, extras: [:work_item_fields])

          epics.each do |epic|
            process_epic(epic)
          end
        end

        private

        attr_reader :token, :project, :dry_run, :group, :limit_to_minutes, :search_labels, :issue_is_blocking

        def issue_client
          @issue_client ||= (dry_run ? GitlabClient::IssuesDryClient : GitlabClient::IssuesClient).new(token: token, project: project)
        end

        def work_items_client
          @work_items_client ||= (dry_run ? GitlabClient::WorkItemsDryClient : GitlabClient::WorkItemsClient).new(token: token, project: project, group: group)
        end

        def label_client
          @label_client ||= GitlabClient::LabelsClient.new(token: token, project: project)
        end

        def repository_files_client
          @repository_files_client ||= GitlabClient::RepositoryFilesClient.new(token: token, project: project, file_path: OPERATIONAL_READINESS_ISSUE_TEMPLATE_PATH, ref: 'master')
        end

        def utc_time_minus_mins(mins)
          (Time.now - (mins * 60)).utc.iso8601 if mins
        end

        def process_epic(epic) # rubocop:disable Metrics/AbcSize
          epic = fetch_work_item(epic[:iid], work_items_client, [:notes, :linked_items, :labels, :hierarchy])

          return if has_a_child_epic?(epic)

          pre_check_comment = add_operational_readiness_precheck_comment(epic, work_items_client, label_client)

          return unless note_has_emoji?(pre_check_comment, 'white_check_mark') && !has_operational_readiness_issue_linked?(linked_issue_iids(epic), issue_client)

          issue = create_operation_readiness_issue(epic[:title], work_item_author_id(epic), issue_client, repository_files_client)

          link_operation_readiness_issue(issue, epic, link_type, work_items_client)

          post_comment_about_operation_readiness_issue_created(epic, issue, pre_check_comment, work_items_client)
        end

        def link_type
          issue_is_blocking ? 'BLOCKED_BY' : 'RELATED'
        end
      end
    end
  end
end
