# (c) Copyright 2008, 2010-2012. CodeWeavers, Inc.
package CXBT_win98;
use strict;

use CXLog;
use CXUtils;

my $productid=CXUtils::get_product_id();
my $template="win98";
my $datadir="$ENV{CX_ROOT}/share/crossover/bottle_data";
my $templatedir="$ENV{CX_ROOT}/share/crossover/bottle_templates/$template";
my $windows_version="win98";
my $winearch="win32";


my $default_eassocids=".asf:.cfc:.disco:.doc:.dochtml:.dot:.dwr:.dwt:.efx:.em:.emp:.fla:.mdb:.moh:.mov:.mpw:.mpx:.net:.pdf:.pic:.pict:.ppt:.prj:.psd:.psf:.psp:.qdb:.qdf:.qdt:.qfx:.qt:.sch:.swf:.vrml:.vsd:.vst:.vss:.wma:.wmv:.xls";
my $alternative_eassocids=".csv:.rtf";


my %builtin_dxvk_hashes = (
    '48246b47cde50ba04e38430cfdfd6e5c', 'd3d10_1.dll',
    '6277a1e86a31e5fbd2349dbef86813e6', 'd3d10core.dll',
    '36e24c18280bd8043587579b7c4d0696', 'd3d10.dll',
    '5760643e45ff5c71c83c72baf5f571c4', 'd3d11.dll',
    '009e408b22a9ebc18c9578ec3e892023', 'd3d9.dll',
    '54872eb9af613ba6291029a70d2a08ae', 'dxgi.dll',
    '4942e3b9367f965cf9c74f00d1bdb6b1', 'dxvk_config.dll',
    # v1.5.1-4 64-bit
    '88f0d76caf002468487b567242b27149', 'd3d10_1.dll',
    '9eb17ee84e7acb7eced92fa689d03758', 'd3d10core.dll',
    'ff5c05e21857f539aeb1bc9bf96e841c', 'd3d10.dll',
    '82987e5d442a0986cba04cc6329029a1', 'd3d11.dll',
    '6b08f7eed09265774080e72e2013ab9b', 'd3d9.dll',
    '548c80385d1f2401c22ab9a2323c4bf5', 'dxgi.dll',
    '5af43ef518bac478977121eb0152797e', 'dxvk_config.dll',
    # v1.5.1-7 32-bit
    'e0072f1545f27836dd48bfa712b1d619', 'd3d10_1.dll',
    '7fe0d348974b50164297f82feba30c44', 'dd10core.dll',
    '3de7f126f725ccbb6046c4f1234b618a', 'd3d10.dll',
    'd4ce37a3bd5b45f09656d39b273f7272', 'd3d11.dll',
    '45181a5c146d2f1f2c3c61e705cf06d5', 'd3d9.dll',
    '1652b531ab1e0223d5c617bcfcec08fd', 'dxgi.dll',
    # v1.5.1-7 64-bit
    '6ea7f975b7adf3e1cef5e29a51b53b6d', 'd3d10_1.dll',
    '2e7c7b7e876fd211ff84c5aa685de1c2', 'd3d10core.dll',
    '83bdf2ae6202f0a1bac769caad678e5d', 'd3d10.dll',
    'ac91a7d58b852968fb78499cb3c564d2', 'd3d11.dll',
    '66baefc5956f938ffe47cbc492800712', 'd3d9.dll',
    '0ca6075be93f3f203d03ae861b4c9a18', 'dxgi.dll',
    # v1.5.1-8 32-bit
    '89b1a666b5b35d3f81d42eeb6bf9ddb8', 'd3d10_1.dll',
    'a569a5651616b8afe03a7c860b3999e9', 'd3d10core.dll',
    '2bc0c327134fa0cabf4651f79b05744a', 'd3d10.dll',
    '6b654ecc01b69fd0d3958b74e6e18278', 'd3d11.dll',
    '092748859b2b297a7f832661c1675522', 'd3d9.dll',
    'f261ea9fba28b32fa20241e100aa0695', 'dxgi.dll',
    '4942e3b9367f965cf9c74f00d1bdb6b1', 'dxvk_config.dll',
    # v1.5.1-8 64-bit
    '67814b09774e6f1342b85b3fbda0e5ca', 'd3d10_1.dll',
    '217e556b7c1f03f2c13ca483590fa67a', 'd3d10core.dll',
    'b4a0ab5045ebf7618ea51969b6394bd7', 'd3d10.dll',
    '37c6d59416531ed250b417749dba25bd', 'd3d11.dll',
    '30c064928b28231a722806093a5577bf', 'd3d9.dll',
    '3255409c9699ff7c78707198fe4bdf40', 'dxgi.dll',
    '5af43ef518bac478977121eb0152797e', 'dxvk_config.dll',
);

#
# Bottle template query
#

sub new($$)
{
    my ($class)=@_;

    my $self={ };
    bless $self, $class;
    return $self;
}

sub query($$$)
{
    my ($self, $type, $params)=@_;

    if ($type eq "introspect")
    {
        # Return a very general description of the template.
        # This query is supported by all templates.
        return { properties => { category => "crossover" } };
    }

    $@="Unknown query type '$type'";
    return undef;
}


#
# Bottle creation / upgrade helpers
#

sub run_inf($$$;$)
{
    my ($winver, $scope, $upgrade, $unix_paths)=@_;

    my $section;
    if ($upgrade)
    {
        # Upgrades shouldn't change the bottle version
        $section="DefaultInstall";
    }
    elsif (($winver || "") =~ /^(win98|win2000|winxp|vista|win7|win8|win10|win11)$/)
    {
        # Registry key names are case insensitive :-)
        $section="${winver}Install";
    }
    else
    {
        cxerr("unknown Windows version '$winver'\n");
    }

    my @cxoptions=();
    if (defined $winver)
    {
        # We must both use --winver, for when called from cxbottle, and set
        # CX_WINDOWS_VERSION, for when called by wine, i.e. when the Wine
        # environment is already set.
        $ENV{CX_WINDOWS_VERSION}=$winver;
        push @cxoptions, "--winver", $winver;
    }

    my $shdocvwmode="b";
    my $shdocvwpath=$unix_paths->{"c:/windows/system32/shdocvw.dll"};
    if ($shdocvwpath and open(my $shdocvwdll, "<", $shdocvwpath))
    {
        binmode $shdocvwdll;
        seek($shdocvwdll, 0x40, 0);
        my $tag;
        read($shdocvwdll, $tag, 20);
        close($shdocvwdll);
        $shdocvwmode="d" if ($tag ne "Wine placeholder DLL");
    }

    # Set up some dll overrides to register Wine's dlls
    $ENV{WINEDLLOVERRIDES}=join(";",
        "advpack=b",      # Older native advpack miss an API for builtin IE
        "atl=b",          # Older native atl break registration
        "oleaut32=b",     # Workaround for a native oleaut32 bug
        "rpcrt4=b",       # Incompatibility with native rpcrt4
        "shdocvw=$shdocvwmode", # wine.inf invokes 'iexplore -regserver' which
        "*iexplore.exe=b"       # is specific to the builtin version
    );
    # Note that this will also run wine.inf if needed
    if ($section)
    {
        if (cxsystem("$ENV{CX_ROOT}/bin/wine", "--wl-app", "rundll32.exe",
                     "--no-quotes", "--scope", $scope, @cxoptions,
                     "--desktop", "root", "--dll", $ENV{WINEDLLOVERRIDES},
                     "setupapi.dll,InstallHinfSection", $section, "128",
                     "$datadir/crossover.inf"))
        {
            $@="'rundll32 $section crossover.inf' failed\n";
            return undef;
        }

        if ($winearch eq "win64" and
            -d "$ENV{CX_ROOT}/lib/wine/i386-windows" and
            cxsystem("$ENV{CX_ROOT}/bin/wine", "--wl32-app", "rundll32.exe",
                     "--no-quotes", "--scope", $scope, @cxoptions,
                     "--desktop", "root", "--dll", $ENV{WINEDLLOVERRIDES},
                     "setupapi.dll,InstallHinfSection", $section, "128",
                     "$datadir/crossover.inf"))
        {
            $@="'rundll32 $section crossover.inf' failed\n";
            return undef;
        }
    }
    delete $ENV{CX_WINDOWS_VERSION};
    return 1;
}


#
# Bottle creation
#

sub create($$$)
{
    my ($self, $cxconfig, $params)=@_;
    $@="";

    # Set up the cxbottle.conf configuration file
    require CXRWConfig;
    my $cxbottle=CXRWConfig->new("$datadir/cxbottle.conf");
    my $s=$cxbottle->append_section("Bottle");
    $s->set("BottleID", CXUtils::get_unique_id($ENV{WINEPREFIX}));
    $s->set("Timestamp", $cxconfig->get("CrossOver", "BuildTimestamp", ""));
    $s->set("Encoding", CXUtils::get_system_encoding(1));
    $s->set("Template", $template);
    $s->set("Description", $params->{description}) if (defined $params->{description});
    $s->set("Updater", $params->{updater}) if (defined $params->{updater});
    my $menu_root=$cxconfig->get("BottleDefaults", "MenuRoot");
    $s->set("MenuRoot", $menu_root) if (defined $menu_root);
    my $menu_strip=$cxconfig->get("BottleDefaults", "MenuStrip");
    $s->set("MenuStrip", $menu_strip) if (defined $menu_strip);
    $s->set("MenuMode", "ignore");
    $s->set("AssocMode", "ignore");
    $s->set("WineArch", $winearch);
    $cxbottle->write("$ENV{WINEPREFIX}/cxbottle.conf");

    # CXBottle::find_bottle() checks for the presence of system.reg to detect
    # invalid bottle directories. This creates a race condition due to
    # wineserver's delayed save. So touch system.reg now.
    if (open(my $fh, ">", "$ENV{WINEPREFIX}/system.reg"))
    {
        print $fh "WINE REGISTRY Version 2\n";
        print $fh "#arch=$winearch\n";
        close($fh);
    }

    return undef if (!run_inf($windows_version, $params->{scope}, 0));

    # Force the re-creation of the y: drive
    my $home=$ENV{HOME} || (getpwuid($>))[7];
    my $drive="$ENV{WINEPREFIX}/dosdevices/y:";
    unlink $drive;
    if (!symlink($home, $drive))
    {
        cxwarn("unable to create the y: drive symlink: $!\n");
    }

    # Update the registry
    my $start=CXLog::cxtime();
    my $cmd=shquote_string("$ENV{CX_ROOT}/bin/wine") .
            " --scope $params->{scope} --wl-app regedit.exe -";
    cxlog("Piping into $cmd\n");
    if (open(my $regedit, "| $cmd"))
    {
        my $outlook=$cxconfig->get("BottleDefaults", "OutlookSecurity");
        if ($outlook)
        {
            $outlook=$cxconfig->get("BottleDefaults", "BlockedExtensions");
            print $regedit "[HKEY_CURRENT_USER\\Software\\Wine\\AppDefaults\\Outlook.exe]\n";
            print $regedit "\"DenyShellExecute\"=\"$outlook\"\n";
        }

        close($regedit);
        cxlog("-> rc=$?  (took ", CXLog::cxtime()-$start, " seconds)\n");
        if ($?)
        {
            $@="'$cmd' failed: $?\n";
            return undef;
        }
        # Wait for the bottle shutdown to work around Wine bug 41713
        cxsystem("$ENV{CX_ROOT}/bin/wine", "--scope", $params->{scope},
               "--ux-app", "wineserver", "-w");
        cxlog("-> rc=$?  (took ", CXLog::cxtime()-$start, " seconds)\n");
    }

    # Notify the bottle hooks
    CXBottle::run_bottle_hooks(["create", $templatedir]);
    return 1;
}


#
# Bottle upgrade
#

sub upgrade_cxbottle($$)
{
    my ($cxconfig, $params)=@_;

    require CXUpgrade;
    my $filename="$ENV{WINEPREFIX}/cxbottle.conf";
    cxlog("\n** Upgrading $filename\n\n");
    my $src=CXUpgrade->new($filename);
    $src=CXUpgrade->new(undef) if (!defined $src);
    my $s=$src->get_section("Bottle");
    my $old_version;
    if (defined $s)
    {
        $old_version=$s->get("Version", "2.0.0");
        unlink "$ENV{WINEPREFIX}/.eval" if ($old_version ne CXUtils::get_product_version());
        $old_version=~s/[^0-9.].*$//;
    }
    else
    {
        $old_version=$params->{old_version} || "1.3.1";
    }
    $old_version=~s/^([0-9]\.)/0$1/;
    cxlog("Old Version=$old_version\n");

    my $template="$datadir/cxbottle.conf";
    my $dst=CXUpgrade->new($template);
    if (!defined $dst)
    {
        cxerr("unable to read '$template': $!\n");
        return ($old_version, $src);
    }

    # Tweak the configuration values
    my ($encoding, $bottleid);
    $s=$src->get_section("Bottle");
    if (defined $s)
    {
        # Reset the Version field
        $s->remove_field("Version");
        $s->set("Timestamp", $cxconfig->get("CrossOver", "BuildTimestamp", ""));
        $encoding=$s->get("Encoding");
        $bottleid=$s->get("BottleID");
    }
    else
    {
        $s=$src->add_section("Bottle");
    }

    # Before these existed, bottles were always considered to be installed
    $s->set("MenuMode", "install") if (!defined $s->get("MenuMode"));
    $s->set("AssocMode", "install") if (!defined $s->get("AssocMode"));

    if (!defined $bottleid)
    {
        # Make sure the bottle gets an id
        # (in case of an upgrade from a pre-2.0 tree)
        $s->set("BottleID", CXUtils::get_unique_id($ENV{WINEPREFIX}));
    }
    if (($encoding || "ANSI_X3.4-1968") eq "ANSI_X3.4-1968")
    {
        # ANSI_X3.4-1968 is ASCII's little name so we assume that it's
        # going to be compatible with whatever encoding is used by
        # this system.
        $s->set("Encoding", CXUtils::get_system_encoding(1));
    }
    if (!$src->get_filename())
    {
        # Brand new cxbottle.conf, get MenuRoot from $productid.conf
        require CXConfig;
        my $cxconfig=CXConfig->new("$ENV{CX_ROOT}/etc/$productid.conf");
        my $menu_root=$cxconfig->get("BottleDefaults", "MenuRoot");
        $s->set("MenuRoot", $menu_root) if (defined $menu_root);
        my $menu_strip=$cxconfig->get("BottleDefaults", "MenuStrip");
        $s->set("MenuStrip", $menu_strip) if (defined $menu_strip);
    }

    $s = $src->get_section("EnvironmentVariables");
    $s = $src->add_section("EnvironmentVariables") if (!defined $s);

    if ($old_version lt "15.0.0")
    {
        # for CrossOver hack 12735
        $s->set("CX_REPORT_REAL_USERNAME", "yes");
        $s->remove_field("PULSE_LATENCY_MSEC");
    }

    # Do the final merge and write the new configuration file
    $dst->merge($src);
    $dst->write($filename);

    return ($old_version, $src);
}

# builtin dxvk upgrade functions
sub get_file_hash($)
{
    my ($filename) = @_;

    require Digest::MD5;
    my $hasher = Digest::MD5->new;

    open(my $f, '<', $filename) or return undef;

    binmode($f);

    $hasher->addfile($f);

    close($f);

    return $hasher->hexdigest;
}

sub upgrade_dxvk($$$$$$)
{
    my ($self, $reg_lines, $config, $scope, $winsystem_path, $src_path) = @_;

    my @dll_filenames = ("dxgi", "d3d9", "d3d11", "d3d10", "d3d10_1", "d3d10core");

    my @win_paths = map { $winsystem_path . $_ . ".dll" } @dll_filenames;

    my $winepath=shquote_string("$ENV{CX_ROOT}/bin/wine") .
                 " --no-convert --scope $scope --wl-app winepath.exe -- " .
                 join(" ", map { shquote_string($_) } @win_paths);

    my @unix_paths = cxbackquote("$winepath 2>/dev/null", 1);

    cxlog("Removing DXVK files from $winsystem_path\n");

    foreach my $i (0 .. $#dll_filenames)
    {
        my $dll_name = $dll_filenames[$i];
        my $win_path = $win_paths[$i];
        my $unix_path = $unix_paths[$i];

        chomp $unix_path;

        my $hash = get_file_hash($unix_path);

        if (defined $hash &&
            (defined $builtin_dxvk_hashes{$hash} ||
             $hash eq $config->get("DxvkFiles", $win_path, "")))
        {
            require File::Copy;
            my $src_filename = "$src_path/$dll_name" . ".dll";
            cxlog("Copying $src_filename to $unix_path\n");
            File::Copy::copy($src_filename, $unix_path);

            push @{$reg_lines},
                "[HKEY_CURRENT_USER\\Software\\Wine\\DllOverrides]\n",
                "\"$dll_name\"=-\n";
        }
    }
}

sub upgrade_cxmenu_config()
{
    my $filename="$ENV{WINEPREFIX}/cxmenu.conf";
    return if (!-f $filename);

    # The whole bottle is locked so we don't need to lock cxmenu.conf
    require CXRWConfig;
    my $cxmenu=CXRWConfig->new($filename);
    if (!$cxmenu)
    {
        cxwarn("could not upgrade '$filename': $!\n");
        return;
    }

    foreach my $section ($cxmenu->get_section_names())
    {
        if ($section !~ /\.lnk$/i)
        {
            my $new="$section.lnk";
            $cxmenu->rename_section($section, $new);
        }
    }
    cxwarn("could not upgrade '$filename': $!\n") if (!$cxmenu->save());
}

sub upgrade_graphics_settings()
{
    require CXRWConfig;
    my $bottle_config=CXRWConfig->new("$ENV{WINEPREFIX}/cxbottle.conf");
    my $env_var_config=$bottle_config->get_section("EnvironmentVariables");

    my $enable_dxvk=$env_var_config->get("WINEDXVK", "0");
    $env_var_config->set("CX_GRAPHICS_BACKEND", "dxvk") if (
        !defined $env_var_config->get("CX_GRAPHICS_BACKEND") and $enable_dxvk eq "1");

    my $enable_d3dmetal=$env_var_config->get("WINED3DMETAL", "0");
    $env_var_config->set("CX_GRAPHICS_BACKEND", "d3dmetal") if (
        !defined $env_var_config->get("CX_GRAPHICS_BACKEND") and $enable_d3dmetal eq "1");


    $bottle_config->save();
}

sub upgrade($$$)
{
    my ($self, $cxconfig, $params)=@_;
    $@="";

    # Upgrade the configuration files
    my ($old_version, $cxbottle)=upgrade_cxbottle($cxconfig, $params);

    # Grab all the Wine paths we'll need in one go
    my @wine_paths=("c:/Windows/Icons",
                    "c:/windows/system32/shdocvw.dll", # for run_inf()
                    "c:/windows/system32/d3d11.dll", # for upgrade_dxvk
                    "c:/users/crossover/AppData/Roaming/Microsoft/Windows/Templates", # for bug 22422
                   );

    my $winepath=shquote_string("$ENV{CX_ROOT}/bin/wine") .
                 " --no-convert --scope $params->{scope} --wl-app winepath.exe -- " .
                 join(" ", map { shquote_string($_) } @wine_paths);
    my %unix_paths;
    my $i=0;
    foreach my $path (cxbackquote("$winepath 2>/dev/null", 1))
    {
        chomp $path;
        $unix_paths{$wine_paths[$i]}=$path;
        cxlog("[$wine_paths[$i]] -> [$path]\n");
        $i++;
    }

    # Remove Templates symlink if it points to ~, for bug 22422
    my $dir=$unix_paths{"c:/users/crossover/AppData/Roaming/Microsoft/Windows/Templates"};
    if (-l $dir and readlink($dir) eq $ENV{HOME})
    {
        cxlog("Deleting '$dir' symlink\n");
        unlink $dir;
    }

    my $reg_lines=[];

    return undef if (!run_inf($windows_version, $params->{scope}, 1, \%unix_paths));

    $dir=$unix_paths{"c:/Windows/Icons"};
    if (defined $dir and opendir(my $dh, $dir))
    {
        foreach my $dentry (readdir $dh)
        {
            if ($dentry !~ /^\.\.?$/ and -d "$dir/$dentry")
            {
                cxlog("Deleting the '$dir/$dentry' directory\n");
                require File::Path;
                if (!File::Path::rmtree("$dir/$dentry"))
                {
                    cxwarn("unable to delete the '$dir/$dentry' directory\n");
                }
            }
        }
        closedir($dh);
    }

    if ($old_version lt "18.0.0" &&
        (-e "$ENV{WINEPREFIX}/drive_c/Program Files/Microsoft Office/Office15" ||
         -e "$ENV{WINEPREFIX}/drive_c/Program Files/Microsoft Office/Office16"))
    {
        # for bug 15944
        push @{$reg_lines},
            "[HKEY_CURRENT_USER\\Software\\Wine\\Direct2D]\n",
            "\"max_version_factory\"=dword:00000000\n";
    }

    if ($old_version lt "18.0.0" &&
        (-e "$ENV{WINEPREFIX}/drive_c/Program Files/Microsoft Office/Office14" ||
         -e "$ENV{WINEPREFIX}/drive_c/Program Files/Microsoft Office/Office15" ||
         -e "$ENV{WINEPREFIX}/drive_c/Program Files/Microsoft Office/Office16"))
    {
        # for bug 15944
        push @{$reg_lines},
            "[HKEY_CURRENT_USER\\Software\\Wine\\X11 Driver]\n",
            "\"ScreenDepth\"=\"32\"\n";
    }

    # Upgrade builtin dxvk, which was copied until CrossOver 24
    if (-e "$ENV{WINEPREFIX}/dxvkfiles.conf" ||
        defined $builtin_dxvk_hashes{get_file_hash($unix_paths{"c:/windows/system32/d3d11.dll"})})
    {
        cxlog("Upgrading builtin DXVK\n");

        require CXRWConfig;
        my $bottle_config=CXRWConfig->new("$ENV{WINEPREFIX}/cxbottle.conf");
        $bottle_config->get_section("EnvironmentVariables")->set("WINEDXVK", "1");
        $bottle_config->save();

        require CXRWConfig;
        my $dxvkfiles=CXRWConfig->new("$ENV{WINEPREFIX}/dxvkfiles.conf");

        if ($winearch eq "win64")
        {
            upgrade_dxvk($self, $reg_lines, $dxvkfiles, $params->{scope}, "C:\\windows\\system32\\",
                "$ENV{CX_ROOT}/lib/wine/x86_64-windows");
            upgrade_dxvk($self, $reg_lines, $dxvkfiles, $params->{scope}, "C:\\windows\\syswow64\\",
                "$ENV{CX_ROOT}/lib/wine/i386-windows");
        }
        else
        {
            upgrade_dxvk($self, $reg_lines, $dxvkfiles, $params->{scope}, "C:\\windows\\system32\\",
                "$ENV{CX_ROOT}/lib/wine/i386-windows");
        }

        unlink "$ENV{WINEPREFIX}/dxvkfiles.conf";
    }

    # Upgrade graphics settings to the format used since CrossOver 25
    upgrade_graphics_settings();

    # Upgrade the registry
    if (@$reg_lines)
    {
        my $cmd=shquote_string("$ENV{CX_ROOT}/bin/wine") .
                " --scope $params->{scope} --wl-app regedit.exe -";
        cxlog("Piping into $cmd\n");
        cxlog(@$reg_lines);
        if (open(my $regedit, "| $cmd"))
        {
            print $regedit "REGEDIT4\n\n";
            print $regedit @$reg_lines;
            close($regedit);
            if ($?)
            {
                $@="'$cmd' failed: $?\n";
                return undef;
            }
        }
    }

    my $uassoc="";
    my $umenu="";
    if ($old_version lt "14.0.4")
    {
        # The XDG data directories have changed. Furthermore, due to a bug in
        # assocscan some associations were not detected. So re-sync.
        $uassoc.="+removeall+install+sync";
        $umenu.="+removeall+install";
    }
    if ($old_version lt "20.0")
    {
        # The Arch field often contained garbage and the StartupWMClass field
        # was not set.
        $umenu.="+removeall+sync+install";
    }

    my $bottle_mode=$ENV{CX_BOTTLE_MODE} || $cxbottle->get("Bottle", "AssocMode", "");
    my $installed=($bottle_mode =~ /^install$/i);
    if ($uassoc)
    {
        my @cmd=("$ENV{CX_ROOT}/bin/cxassoc", "--scope", $params->{scope});
        if ($installed)
        {
            if ($uassoc =~ /nukeall/)
            {
                # This action must be used when locate_gui.sh's arbitration
                # between association systems changes. Otherwise we risk not
                # removing associations for systems that were used but are now
                # in desktop_assoc_ignore_list. This task must be performed as
                # a separate step because of the --ignorelist parameter.
                cxsystem(@cmd, "--removeall", "--ignorelist", "");
                # No need to do a plain --removeall on top
                $uassoc =~ s/removeall//g;
            }
            push @cmd, "--removeall" if ($uassoc =~ /removeall/);
            push @cmd, "--install" if ($uassoc =~ /install/);
        }
        if ($uassoc =~ /sync/)
        {
            my $mode="mime;default=$default_eassocids;alternative=$alternative_eassocids";
            $mode =~ s/\./\\./g;
            push @cmd, "--sync-install-none", "--sync-uninstall-none" if (!$installed);
            push @cmd, "--sync", "--mode", $mode;
        }
        cxsystem(@cmd) if (@cmd ne 3);
    }
    $bottle_mode=$ENV{CX_BOTTLE_MODE} || $cxbottle->get("Bottle", "MenuMode", "");
    $installed=($bottle_mode =~ /^install$/i);
    if ($umenu)
    {
        my @cmd=("$ENV{CX_ROOT}/bin/cxmenu", "--scope", $params->{scope});
        if ($installed)
        {
            if ($umenu =~ /nukeall/)
            {
                # See the comments for the associations nukeall action.
                cxsystem(@cmd, "--removeall", "--ignorelist", "");
                $umenu =~ s/removeall//g;
            }
            push @cmd, "--removeall" if ($umenu =~ /removeall/);
            push @cmd, "--install" if ($umenu =~ /install/);
        }
        if ($umenu =~ /sync/)
        {
          push @cmd, "--sync-install-none", "--sync-uninstall-none" if (!$installed);
          push @cmd, "--sync", "--mode", "install";
        }
        cxsystem(@cmd) if (@cmd ne 3);
    }

    # Notify the bottle hooks
    require CXBottle;
    CXBottle::run_bottle_hooks(["upgrade-from", $old_version]);

    return 1;
}


#
# Bottle resync
#

sub needs_resync($$)
{
    # This type of bottle never needs to be resync-ed with an external source
    return 0;
}

return 1;
