<?php declare(strict_types = 1);

namespace PHPStan\Rules\Operators;

use PhpParser\Node;
use PhpParser\Node\Expr;
use PhpParser\Node\Expr\AssignOp\Plus as AssignOpPlus;
use PhpParser\Node\Expr\BinaryOp\Plus as BinaryOpPlus;
use PHPStan\Analyser\Scope;
use PHPStan\Rules\Rule;
use PHPStan\Rules\RuleErrorBuilder;
use PHPStan\Type\VerbosityLevel;
use function count;
use function sprintf;

/**
 * @implements Rule<Expr>
 */
class OperandsInArithmeticAdditionRule implements Rule
{

	private OperatorRuleHelper $helper;

	public function __construct(OperatorRuleHelper $helper)
	{
		$this->helper = $helper;
	}

	public function getNodeType(): string
	{
		return Expr::class;
	}

	public function processNode(Node $node, Scope $scope): array
	{
		if ($node instanceof BinaryOpPlus) {
			$left = $node->left;
			$right = $node->right;
		} elseif ($node instanceof AssignOpPlus) {
			$left = $node->var;
			$right = $node->expr;
		} else {
			return [];
		}

		$leftType = $scope->getType($left);
		$rightType = $scope->getType($right);
		if (count($leftType->getArrays()) > 0 && count($rightType->getArrays()) > 0) {
			return [];
		}

		$messages = [];
		if (!$this->helper->isValidForArithmeticOperation($scope, $left)) {
			$messages[] = RuleErrorBuilder::message(sprintf(
				'Only numeric types are allowed in +, %s given on the left side.',
				$leftType->describe(VerbosityLevel::typeOnly()),
			))->identifier('plus.leftNonNumeric')->build();
		}
		if (!$this->helper->isValidForArithmeticOperation($scope, $right)) {
			$messages[] = RuleErrorBuilder::message(sprintf(
				'Only numeric types are allowed in +, %s given on the right side.',
				$rightType->describe(VerbosityLevel::typeOnly()),
			))->identifier('plus.rightNonNumeric')->build();
		}

		return $messages;
	}

}
