#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Ansible module to manage CheckPoint Firewall (c) 2019
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#

from __future__ import (absolute_import, division, print_function)

__metaclass__ = type

ANSIBLE_METADATA = {'metadata_version': '1.1',
                    'status': ['preview'],
                    'supported_by': 'community'}

DOCUMENTATION = """
---
module: cp_mgmt_mobile_profile
short_description: Manages mobile-profile objects on Checkpoint over Web Services API
description:
  - Manages mobile-profile objects on Checkpoint devices including creating, updating and removing objects.
  - All operations are performed over Web Services API.
  - Available from R82 management version.
version_added: "6.0.0"
author: "Eden Brillant (@chkp-edenbr)"
options:
  name:
    description:
      - Object name.
    type: str
    required: True
  applications:
    description:
      - Applications settings.
    type: dict
    suboptions:
      enable_print_mails:
        description:
          - Allow to print mails.
        type: bool
      max_attachments_size:
        description:
          - Maximum size of attachments allowed for downloading -  you can choose a unit (gbs, kbs, mbs, bytes) in "max-attachments-unit" field.
        type: int
      calendar_from_the_last:
        description:
          - How far back to see your Calendar from the current date - you can choose a unit (day, week, month) in "calendar-from-the-last-unit" field.
        type: int
      calendar_from_the_last_unit:
        description:
          - Unit for "calendar-from-the-last" numeric value.
        type: str
        choices: ['weeks', 'months', 'days']
      calendar_to_the_following:
        description:
          - How much ahead to see your Calendar from the current date - you can choose a unit (day, week, month) in "calendar-to-the-following-unit" field.
        type: int
      calendar_to_the_following_unit:
        description:
          - Unit for "calendar-to-the-following" numeric value.
        type: str
        choices: ['weeks', 'months', 'days']
      mail_from_the_last:
        description:
          - How far back to see your emails from the current date - choose a unit (day, week, month) in "mail-from-the-last-unit" field.
        type: int
      mail_from_the_last_unit:
        description:
          - Unit for "mail-from-the-last" numeric value.
        type: str
        choices: ['weeks', 'months', 'days']
      synchronize_contacts:
        description:
          - Contacts synchronization method - from the mail server to device and the app and vice versa or from the mail server to device and the
            app or from the mail server to the app.
        type: str
        choices: ['mail srv to app and device and vice versa', 'mail srv to app and device', 'mail srv to app']
      allow_push_notification:
        description:
          - Allow to receive push notifications of mails and meetings.
        type: bool
      allow_calendar_sync:
        description:
          - Allow synchronization between business calendar to device calendar.
        type: bool
      allow_contacts_from_global_address_list:
        description:
          - Allow to add additional contacts from Global Address List to the app.
        type: bool
      allow_contacts_from_local_phone:
        description:
          - Allow to add additional contacts from local phone to the app.
        type: bool
      save_local_web_cache:
        description:
          - Configure whether local cache data generated by web browser should be preserved.
        type: bool
      allow_caching_docsec_credentials:
        description:
          - Allow store encrypted document credentials in application secure storage.
        type: bool
      allow_caching_docsec_keys:
        description:
          - Allow store encrypted document keys in application secure storage.
        type: bool
  client_customization:
    description:
      - Client customization settings.
    type: dict
    suboptions:
      app_theme_color_dark:
        description:
          - Configure the application display colors in Dark mode. 6 hex digits that define RGB color - relevant for IOS.
        type: str
      app_theme_color_light:
        description:
          - Configure the application display colors in light mode. 6 hex digits that define RGB color - relevant for IOS.
        type: str
      allow_calendar:
        description:
          - Allow sync business calendar to device calendar.
        type: bool
      allow_contacts:
        description:
          - Enable/Disable contacts app.
        type: bool
      allow_mail:
        description:
          - Enable/Disable email app.
        type: bool
      allow_notes_sync:
        description:
          - Allow sync business notes to device notes.
        type: bool
      allow_saved_file_apps:
        description:
          - Allow the appearance of 'Saved file app' in the app list.
        type: bool
      allow_secure_chat:
        description:
          - Enable/Disable Messages app (depends on Mail app).
        type: bool
      allow_tasks:
        description:
          - Enable/Disable Tasks app.
        type: bool
      certificate_expire_message:
        description:
          - message to show users when certificate is expired - for admin to fill - can contain only English characters, digits, comma, spaces and points.
        type: str
  data_leak_prevention:
    description:
      - Data leak prevention settings.
    type: dict
    suboptions:
      open_extension_with_external_app:
        description:
          - Open the following extensions from your app with external apps when they cannot be opened with Capsule viewer.
        type: list
        elements: str
        choices: ['any file', 'word documents', 'excel documents', 'powerpoint documents', 'any microsoft office documents', 'media files']
      share_protected_extension:
        description:
          - Share protected files extensions to external apps.
        type: list
        elements: str
        choices: ['any file', 'word documents', 'excel documents', 'powerpoint documents', 'any microsoft office documents', 'media files']
      share_unprotected_extension:
        description:
          - Share unprotected files extensions to external apps.
        type: list
        elements: str
        choices: ['any file', 'word documents', 'excel documents', 'powerpoint documents', 'any microsoft office documents', 'media files']
      allow_copy_paste:
        description:
          - Allow copy paste of mail content.
        type: bool
      block_forward_attachments:
        description:
          - Allow share mail attachments with external mails.
        type: bool
      block_screenshot:
        description:
          - If true - you can't make a screenshot from your app.
        type: bool
      allowed_domains_forward_attachment:
        description:
          - exclusion of domains which attachments are allowed to be sent, even that shared policy prevents sharing these kinds of attached files
            - can contain only English characters, digits, comma, spaces and points.
        type: str
      accept_protected_file_extensions:
        description:
          - Accept protected files with these extensions from external apps to your app.
        type: list
        elements: str
        choices: ['any file', 'word documents', 'excel documents', 'powerpoint documents', 'any microsoft office documents', 'media files']
      accept_unprotected_file_extensions:
        description:
          - Accept unprotected files with these extensions from external apps to your app.
        type: list
        elements: str
        choices: ['any file', 'word documents', 'excel documents', 'powerpoint documents', 'any microsoft office documents', 'media files']
      allow_import_from_gallery:
        description:
          - Allow import media from gallery.
        type: bool
      allow_taking_photos_and_videos:
        description:
          - Allow the camera to be used from your app.
        type: bool
      offer_capsule_as_viewer:
        description:
          - Offer Capsule as a viewer for external protected documents.
        type: bool
  harmony_mobile:
    description:
      - Integrations settings.
    type: dict
    suboptions:
      protect_policy_enabled:
        description:
          - Enable/disable Protect Application- cannot be enable if Harmony SDK is enable.
        type: bool
      protect_high_risk_action:
        description:
          - What is the action if there is high risk found by Harmony Mobile.
        type: str
        choices: ['none', 'wipe', 'block']
      protect_high_risk_message:
        description:
          - The message can contain only English characters, digits, comma, spaces and points.
        type: str
      protect_medium_risk_action:
        description:
          - What is the action if there is medium risk found by Harmony Mobile.
        type: str
        choices: ['none', 'wipe', 'block']
      protect_medium_risk_message:
        description:
          - The message can contain only English characters, digits, comma, spaces and points.
        type: str
      protect_not_activated_action:
        description:
          - What is the action if there is policy violation (configuration for Harmony Mobile).
        type: str
        choices: ['none', 'wipe', 'block']
      protect_not_activated_message:
        description:
          - The message can contain only English characters, digits, comma, spaces and points.
        type: str
      enable_harmony_mobile_sdk:
        description:
          - Enable/disable Harmony SDK - cannot be enable if Harmony Mobile Application is enable.
        type: bool
      compromised_behavior:
        description:
          - Device configuration - response to malicious behavior (configuration for Harmony SDK).
        type: str
        choices: ['block', 'notify', 'ignore']
      harmony_mobile_sdk_license:
        description:
          - License for Harmony Mobile Sdk (configuration for Harmony SDK) - can contain only English characters, digits, comma, spaces and point.
        type: str
      malware_behavior:
        description:
          - Behavior when App is identified as malicious (configuration for Harmony SDK).
        type: str
        choices: ['block', 'notify', 'ignore']
      man_in_the_middle_attack:
        description:
          - Behavior when there is a network man-in-the-middle attack (configuration for Harmony SDK).
        type: str
        choices: ['block', 'notify', 'ignore']
      os_integrity_compromised:
        description:
          - Behavior when Device OS is compromised (configuration for Harmony SDK).
        type: str
        choices: ['block', 'notify', 'ignore']
      suspicious_app:
        description:
          - Behavior when App is suspected as malicious (configuration for Harmony SDK).
        type: str
        choices: ['block', 'notify', 'ignore']
      suspicious_enterprise_certificate:
        description:
          - Behavior when a certificate profile has been installed allowing the installing of apps on device from unknown source - iOS only
            (configuration for Harmony SDK).
        type: str
        choices: ['block', 'notify', 'ignore']
  security:
    description:
      - Security settings.
    type: dict
    suboptions:
      session_timeout:
        description:
          - Session timeout - you can choose a unit (day, week, month) in "session-timeout-unit" field.
        type: int
      session_timeout_unit:
        description:
          - Unit for "session-timeout" numeric value.
        type: str
        choices: ['weeks', 'days', 'hours', 'minutes']
      activate_passcode_lock:
        description:
          - Require passcode to the application.
        type: bool
      allow_store_credentials:
        description:
          - Allow storing the credentials on the device.
        type: bool
      passcode_profile:
        description:
          - Passcode Policy object identified by the name or UID.
        type: str
      report_jailbroken:
        description:
          - Issue log when device is detected as jail broken.
        type: bool
      block_jailbroken:
        description:
          - Action upon detection of jail broken devices.
        type: str
        choices: ['block', 'none']
      block_3rd_party_keyboard:
        description:
          - Block 3rd party keyboard.
        type: bool
      hide_ssl_connect_anyway_button:
        description:
          - Hide connect button on critical SSL trust failures.
        type: bool
  tags:
    description:
      - Collection of tag identifiers.
    type: list
    elements: str
  color:
    description:
      - Color of the object. Should be one of existing colors.
    type: str
    choices: ['aquamarine', 'black', 'blue', 'crete blue', 'burlywood', 'cyan', 'dark green', 'khaki', 'orchid', 'dark orange', 'dark sea green',
             'pink', 'turquoise', 'dark blue', 'firebrick', 'brown', 'forest green', 'gold', 'dark gold', 'gray', 'dark gray', 'light green', 'lemon chiffon',
             'coral', 'sea green', 'sky blue', 'magenta', 'purple', 'slate blue', 'violet red', 'navy blue', 'olive', 'orange', 'red', 'sienna', 'yellow']
  comments:
    description:
      - Comments string.
    type: str
  details_level:
    description:
      - The level of detail for some of the fields in the response can vary from showing only the UID value of the object to a fully detailed
        representation of the object.
    type: str
    choices: ['uid', 'standard', 'full']
  domains_to_process:
    description:
      - Indicates which domains to process the commands on. It cannot be used with the details-level full, must be run from the System Domain only and
        with ignore-warnings true. Valid values are, CURRENT_DOMAIN, ALL_DOMAINS_ON_THIS_SERVER.
    type: list
    elements: str
  ignore_warnings:
    description:
      - Apply changes ignoring warnings.
    type: bool
  ignore_errors:
    description:
      - Apply changes ignoring errors. You won't be able to publish such a changes. If ignore-warnings flag was omitted - warnings will also be ignored.
    type: bool
extends_documentation_fragment: check_point.mgmt.checkpoint_objects
"""

EXAMPLES = """
- name: add-mobile-profile
  cp_mgmt_mobile_profile:
    name: New Mobile Profile
    state: present

- name: set-mobile-profile
  cp_mgmt_mobile_profile:
    data_leak_prevention:
      share_protected_extension: word documents
    name: New Mobile Profile
    state: present

- name: delete-mobile-profile
  cp_mgmt_mobile_profile:
    name: New Mobile Profile
    state: absent
"""

RETURN = """
cp_mgmt_mobile_profile:
  description: The checkpoint object created or updated.
  returned: always, except when deleting the object.
  type: dict
"""

from ansible.module_utils.basic import AnsibleModule
from ansible_collections.check_point.mgmt.plugins.module_utils.checkpoint import checkpoint_argument_spec_for_objects, api_call


def main():
    argument_spec = dict(
        name=dict(type='str', required=True),
        applications=dict(type='dict', options=dict(
            enable_print_mails=dict(type='bool'),
            max_attachments_size=dict(type='int'),
            calendar_from_the_last=dict(type='int'),
            calendar_from_the_last_unit=dict(type='str', choices=['weeks', 'months', 'days']),
            calendar_to_the_following=dict(type='int'),
            calendar_to_the_following_unit=dict(type='str', choices=['weeks', 'months', 'days']),
            mail_from_the_last=dict(type='int'),
            mail_from_the_last_unit=dict(type='str', choices=['weeks', 'months', 'days']),
            synchronize_contacts=dict(type='str', choices=['mail srv to app and device and vice versa', 'mail srv to app and device', 'mail srv to app']),
            allow_push_notification=dict(type='bool'),
            allow_calendar_sync=dict(type='bool'),
            allow_contacts_from_global_address_list=dict(type='bool'),
            allow_contacts_from_local_phone=dict(type='bool'),
            save_local_web_cache=dict(type='bool'),
            allow_caching_docsec_credentials=dict(type='bool'),
            allow_caching_docsec_keys=dict(type='bool')
        )),
        client_customization=dict(type='dict', options=dict(
            app_theme_color_dark=dict(type='str'),
            app_theme_color_light=dict(type='str'),
            allow_calendar=dict(type='bool'),
            allow_contacts=dict(type='bool'),
            allow_mail=dict(type='bool'),
            allow_notes_sync=dict(type='bool'),
            allow_saved_file_apps=dict(type='bool'),
            allow_secure_chat=dict(type='bool'),
            allow_tasks=dict(type='bool'),
            certificate_expire_message=dict(type='str')
        )),
        data_leak_prevention=dict(type='dict', options=dict(
            open_extension_with_external_app=dict(type='list', elements='str', choices=['any file', 'word documents',
                                                                                        'excel documents',
                                                                                        'powerpoint documents',
                                                                                        'any microsoft office documents',
                                                                                        'media files']),
            share_protected_extension=dict(type='list', elements='str', choices=['any file', 'word documents', 'excel documents',
                                                                                 'powerpoint documents',
                                                                                 'any microsoft office documents', 'media files']),
            share_unprotected_extension=dict(type='list', elements='str', choices=['any file', 'word documents', 'excel documents',
                                                                                   'powerpoint documents',
                                                                                   'any microsoft office documents', 'media files']),
            allow_copy_paste=dict(type='bool'),
            block_forward_attachments=dict(type='bool'),
            block_screenshot=dict(type='bool'),
            allowed_domains_forward_attachment=dict(type='str'),
            accept_protected_file_extensions=dict(type='list', elements='str', choices=['any file', 'word documents', 'excel documents',
                                                                                        'powerpoint documents',
                                                                                        'any microsoft office documents', 'media files']),
            accept_unprotected_file_extensions=dict(type='list', elements='str', choices=['any file', 'word documents', 'excel documents',
                                                                                          'powerpoint documents',
                                                                                          'any microsoft office documents', 'media files']),
            allow_import_from_gallery=dict(type='bool'),
            allow_taking_photos_and_videos=dict(type='bool'),
            offer_capsule_as_viewer=dict(type='bool')
        )),
        harmony_mobile=dict(type='dict', options=dict(
            protect_policy_enabled=dict(type='bool'),
            protect_high_risk_action=dict(type='str', choices=['none', 'wipe', 'block']),
            protect_high_risk_message=dict(type='str'),
            protect_medium_risk_action=dict(type='str', choices=['none', 'wipe', 'block']),
            protect_medium_risk_message=dict(type='str'),
            protect_not_activated_action=dict(type='str', choices=['none', 'wipe', 'block']),
            protect_not_activated_message=dict(type='str'),
            enable_harmony_mobile_sdk=dict(type='bool'),
            compromised_behavior=dict(type='str', choices=['block', 'notify', 'ignore']),
            harmony_mobile_sdk_license=dict(type='str'),
            malware_behavior=dict(type='str', choices=['block', 'notify', 'ignore']),
            man_in_the_middle_attack=dict(type='str', choices=['block', 'notify', 'ignore']),
            os_integrity_compromised=dict(type='str', choices=['block', 'notify', 'ignore']),
            suspicious_app=dict(type='str', choices=['block', 'notify', 'ignore']),
            suspicious_enterprise_certificate=dict(type='str', choices=['block', 'notify', 'ignore'])
        )),
        security=dict(type='dict', options=dict(
            session_timeout=dict(type='int'),
            session_timeout_unit=dict(type='str', choices=['weeks', 'days', 'hours', 'minutes']),
            activate_passcode_lock=dict(type='bool'),
            allow_store_credentials=dict(type='bool'),
            passcode_profile=dict(type='str', no_log=False),
            report_jailbroken=dict(type='bool'),
            block_jailbroken=dict(type='str', choices=['block', 'none']),
            block_3rd_party_keyboard=dict(type='bool'),
            hide_ssl_connect_anyway_button=dict(type='bool')
        )),
        tags=dict(type='list', elements='str'),
        color=dict(type='str', choices=['aquamarine', 'black', 'blue', 'crete blue', 'burlywood', 'cyan', 'dark green',
                                        'khaki', 'orchid', 'dark orange', 'dark sea green', 'pink', 'turquoise', 'dark blue', 'firebrick', 'brown',
                                        'forest green', 'gold', 'dark gold', 'gray', 'dark gray', 'light green', 'lemon chiffon', 'coral', 'sea green',
                                        'sky blue', 'magenta', 'purple', 'slate blue', 'violet red', 'navy blue', 'olive', 'orange', 'red', 'sienna',
                                        'yellow']),
        comments=dict(type='str'),
        details_level=dict(type='str', choices=['uid', 'standard', 'full']),
        domains_to_process=dict(type='list', elements='str'),
        ignore_warnings=dict(type='bool'),
        ignore_errors=dict(type='bool')
    )
    argument_spec.update(checkpoint_argument_spec_for_objects)

    module = AnsibleModule(argument_spec=argument_spec, supports_check_mode=True)
    api_call_object = 'mobile-profile'

    result = api_call(module, api_call_object)
    module.exit_json(**result)


if __name__ == '__main__':
    main()
