﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/personalize/PersonalizeRequest.h>
#include <aws/personalize/Personalize_EXPORTS.h>
#include <aws/personalize/model/Tag.h>

#include <utility>

namespace Aws {
namespace Personalize {
namespace Model {

/**
 */
class CreateFilterRequest : public PersonalizeRequest {
 public:
  AWS_PERSONALIZE_API CreateFilterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateFilter"; }

  AWS_PERSONALIZE_API Aws::String SerializePayload() const override;

  AWS_PERSONALIZE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the filter to create.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateFilterRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the dataset group that the filter will belong to.</p>
   */
  inline const Aws::String& GetDatasetGroupArn() const { return m_datasetGroupArn; }
  inline bool DatasetGroupArnHasBeenSet() const { return m_datasetGroupArnHasBeenSet; }
  template <typename DatasetGroupArnT = Aws::String>
  void SetDatasetGroupArn(DatasetGroupArnT&& value) {
    m_datasetGroupArnHasBeenSet = true;
    m_datasetGroupArn = std::forward<DatasetGroupArnT>(value);
  }
  template <typename DatasetGroupArnT = Aws::String>
  CreateFilterRequest& WithDatasetGroupArn(DatasetGroupArnT&& value) {
    SetDatasetGroupArn(std::forward<DatasetGroupArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filter expression defines which items are included or excluded from
   * recommendations. Filter expression must follow specific format rules. For
   * information about filter expression structure and syntax, see <a
   * href="https://docs.aws.amazon.com/personalize/latest/dg/filter-expressions.html">Filter
   * expressions</a>.</p>
   */
  inline const Aws::String& GetFilterExpression() const { return m_filterExpression; }
  inline bool FilterExpressionHasBeenSet() const { return m_filterExpressionHasBeenSet; }
  template <typename FilterExpressionT = Aws::String>
  void SetFilterExpression(FilterExpressionT&& value) {
    m_filterExpressionHasBeenSet = true;
    m_filterExpression = std::forward<FilterExpressionT>(value);
  }
  template <typename FilterExpressionT = Aws::String>
  CreateFilterRequest& WithFilterExpression(FilterExpressionT&& value) {
    SetFilterExpression(std::forward<FilterExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <a
   * href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a>
   * to apply to the filter.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateFilterRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateFilterRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_datasetGroupArn;
  bool m_datasetGroupArnHasBeenSet = false;

  Aws::String m_filterExpression;
  bool m_filterExpressionHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
