# frozen_string_literal: true

require "httparty"
require "json"
require "logger"
require "active_support/core_ext/object/blank"

module GitlabQuality
  module TestTooling
    module ClickHouse
      class Client
        DEFAULT_BATCH_SIZE = 100_000
        LOG_PREFIX = "[ClickHouse]"

        def initialize(url:, database:, username: nil, password: nil, logger: ::Logger.new($stdout, level: 1))
          @url = url
          @database = database
          @username = username
          @password = password
          @logger = logger
        end

        # Perform sql query
        #
        # @param sql [String]
        # @param format [String]
        # @return [Array, Hash, String]
        def query(sql, format: "JSONEachRow")
          logger.debug("Running #{sql}")
          response = post(
            body: sql,
            content_type: "text/plain",
            query_opts: { default_format: format }
          )
          raise "ClickHouse query failed: code: #{response.code}, error: #{response.body}" if response.code != 200

          if format == "JSONEachRow"
            response.body.split("\n").map { |row| JSON.parse(row) }
          elsif %w[JSON JSONCompact].include?(format)
            JSON.parse(response.body.presence || "{}")
          else
            response.body
          end
        end

        # Push data to ClickHouse
        #
        # @param table_name [String]
        # @param data [Array<Hash>]
        # @param batch_size [Integer]
        # @return [void]
        def insert_json_data(table_name, data, batch_size: DEFAULT_BATCH_SIZE) # rubocop:disable Metrics/AbcSize
          raise ArgumentError, "Expected data to be an Array, got #{data.class}" unless data.is_a?(Array)
          raise ArgumentError, "Expected all elements of array to be hashes" unless data.is_a?(Array) && data.all?(Hash)
          raise ArgumentError, "Expected data to not be empty" if data.empty?

          total_batches = (data.size.to_f / batch_size).ceil
          results = data.each_slice(batch_size).with_index.map do |batch, index|
            logger.debug("#{LOG_PREFIX} Pushing batch #{index + 1} of #{total_batches}")
            send_batch(table_name, batch)
          end
          logger.debug("#{LOG_PREFIX} Processed #{results.size} result batches")
          return if results.all? { |res| res[:success] }

          err = results
            .reject { |res| res[:success] }
            .map { |res| "batch_size: #{res[:count]}, err: #{res[:error]}" }
            .join("\n")
          raise "Failures detected when pushing data to ClickHouse, errors:\n#{err}"
        end

        private

        attr_reader :url, :database, :username, :password, :logger

        # Push batch of data
        #
        # @param table_name [String] table name
        # @param batch [Array<Hash>] data batch
        # @return [Hash]
        def send_batch(table_name, batch)
          response = post(
            body: batch.map(&:to_json).join("\n"),
            content_type: 'application/json',
            query_opts: { query: "INSERT INTO #{table_name} FORMAT JSONEachRow" }
          )
          return { success: true, count: batch.size, response: response.body } if response.code == 200

          { success: false, count: batch.size, error: response.body }
        rescue StandardError => e
          { success: false, count: batch.size, error: e.message }
        end

        # Execute post request
        #
        # @param body [String]
        # @param content_type [String]
        # @param query_opts [Hash] additional query options
        # @return [HTTParty::Response]
        def post(body:, content_type:, query_opts: {})
          HTTParty.post(url, {
            body: body,
            headers: { "Content-Type" => content_type },
            query: { database: database, **query_opts }.compact,
            basic_auth: !!(username && password) ? { username: username, password: password } : nil
          }.compact)
        end
      end
    end
  end
end
