# frozen_string_literal: true

require 'json'

module GitlabQuality
  module TestTooling
    module Slack
      class PostToSlack
        MAX_PATTERN_MESSAGE_LENGTH = 150
        MAX_GROUPED_FAILURES_TO_DISPLAY = 10

        def initialize(slack_webhook_url:, channel:, message:, username:, icon_emoji:, environment_issues_file: nil)
          @slack_webhook_url = slack_webhook_url
          @channel = channel
          @message = message
          @username = username
          @icon_emoji = icon_emoji
          @environment_issues_file = environment_issues_file
        end

        def invoke!
          params = {}
          params['channel'] = channel
          params['username'] = username
          params['icon_emoji'] = icon_emoji
          params['text'] = build_message

          Support::HttpRequest.make_http_request(
            method: 'post',
            url: slack_webhook_url,
            params: params,
            show_response: true
          )
        end

        private

        attr_reader :slack_webhook_url, :channel, :message, :username, :icon_emoji, :environment_issues_file

        def build_message
          messages = []
          messages << message if message && !message.empty?
          messages << format_environment_issues if environment_issues_file && File.exist?(environment_issues_file)

          messages.join("\n\n")
        end

        def format_environment_issues
          issues_data = JSON.parse(File.read(environment_issues_file))
          return nil if issues_data.nil? || issues_data['grouped_failures'].empty?

          build_slack_message(issues_data)
        rescue JSON::ParserError => e
          ":x: Error parsing environment issues file: #{e.message}"
        rescue StandardError => e
          ":x: Error formatting environment issues: #{e.message}"
        end

        def format_single_environment_issue(failure)
          pattern_title = pattern_display_name(failure['pattern_name'])

          issue_text = build_issue_header(pattern_title, failure)
          issue_text + build_job_info(failure)
        end

        def pattern_display_name(pattern_name)
          case pattern_name&.downcase
          when /http_500/
            "HTTP 500 Internal Server Errors"
          when /http_400/
            "HTTP 400 Bad Request Errors"
          when /http_503/
            "HTTP 503 Service Unavailable"
          when /timeout/
            "Timeout Errors"
          when /git_rpc|repository/
            "Git/Repository Errors"
          else
            "#{pattern_name&.humanize || 'Unknown'} Errors"
          end
        end

        def build_slack_message(issues_data)
          header = ":warning: *Environment Issues Detected*\n"

          issue_messages = format_issue_messages(issues_data['grouped_failures'])
          truncation_note = build_truncation_note(issues_data['grouped_failures'].size)
          summary = build_summary_text(issues_data['summary'])

          header + issue_messages + truncation_note + summary
        end

        def format_issue_messages(grouped_failures)
          failures_to_show = grouped_failures.first(MAX_GROUPED_FAILURES_TO_DISPLAY)
          failures_to_show.map { |failure| format_single_environment_issue(failure) }.join("\n\n")
        end

        def build_truncation_note(total_failures)
          return "" unless total_failures > MAX_GROUPED_FAILURES_TO_DISPLAY

          "\n_... and #{total_failures - MAX_GROUPED_FAILURES_TO_DISPLAY} more environment issue(s)_"
        end

        def build_issue_header(pattern_title, failure)
          <<~TEXT
            *#{pattern_title}*
            • Affected tests: #{failure['failure_count']}
            • Pattern: `#{truncate_message(failure['normalized_message'])}`
          TEXT
        end

        def build_job_info(failure)
          return "" unless failure['failures']&.any?

          job_urls = failure['failures'].filter_map { |f| f['ci_job_url'] }.uniq
          job_urls.any? ? "• Jobs affected: #{job_urls.size}\n" : ""
        end

        def build_summary_text(summary)
          <<~TEXT

            *Summary:* #{summary['grouped_issues']} environment issue(s) affecting #{summary['total_grouped_failures']} test(s)

            _Note: Future improvements will include direct GitLab issue links and enhanced filtering._
            _Track progress: https://gitlab.com/groups/gitlab-org/quality/quality-engineering/-/epics/168_
          TEXT
        end

        def truncate_message(message)
          text = message.to_s
          text.length > MAX_PATTERN_MESSAGE_LENGTH ? "#{text[0..MAX_PATTERN_MESSAGE_LENGTH]}..." : text
        end
      end
    end
  end
end
