// Package exif provides functionality for extracting EXIF metadata from images.
package exif

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"io"
	"os"
	"os/exec"
	"regexp"

	"gitlab.com/gitlab-org/labkit/log"
)

// ErrRemovingExif is an error returned when there is an issue while removing EXIF metadata from an image.
var ErrRemovingExif = errors.New("error while removing EXIF")

type cleaner struct {
	ctx            context.Context
	preStripExif   *exec.Cmd
	preStripStderr bytes.Buffer
	cmd            *exec.Cmd
	stdout         io.Reader
	stderr         bytes.Buffer
	eof            bool
	waited         bool  // Track if we already called Wait()
	cmdErr         error // Store cmd error
	preStripErr    error // Store preStripExif error
}

// FileType represents the type of an image file.
type FileType int

const (
	// TypeUnknown represents an unknown file type.
	TypeUnknown FileType = iota
	// TypeJPEG represents the JPEG image file type.
	TypeJPEG
	// TypeTIFF represents the TIFF image file type.
	TypeTIFF
)

// NewCleaner creates a new EXIF cleaner instance using the provided context and stdin.
// It processes the input from stdin to remove EXIF data from images.
func NewCleaner(ctx context.Context, stdin io.Reader) (io.ReadCloser, error) {
	c := &cleaner{ctx: ctx}

	if err := c.startProcessing(stdin); err != nil {
		return nil, err
	}

	return c, nil
}

func (c *cleaner) Close() error {
	if c.cmd == nil {
		return nil
	}

	// If we already waited in Read(), return stored errors
	if c.waited {
		if c.preStripErr != nil {
			return c.preStripErr
		}
		return c.cmdErr
	}

	// Otherwise wait now
	cmdErr := c.cmd.Wait()
	var preStripErr error
	if c.preStripExif != nil {
		preStripErr = c.preStripExif.Wait()
	}

	// Return first error
	if preStripErr != nil {
		return preStripErr
	}
	return cmdErr
}

func (c *cleaner) Read(p []byte) (int, error) {
	if c.eof {
		return 0, io.EOF
	}

	n, err := c.stdout.Read(p)
	if err == io.EOF {
		// Wait for BOTH commands
		c.cmdErr = c.cmd.Wait()
		if c.preStripExif != nil {
			c.preStripErr = c.preStripExif.Wait()
		}
		c.waited = true
		c.eof = true

		if c.preStripErr != nil {
			log.WithContextFields(c.ctx, log.Fields{
				"command": c.preStripExif.Args,
				"stderr":  c.preStripStderr.String(),
				"error":   c.preStripErr.Error(),
			}).Print("preStripExif command failed")
			return n, ErrRemovingExif
		}

		if c.cmdErr != nil {
			log.WithContextFields(c.ctx, log.Fields{
				"command": c.cmd.Args,
				"stderr":  c.stderr.String(),
				"error":   c.cmdErr.Error(),
			}).Print("exiftool command failed")
			return n, ErrRemovingExif
		}
	}

	return n, err
}

func (c *cleaner) startProcessing(stdin io.Reader) error {
	var err error

	whitelistedTags := []string{
		"-ResolutionUnit",
		"-XResolution",
		"-YResolution",
		"-YCbCrSubSampling",
		"-YCbCrPositioning",
		"-BitsPerSample",
		"-ImageHeight",
		"-ImageWidth",
		"-ImageSize",
		"-Orientation",
	}

	// Strip IPTC and XMP that might contain unboundedly many tags to avoid problems extracting orientation
	preStripExif := exec.CommandContext(c.ctx, "exiftool", "-IPTC=", "-XMP=", "-")
	preStripExif.Stderr = &c.preStripStderr
	preStripExif.Stdin = stdin
	c.preStripExif = preStripExif

	// Strip all remaining EXIF but preserve Orientation
	args := append([]string{"-all=", "-tagsFromFile", "@"}, whitelistedTags...)
	args = append(args, "-")
	//nolint:gosec // G204: Command is hardcoded "exiftool", args are from whitelisted constant slice
	c.cmd = exec.CommandContext(c.ctx, "exiftool", args...)
	c.cmd.Stderr = &c.stderr
	c.cmd.Stdin, err = preStripExif.StdoutPipe()

	if err != nil {
		return fmt.Errorf("failed to create stdout pipe for removing iptc and xmp: %v", err)
	}

	c.stdout, err = c.cmd.StdoutPipe()
	if err != nil {
		return fmt.Errorf("failed to create stdout pipe for all exif: %v", err)
	}

	if err = preStripExif.Start(); err != nil {
		return fmt.Errorf("start %v: %v", preStripExif.Args, err)
	}

	if err = c.cmd.Start(); err != nil {
		return fmt.Errorf("start %v: %v", c.cmd.Args, err)
	}

	return nil
}

// FileTypeFromSuffix returns the FileType inferred from the filename's suffix.
func FileTypeFromSuffix(filename string) FileType {
	if os.Getenv("SKIP_EXIFTOOL") == "1" {
		return TypeUnknown
	}

	jpegMatch := regexp.MustCompile(`(?i)^[^\n]*\.(jpg|jpeg)$`)
	if jpegMatch.MatchString(filename) {
		return TypeJPEG
	}

	tiffMatch := regexp.MustCompile(`(?i)^[^\n]*\.tiff$`)
	if tiffMatch.MatchString(filename) {
		return TypeTIFF
	}

	return TypeUnknown
}
