/*
 * Decompiled with CFR 0.152.
 */
package jadx.gui.cache.code.disk;

import jadx.api.ICodeCache;
import jadx.api.ICodeInfo;
import jadx.api.JadxArgs;
import jadx.api.JadxDecompiler;
import jadx.core.Jadx;
import jadx.core.dex.nodes.ClassNode;
import jadx.core.dex.nodes.RootNode;
import jadx.core.utils.StringUtils;
import jadx.core.utils.Utils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.core.utils.files.FileUtils;
import jadx.gui.cache.code.disk.CodeMetadataAdapter;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskCodeCache
implements ICodeCache {
    private static final Logger LOG = LoggerFactory.getLogger(DiskCodeCache.class);
    private static final int DATA_FORMAT_VERSION = 15;
    private final Path baseDir;
    private final Path srcDir;
    private final Path metaDir;
    private final Path codeVersionFile;
    private final String codeVersion;
    private final CodeMetadataAdapter codeMetadataAdapter;
    private final ExecutorService writePool;
    private final Map<String, CacheData> clsDataMap;

    public DiskCodeCache(RootNode root, Path projectCacheDir) {
        this.baseDir = projectCacheDir.resolve("code");
        this.srcDir = this.baseDir.resolve("sources");
        this.metaDir = this.baseDir.resolve("metadata");
        this.codeVersionFile = this.baseDir.resolve("code-version");
        JadxArgs args = root.getArgs();
        this.codeVersion = this.buildCodeVersion(args, root.getDecompiler());
        this.writePool = Executors.newFixedThreadPool(args.getThreadsCount());
        this.codeMetadataAdapter = new CodeMetadataAdapter(root);
        this.clsDataMap = this.buildClassDataMap(root.getClasses());
        if (this.checkCodeVersion()) {
            this.loadCachedSet();
        } else {
            this.reset();
        }
    }

    private boolean checkCodeVersion() {
        try {
            if (!Files.exists(this.codeVersionFile, new LinkOption[0])) {
                return false;
            }
            String currentCodeVer = FileUtils.readFile(this.codeVersionFile);
            return currentCodeVer.equals(this.codeVersion);
        }
        catch (Exception e15) {
            LOG.warn("Failed to load code version file", e15);
            return false;
        }
    }

    private void reset() {
        try {
            long start = System.currentTimeMillis();
            LOG.info("Resetting disk code cache, base dir: {}", (Object)this.baseDir.toAbsolutePath());
            FileUtils.deleteDirIfExists(this.baseDir);
            if (Files.exists(this.baseDir.getParent().resolve(this.codeVersionFile.getFileName()), new LinkOption[0])) {
                FileUtils.deleteDirIfExists(this.baseDir.getParent());
            }
            FileUtils.makeDirs(this.srcDir);
            FileUtils.makeDirs(this.metaDir);
            FileUtils.writeFile(this.codeVersionFile, this.codeVersion);
            if (LOG.isDebugEnabled()) {
                LOG.info("Reset done in: {}ms", (Object)(System.currentTimeMillis() - start));
            }
        }
        catch (Exception e15) {
            throw new JadxRuntimeException("Failed to reset code cache", e15);
        }
        finally {
            this.clsDataMap.values().forEach(d15 -> d15.setCached(false));
        }
    }

    @Override
    public void add(String clsFullName, ICodeInfo codeInfo) {
        CacheData clsData = this.getClsData(clsFullName);
        clsData.setTmpCodeInfo(codeInfo);
        clsData.setCached(true);
        this.writePool.execute(() -> {
            try {
                int clsId = clsData.getClsId();
                ICodeInfo code = clsData.getTmpCodeInfo();
                if (code != null) {
                    FileUtils.writeFile(this.getJavaFile(clsId), code.getCodeStr());
                    this.codeMetadataAdapter.write(this.getMetadataFile(clsId), code.getCodeMetadata());
                }
            }
            catch (Exception e15) {
                LOG.error("Failed to write code cache for " + clsFullName, e15);
                this.remove(clsFullName);
            }
            finally {
                clsData.setTmpCodeInfo(null);
            }
        });
    }

    @Override
    @Nullable
    public String getCode(String clsFullName) {
        try {
            if (!this.contains(clsFullName)) {
                return null;
            }
            CacheData clsData = this.getClsData(clsFullName);
            ICodeInfo tmpCodeInfo = clsData.getTmpCodeInfo();
            if (tmpCodeInfo != null) {
                return tmpCodeInfo.getCodeStr();
            }
            Path javaFile = this.getJavaFile(clsData.getClsId());
            if (!Files.exists(javaFile, new LinkOption[0])) {
                return null;
            }
            return FileUtils.readFile(javaFile);
        }
        catch (Exception e15) {
            LOG.error("Failed to read class code for {}", (Object)clsFullName, (Object)e15);
            return null;
        }
    }

    @Override
    @NotNull
    public ICodeInfo get(String clsFullName) {
        try {
            if (!this.contains(clsFullName)) {
                return ICodeInfo.EMPTY;
            }
            CacheData clsData = this.getClsData(clsFullName);
            ICodeInfo tmpCodeInfo = clsData.getTmpCodeInfo();
            if (tmpCodeInfo != null) {
                return tmpCodeInfo;
            }
            int clsId = clsData.getClsId();
            Path javaFile = this.getJavaFile(clsId);
            if (!Files.exists(javaFile, new LinkOption[0])) {
                return ICodeInfo.EMPTY;
            }
            String code = FileUtils.readFile(javaFile);
            return this.codeMetadataAdapter.readAndBuild(this.getMetadataFile(clsId), code);
        }
        catch (Exception e15) {
            LOG.error("Failed to read code cache for {}", (Object)clsFullName, (Object)e15);
            return ICodeInfo.EMPTY;
        }
    }

    @Override
    public boolean contains(String clsFullName) {
        return this.getClsData(clsFullName).isCached();
    }

    @Override
    public void remove(String clsFullName) {
        try {
            CacheData clsData = this.getClsData(clsFullName);
            if (clsData.isCached()) {
                clsData.setCached(false);
                if (clsData.getTmpCodeInfo() == null) {
                    LOG.debug("Removing class info from disk: {}", (Object)clsFullName);
                    int clsId = clsData.getClsId();
                    Files.deleteIfExists(this.getJavaFile(clsId));
                    Files.deleteIfExists(this.getMetadataFile(clsId));
                } else {
                    clsData.setTmpCodeInfo(null);
                }
            }
        }
        catch (Exception e15) {
            throw new JadxRuntimeException("Failed to remove code cache for " + clsFullName, e15);
        }
    }

    private String buildCodeVersion(JadxArgs args, @Nullable JadxDecompiler decompiler) {
        ArrayList<File> inputFiles = new ArrayList<File>(args.getInputFiles());
        if (args.getGeneratedRenamesMappingFileMode().shouldRead() && args.getGeneratedRenamesMappingFile() != null && args.getGeneratedRenamesMappingFile().exists()) {
            inputFiles.add(args.getGeneratedRenamesMappingFile());
        }
        return "15:" + Jadx.getVersion() + ":" + args.makeCodeArgsHash(decompiler) + ":" + FileUtils.buildInputsHash(Utils.collectionMap(inputFiles, File::toPath));
    }

    private CacheData getClsData(String clsFullName) {
        CacheData clsData = this.clsDataMap.get(clsFullName);
        if (clsData == null) {
            throw new JadxRuntimeException("Unknown class name: " + clsFullName);
        }
        return clsData;
    }

    private void loadCachedSet() {
        long start = System.currentTimeMillis();
        BitSet cachedSet = new BitSet(this.clsDataMap.size());
        try (Stream<Path> stream = Files.walk(this.metaDir, new FileVisitOption[0]);){
            stream.forEach(file2 -> {
                String fileName = file2.getFileName().toString();
                if (fileName.endsWith(".jadxmd")) {
                    String idStr = StringUtils.removeSuffix(fileName, ".jadxmd");
                    int clsId = Integer.parseInt(idStr, 16);
                    cachedSet.set(clsId);
                }
            });
        }
        catch (Exception e15) {
            throw new JadxRuntimeException("Failed to enumerate cached classes", e15);
        }
        int count = 0;
        for (CacheData data : this.clsDataMap.values()) {
            int clsId = data.getClsId();
            if (!cachedSet.get(clsId)) continue;
            data.setCached(true);
            ++count;
        }
        LOG.info("Found {} classes in disk cache, time: {}ms, dir: {}", count, System.currentTimeMillis() - start, this.metaDir.getParent());
    }

    private Path getJavaFile(int clsId) {
        return this.srcDir.resolve(this.getPathForClsId(clsId, ".java"));
    }

    private Path getMetadataFile(int clsId) {
        return this.metaDir.resolve(this.getPathForClsId(clsId, ".jadxmd"));
    }

    private Path getPathForClsId(int clsId, String ext) {
        String firstByte = FileUtils.byteToHex(clsId);
        return Paths.get(firstByte, FileUtils.intToHex(clsId) + ext);
    }

    private Map<String, CacheData> buildClassDataMap(List<ClassNode> classes) {
        int clsCount = classes.size();
        HashMap<String, CacheData> map = new HashMap<String, CacheData>(clsCount);
        for (int i15 = 0; i15 < clsCount; ++i15) {
            ClassNode cls = classes.get(i15);
            map.put(cls.getRawName(), new CacheData(i15));
        }
        return map;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        DiskCodeCache diskCodeCache = this;
        synchronized (diskCodeCache) {
            try {
                this.writePool.shutdown();
                boolean completed = this.writePool.awaitTermination(1L, TimeUnit.MINUTES);
                if (!completed) {
                    LOG.warn("Disk code cache closing terminated by timeout");
                }
            }
            catch (InterruptedException e15) {
                LOG.error("Failed to close disk code cache", e15);
            }
        }
    }

    private static final class CacheData {
        private final int clsId;
        private boolean cached;
        @Nullable
        private ICodeInfo tmpCodeInfo;

        public CacheData(int clsId) {
            this.clsId = clsId;
        }

        public int getClsId() {
            return this.clsId;
        }

        public boolean isCached() {
            return this.cached;
        }

        public void setCached(boolean cached) {
            this.cached = cached;
        }

        @Nullable
        public ICodeInfo getTmpCodeInfo() {
            return this.tmpCodeInfo;
        }

        public void setTmpCodeInfo(@Nullable ICodeInfo tmpCodeInfo) {
            this.tmpCodeInfo = tmpCodeInfo;
        }
    }
}

