# The result of {#run running} a {#match}.
class MetasploitDataModels::AutomaticExploitation::MatchResult < ApplicationRecord
  #
  # CONSTANTS
  #

  # Running associated exploit did NOT create a session
  FAILED = "failed"
  # Running associated exploit created a session
  SUCCEEDED = "succeeded"

  # Valid values for {#state}
  VALID_STATES = [FAILED, SUCCEEDED]

  #
  # Associations
  #

  # A {MetasploitDataModels::AutomaticExploitation::Match#module_detail Metasploit Module} matched to
  # {MetasploitDataModels::AutomaticExploitation::Match#matchable Mdm::Host or Mdm::Service}.
  belongs_to :match,
             class_name: 'MetasploitDataModels::AutomaticExploitation::Match',
             inverse_of: :match_results,
             dependent: :destroy

  # A mass automatic exploitation run.
  belongs_to :run,
             inverse_of: :match_results,
             class_name: 'MetasploitDataModels::AutomaticExploitation::Run'

  #
  # Attributes
  #

  # @!attribute state
  #   Whether the {#run} of {#match} succeeded.
  #
  #   @return ['failed', 'succeeded']

  #
  # Validations
  #

  # must be present and one of allowable values
  validates :state,
            presence: true,
            inclusion: VALID_STATES

  #
  # Scopes
  #

  # Runs of {#match} that failed
  scope :failed, lambda { where(state:"failed") }

  # Runs of {#match} that succeeded
  scope :succeeded, lambda { where(state:"succeeded") }

  # Runs of {#match} by workspace ID
  scope :by_workspace, lambda { |workspace_id|
                       joins(
                         MetasploitDataModels::AutomaticExploitation::MatchResult.join_association(:match),
                         MetasploitDataModels::AutomaticExploitation::Match.join_association(:match_set)
                       ).where(
                         MetasploitDataModels::AutomaticExploitation::MatchSet[:workspace_id].eq(workspace_id),
                       )
                     }

  Metasploit::Concern.run(self)
end

