"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineResourceAccessManagementRoutes = defineResourceAccessManagementRoutes;
var _configSchema = require("@osd/config-schema");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

/* ******************* Schemas ******************* */

const stringArray = _configSchema.schema.arrayOf(_configSchema.schema.string());

// Inner object allowed on each access level
const recipientsSchema = _configSchema.schema.object({
  users: _configSchema.schema.maybe(stringArray),
  roles: _configSchema.schema.maybe(stringArray),
  backend_roles: _configSchema.schema.maybe(stringArray)
}, {
  unknowns: 'forbid',
  validate: v => {
    if (!v.users && !v.roles && !v.backend_roles) {
      return 'Each share_with entry must include at least one of "users", "roles", or "backend_roles".';
    }
  }
});

// share_with shape used by both PUT and PATCH (add/revoke)
const shareWithSchema = _configSchema.schema.recordOf(_configSchema.schema.string(), recipientsSchema);

// PUT body must include non-empty share_with
const putBodySchema = _configSchema.schema.object({
  resource_id: _configSchema.schema.string({
    minLength: 1
  }),
  resource_type: _configSchema.schema.string({
    minLength: 1
  }),
  share_with: shareWithSchema
}, {
  unknowns: 'allow',
  validate: val => {
    if (Object.keys(val.share_with).length === 0) {
      return '"share_with" must not be empty.';
    }
  }
});

// PATCH schema: add/revoke must be shareWithSchema
const postBodySchema = _configSchema.schema.object({
  resource_id: _configSchema.schema.string({
    minLength: 1
  }),
  resource_type: _configSchema.schema.string({
    minLength: 1
  }),
  add: _configSchema.schema.maybe(shareWithSchema),
  revoke: _configSchema.schema.maybe(shareWithSchema)
}, {
  validate: value => {
    if (!value.add && !value.revoke) {
      return 'Request body must include at least one of "add" or "revoke".';
    }
  }
});

/* *******************Route definitions******************* */

const SECURITY_RESOURCE_API_PREFIX = '/_plugins/_security/api/resource';
const LIST_TYPES_API = `${SECURITY_RESOURCE_API_PREFIX}/types`;
const LIST_SHARING_INFO_API = `${SECURITY_RESOURCE_API_PREFIX}/list`;
const SHARE_API = `${SECURITY_RESOURCE_API_PREFIX}/share`;
function defineResourceAccessManagementRoutes(router, dataSourceEnabled) {
  // GET registered resource types
  router.get({
    path: '/api/resource/types',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query;
      if (dataSourceEnabled && (_request$query = request.query) !== null && _request$query !== void 0 && _request$query.dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        const result = await client.callAPI('opensearch_security.listResourceTypes', {});
        return response.ok({
          body: result
        });
      } else {
        const client = context.core.opensearch.client.asCurrentUser;
        const result = await client.transport.request({
          method: 'GET',
          path: LIST_TYPES_API
        });
        return response.ok({
          body: result.body
        });
      }
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // GET accessible shared resources filtered by resourceType
  router.get({
    path: '/api/resource/list',
    validate: {
      query: _configSchema.schema.object({
        resourceType: _configSchema.schema.string(),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query2;
      const {
        resourceType
      } = request.query;
      if (dataSourceEnabled && (_request$query2 = request.query) !== null && _request$query2 !== void 0 && _request$query2.dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        const result = await client.callAPI('opensearch_security.listResourceSharing', {
          resource_type: resourceType
        });
        return response.ok({
          body: result
        });
      } else {
        const client = context.core.opensearch.client.asCurrentUser;
        const result = await client.transport.request({
          method: 'GET',
          path: LIST_SHARING_INFO_API,
          querystring: {
            resource_type: resourceType
          }
        });
        return response.ok({
          body: result.body
        });
      }
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // GET sharing info for a specific resource
  router.get({
    path: '/api/resource/view',
    validate: {
      query: _configSchema.schema.object({
        resourceId: _configSchema.schema.string(),
        resourceType: _configSchema.schema.string(),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query3;
      const {
        resourceId,
        resourceType
      } = request.query;
      if (dataSourceEnabled && (_request$query3 = request.query) !== null && _request$query3 !== void 0 && _request$query3.dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        const result = await client.callAPI('opensearch_security.getResourceSharing', {
          resource_id: resourceId,
          resource_type: resourceType
        });
        return response.ok({
          body: result
        });
      } else {
        const client = context.core.opensearch.client.asCurrentUser;
        const result = await client.transport.request({
          method: 'GET',
          path: SHARE_API,
          querystring: {
            resource_id: resourceId,
            resource_type: resourceType
          }
        });
        return response.ok({
          body: result.body
        });
      }
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // PUT share a resource — requires `share_with`
  router.put({
    path: '/api/resource/share',
    validate: {
      body: putBodySchema,
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query4;
      if (dataSourceEnabled && (_request$query4 = request.query) !== null && _request$query4 !== void 0 && _request$query4.dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        const result = await client.callAPI('opensearch_security.shareResource', request.body);
        return response.ok({
          body: result
        });
      } else {
        const client = context.core.opensearch.client.asCurrentUser;
        const result = await client.transport.request({
          method: 'PUT',
          path: SHARE_API,
          body: request.body
        });
        return response.ok({
          body: result.body
        });
      }
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // POST update sharing — `add`/`revoke` adhere to share_with schema
  router.post({
    path: '/api/resource/update_sharing',
    validate: {
      body: postBodySchema,
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query5;
      if (dataSourceEnabled && (_request$query5 = request.query) !== null && _request$query5 !== void 0 && _request$query5.dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        const result = await client.callAPI('opensearch_security.updateResourceSharing', request.body);
        return response.ok({
          body: result
        });
      } else {
        const client = context.core.opensearch.client.asCurrentUser;
        const result = await client.transport.request({
          method: 'POST',
          path: SHARE_API,
          body: request.body
        });
        return response.ok({
          body: result.body
        });
      }
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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