// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-FileCopyrightText: Copyright 2008 Sandia Corporation
// SPDX-License-Identifier: LicenseRef-BSD-3-Clause-Sandia-USGov
/**
 * @class   vtkQtListView
 * @brief   A VTK view based on a Qt List view.
 *
 *
 * vtkQtListView is a VTK view using an underlying QListView.
 *
 * @par Thanks:
 * Thanks to Brian Wylie from Sandia National Laboratories for implementing
 * this class
 */

#ifndef vtkQtListView_h
#define vtkQtListView_h

#include "vtkQtView.h"
#include "vtkViewsQtModule.h" // For export macro

#include "vtkSmartPointer.h" // Needed for member variables
#include <QImage>            // Needed for the icon methods
#include <QPointer>          // Needed for the internal list view

class QItemSelection;
class QSortFilterProxyModel;
class QListView;

VTK_ABI_NAMESPACE_BEGIN
class vtkAttributeDataToTableFilter;
class vtkApplyColors;
class vtkQtTableModelAdapter;

class VTKVIEWSQT_EXPORT vtkQtListView : public vtkQtView
{
  Q_OBJECT

public:
  static vtkQtListView* New();
  vtkTypeMacro(vtkQtListView, vtkQtView);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Get the main container of this view (a  QWidget).
   * The application typically places the view with a call
   * to GetWidget(): something like this
   * this->ui->box->layout()->addWidget(this->View->GetWidget());
   */
  QWidget* GetWidget() override;

  enum
  {
    FIELD_DATA = 0,
    POINT_DATA = 1,
    CELL_DATA = 2,
    VERTEX_DATA = 3,
    EDGE_DATA = 4,
    ROW_DATA = 5,
  };

  ///@{
  /**
   * The field type to copy into the output table.
   * Should be one of FIELD_DATA, POINT_DATA, CELL_DATA, VERTEX_DATA, EDGE_DATA.
   */
  vtkGetMacro(FieldType, int);
  void SetFieldType(int);
  ///@}

  /**
   * Enable drag and drop on this widget
   */
  void SetEnableDragDrop(bool);

  /**
   * Have the view alternate its row colors
   */
  void SetAlternatingRowColors(bool);

  /**
   * The strategy for how to decorate rows.
   * Should be one of vtkQtTableModelAdapter::COLORS,
   * vtkQtTableModelAdapter::ICONS, or
   * vtkQtTableModelAdapter::NONE. Default is NONE.
   */
  void SetDecorationStrategy(int);

  ///@{
  /**
   * The array to use for coloring items in view.  Default is "color".
   */
  void SetColorArrayName(const char* name);
  const char* GetColorArrayName();
  ///@}

  ///@{
  /**
   * Whether to color vertices.  Default is off.
   */
  void SetColorByArray(bool vis);
  bool GetColorByArray();
  vtkBooleanMacro(ColorByArray, bool);
  ///@}

  /**
   * The column to display
   */
  void SetVisibleColumn(int col);

  /**
   * The column used to filter on
   */
#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
  void SetFilterRegExp(const QRegularExpression& pattern);
#else
  void SetFilterRegExp(const QRegExp& pattern);
#endif

  ///@{
  /**
   * Set the icon ivars. Only used if the decoration strategy is set to ICONS.
   */
  void SetIconSheet(QImage sheet);
  void SetIconSize(int w, int h);
  void SetIconSheetSize(int w, int h);
  void SetIconArrayName(const char* name);
  ///@}

  void ApplyViewTheme(vtkViewTheme* theme) override;

  /**
   * Updates the view.
   */
  void Update() override;

protected:
  vtkQtListView();
  ~vtkQtListView() override;

  void AddRepresentationInternal(vtkDataRepresentation* rep) override;
  void RemoveRepresentationInternal(vtkDataRepresentation* rep) override;

private Q_SLOTS:
  void slotQtSelectionChanged(const QItemSelection&, const QItemSelection&);

private: // NOLINT(readability-redundant-access-specifiers)
  void SetVTKSelection();

  vtkMTimeType LastSelectionMTime;
  vtkMTimeType LastInputMTime;
  vtkMTimeType LastMTime;

  vtkSetStringMacro(ColorArrayNameInternal);
  vtkGetStringMacro(ColorArrayNameInternal);
  vtkSetStringMacro(IconIndexArrayNameInternal);
  vtkGetStringMacro(IconIndexArrayNameInternal);

  QPointer<QListView> ListView;
  vtkQtTableModelAdapter* TableAdapter;
  QSortFilterProxyModel* TableSorter;
  char* ColorArrayNameInternal;
  char* IconIndexArrayNameInternal;
  char* VisibleColumnName;
  bool SortSelectionToTop;
  bool ApplyRowColors;
  int FieldType;
  int VisibleColumn;

  vtkSmartPointer<vtkAttributeDataToTableFilter> DataObjectToTable;
  vtkSmartPointer<vtkApplyColors> ApplyColors;

  vtkQtListView(const vtkQtListView&) = delete;
  void operator=(const vtkQtListView&) = delete;
};

VTK_ABI_NAMESPACE_END
#endif
