﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/DescribeDBClusterBacktracksMessage">AWS
 * API Reference</a></p>
 */
class DescribeDBClusterBacktracksRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBClusterBacktracksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBClusterBacktracks"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The DB cluster identifier of the DB cluster to be described. This parameter
   * is stored as a lowercase string.</p> <p>Constraints:</p> <ul> <li> <p>Must
   * contain from 1 to 63 alphanumeric characters or hyphens.</p> </li> <li> <p>First
   * character must be a letter.</p> </li> <li> <p>Can't end with a hyphen or contain
   * two consecutive hyphens.</p> </li> </ul> <p>Example: <code>my-cluster1</code>
   * </p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  DescribeDBClusterBacktracksRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If specified, this value is the backtrack identifier of the backtrack to be
   * described.</p> <p>Constraints:</p> <ul> <li> <p>Must contain a valid universally
   * unique identifier (UUID). For more information about UUIDs, see <a
   * href="https://en.wikipedia.org/wiki/Universally_unique_identifier">Universally
   * unique identifier</a>.</p> </li> </ul> <p>Example:
   * <code>123e4567-e89b-12d3-a456-426655440000</code> </p>
   */
  inline const Aws::String& GetBacktrackIdentifier() const { return m_backtrackIdentifier; }
  inline bool BacktrackIdentifierHasBeenSet() const { return m_backtrackIdentifierHasBeenSet; }
  template <typename BacktrackIdentifierT = Aws::String>
  void SetBacktrackIdentifier(BacktrackIdentifierT&& value) {
    m_backtrackIdentifierHasBeenSet = true;
    m_backtrackIdentifier = std::forward<BacktrackIdentifierT>(value);
  }
  template <typename BacktrackIdentifierT = Aws::String>
  DescribeDBClusterBacktracksRequest& WithBacktrackIdentifier(BacktrackIdentifierT&& value) {
    SetBacktrackIdentifier(std::forward<BacktrackIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that specifies one or more DB clusters to describe. Supported
   * filters include the following:</p> <ul> <li> <p>
   * <code>db-cluster-backtrack-id</code> - Accepts backtrack identifiers. The
   * results list includes information about only the backtracks identified by these
   * identifiers.</p> </li> <li> <p> <code>db-cluster-backtrack-status</code> -
   * Accepts any of the following backtrack status values:</p> <ul> <li> <p>
   * <code>applying</code> </p> </li> <li> <p> <code>completed</code> </p> </li> <li>
   * <p> <code>failed</code> </p> </li> <li> <p> <code>pending</code> </p> </li>
   * </ul> <p>The results list includes information about only the backtracks
   * identified by these values.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBClusterBacktracksRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBClusterBacktracksRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so you can retrieve the remaining
   * results.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBClusterBacktracksRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>DescribeDBClusterBacktracks</code> request. If this parameter is
   * specified, the response includes only records beyond the marker, up to the value
   * specified by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBClusterBacktracksRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBClusterIdentifier;
  bool m_dBClusterIdentifierHasBeenSet = false;

  Aws::String m_backtrackIdentifier;
  bool m_backtrackIdentifierHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
