﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/cognito-idp/model/MFAOptionType.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 * <p>You can use this parameter to set an MFA configuration that uses the SMS
 * delivery medium.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-idp-2016-04-18/AdminSetUserSettingsRequest">AWS
 * API Reference</a></p>
 */
class AdminSetUserSettingsRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API AdminSetUserSettingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AdminSetUserSettings"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the user pool that contains the user whose options you're
   * setting.</p>
   */
  inline const Aws::String& GetUserPoolId() const { return m_userPoolId; }
  inline bool UserPoolIdHasBeenSet() const { return m_userPoolIdHasBeenSet; }
  template <typename UserPoolIdT = Aws::String>
  void SetUserPoolId(UserPoolIdT&& value) {
    m_userPoolIdHasBeenSet = true;
    m_userPoolId = std::forward<UserPoolIdT>(value);
  }
  template <typename UserPoolIdT = Aws::String>
  AdminSetUserSettingsRequest& WithUserPoolId(UserPoolIdT&& value) {
    SetUserPoolId(std::forward<UserPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the user that you want to query or modify. The value of this
   * parameter is typically your user's username, but it can be any of their alias
   * attributes. If <code>username</code> isn't an alias attribute in your user pool,
   * this value must be the <code>sub</code> of a local user or the username of a
   * user from a third-party IdP.</p>
   */
  inline const Aws::String& GetUsername() const { return m_username; }
  inline bool UsernameHasBeenSet() const { return m_usernameHasBeenSet; }
  template <typename UsernameT = Aws::String>
  void SetUsername(UsernameT&& value) {
    m_usernameHasBeenSet = true;
    m_username = std::forward<UsernameT>(value);
  }
  template <typename UsernameT = Aws::String>
  AdminSetUserSettingsRequest& WithUsername(UsernameT&& value) {
    SetUsername(std::forward<UsernameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>You can use this parameter only to set an SMS configuration that uses SMS for
   * delivery.</p>
   */
  inline const Aws::Vector<MFAOptionType>& GetMFAOptions() const { return m_mFAOptions; }
  inline bool MFAOptionsHasBeenSet() const { return m_mFAOptionsHasBeenSet; }
  template <typename MFAOptionsT = Aws::Vector<MFAOptionType>>
  void SetMFAOptions(MFAOptionsT&& value) {
    m_mFAOptionsHasBeenSet = true;
    m_mFAOptions = std::forward<MFAOptionsT>(value);
  }
  template <typename MFAOptionsT = Aws::Vector<MFAOptionType>>
  AdminSetUserSettingsRequest& WithMFAOptions(MFAOptionsT&& value) {
    SetMFAOptions(std::forward<MFAOptionsT>(value));
    return *this;
  }
  template <typename MFAOptionsT = MFAOptionType>
  AdminSetUserSettingsRequest& AddMFAOptions(MFAOptionsT&& value) {
    m_mFAOptionsHasBeenSet = true;
    m_mFAOptions.emplace_back(std::forward<MFAOptionsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_userPoolId;
  bool m_userPoolIdHasBeenSet = false;

  Aws::String m_username;
  bool m_usernameHasBeenSet = false;

  Aws::Vector<MFAOptionType> m_mFAOptions;
  bool m_mFAOptionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
