﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/DataSourceConfigurationInput.h>
#include <aws/datazone/model/EnableSetting.h>
#include <aws/datazone/model/FormInput.h>
#include <aws/datazone/model/RecommendationConfiguration.h>
#include <aws/datazone/model/ScheduleConfiguration.h>

#include <utility>

namespace Aws {
namespace DataZone {
namespace Model {

/**
 */
class CreateDataSourceRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API CreateDataSourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDataSource"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the data source.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateDataSourceRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the data source.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateDataSourceRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon DataZone domain where the data source is created.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  CreateDataSourceRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone project in which you want to add this
   * data source.</p>
   */
  inline const Aws::String& GetProjectIdentifier() const { return m_projectIdentifier; }
  inline bool ProjectIdentifierHasBeenSet() const { return m_projectIdentifierHasBeenSet; }
  template <typename ProjectIdentifierT = Aws::String>
  void SetProjectIdentifier(ProjectIdentifierT&& value) {
    m_projectIdentifierHasBeenSet = true;
    m_projectIdentifier = std::forward<ProjectIdentifierT>(value);
  }
  template <typename ProjectIdentifierT = Aws::String>
  CreateDataSourceRequest& WithProjectIdentifier(ProjectIdentifierT&& value) {
    SetProjectIdentifier(std::forward<ProjectIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the Amazon DataZone environment to which the data
   * source publishes assets. </p>
   */
  inline const Aws::String& GetEnvironmentIdentifier() const { return m_environmentIdentifier; }
  inline bool EnvironmentIdentifierHasBeenSet() const { return m_environmentIdentifierHasBeenSet; }
  template <typename EnvironmentIdentifierT = Aws::String>
  void SetEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    m_environmentIdentifierHasBeenSet = true;
    m_environmentIdentifier = std::forward<EnvironmentIdentifierT>(value);
  }
  template <typename EnvironmentIdentifierT = Aws::String>
  CreateDataSourceRequest& WithEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    SetEnvironmentIdentifier(std::forward<EnvironmentIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the connection.</p>
   */
  inline const Aws::String& GetConnectionIdentifier() const { return m_connectionIdentifier; }
  inline bool ConnectionIdentifierHasBeenSet() const { return m_connectionIdentifierHasBeenSet; }
  template <typename ConnectionIdentifierT = Aws::String>
  void SetConnectionIdentifier(ConnectionIdentifierT&& value) {
    m_connectionIdentifierHasBeenSet = true;
    m_connectionIdentifier = std::forward<ConnectionIdentifierT>(value);
  }
  template <typename ConnectionIdentifierT = Aws::String>
  CreateDataSourceRequest& WithConnectionIdentifier(ConnectionIdentifierT&& value) {
    SetConnectionIdentifier(std::forward<ConnectionIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the data source. In Amazon DataZone, you can use data sources to
   * import technical metadata of assets (data) from the source databases or data
   * warehouses into Amazon DataZone. In the current release of Amazon DataZone, you
   * can create and run data sources for Amazon Web Services Glue and Amazon
   * Redshift.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  CreateDataSourceRequest& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the configuration of the data source. It can be set to either
   * <code>glueRunConfiguration</code> or <code>redshiftRunConfiguration</code>.</p>
   */
  inline const DataSourceConfigurationInput& GetConfiguration() const { return m_configuration; }
  inline bool ConfigurationHasBeenSet() const { return m_configurationHasBeenSet; }
  template <typename ConfigurationT = DataSourceConfigurationInput>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = DataSourceConfigurationInput>
  CreateDataSourceRequest& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the business name generation is to be enabled for this data
   * source.</p>
   */
  inline const RecommendationConfiguration& GetRecommendation() const { return m_recommendation; }
  inline bool RecommendationHasBeenSet() const { return m_recommendationHasBeenSet; }
  template <typename RecommendationT = RecommendationConfiguration>
  void SetRecommendation(RecommendationT&& value) {
    m_recommendationHasBeenSet = true;
    m_recommendation = std::forward<RecommendationT>(value);
  }
  template <typename RecommendationT = RecommendationConfiguration>
  CreateDataSourceRequest& WithRecommendation(RecommendationT&& value) {
    SetRecommendation(std::forward<RecommendationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the data source is enabled.</p>
   */
  inline EnableSetting GetEnableSetting() const { return m_enableSetting; }
  inline bool EnableSettingHasBeenSet() const { return m_enableSettingHasBeenSet; }
  inline void SetEnableSetting(EnableSetting value) {
    m_enableSettingHasBeenSet = true;
    m_enableSetting = value;
  }
  inline CreateDataSourceRequest& WithEnableSetting(EnableSetting value) {
    SetEnableSetting(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The schedule of the data source runs.</p>
   */
  inline const ScheduleConfiguration& GetSchedule() const { return m_schedule; }
  inline bool ScheduleHasBeenSet() const { return m_scheduleHasBeenSet; }
  template <typename ScheduleT = ScheduleConfiguration>
  void SetSchedule(ScheduleT&& value) {
    m_scheduleHasBeenSet = true;
    m_schedule = std::forward<ScheduleT>(value);
  }
  template <typename ScheduleT = ScheduleConfiguration>
  CreateDataSourceRequest& WithSchedule(ScheduleT&& value) {
    SetSchedule(std::forward<ScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the assets that this data source creates in the inventory
   * are to be also automatically published to the catalog.</p>
   */
  inline bool GetPublishOnImport() const { return m_publishOnImport; }
  inline bool PublishOnImportHasBeenSet() const { return m_publishOnImportHasBeenSet; }
  inline void SetPublishOnImport(bool value) {
    m_publishOnImportHasBeenSet = true;
    m_publishOnImport = value;
  }
  inline CreateDataSourceRequest& WithPublishOnImport(bool value) {
    SetPublishOnImport(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata forms that are to be attached to the assets that this data
   * source works with.</p>
   */
  inline const Aws::Vector<FormInput>& GetAssetFormsInput() const { return m_assetFormsInput; }
  inline bool AssetFormsInputHasBeenSet() const { return m_assetFormsInputHasBeenSet; }
  template <typename AssetFormsInputT = Aws::Vector<FormInput>>
  void SetAssetFormsInput(AssetFormsInputT&& value) {
    m_assetFormsInputHasBeenSet = true;
    m_assetFormsInput = std::forward<AssetFormsInputT>(value);
  }
  template <typename AssetFormsInputT = Aws::Vector<FormInput>>
  CreateDataSourceRequest& WithAssetFormsInput(AssetFormsInputT&& value) {
    SetAssetFormsInput(std::forward<AssetFormsInputT>(value));
    return *this;
  }
  template <typename AssetFormsInputT = FormInput>
  CreateDataSourceRequest& AddAssetFormsInput(AssetFormsInputT&& value) {
    m_assetFormsInputHasBeenSet = true;
    m_assetFormsInput.emplace_back(std::forward<AssetFormsInputT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that is provided to ensure the
   * idempotency of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateDataSourceRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_description;

  Aws::String m_domainIdentifier;

  Aws::String m_projectIdentifier;

  Aws::String m_environmentIdentifier;

  Aws::String m_connectionIdentifier;

  Aws::String m_type;

  DataSourceConfigurationInput m_configuration;

  RecommendationConfiguration m_recommendation;

  EnableSetting m_enableSetting{EnableSetting::NOT_SET};

  ScheduleConfiguration m_schedule;

  bool m_publishOnImport{false};

  Aws::Vector<FormInput> m_assetFormsInput;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_domainIdentifierHasBeenSet = false;
  bool m_projectIdentifierHasBeenSet = false;
  bool m_environmentIdentifierHasBeenSet = false;
  bool m_connectionIdentifierHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_configurationHasBeenSet = false;
  bool m_recommendationHasBeenSet = false;
  bool m_enableSettingHasBeenSet = false;
  bool m_scheduleHasBeenSet = false;
  bool m_publishOnImportHasBeenSet = false;
  bool m_assetFormsInputHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
