﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes the Amazon S3 bucket for the disk image.</p><p><h3>See Also:</h3>
 * <a href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/UserBucket">AWS
 * API Reference</a></p>
 */
class UserBucket {
 public:
  AWS_EC2_API UserBucket() = default;
  AWS_EC2_API UserBucket(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API UserBucket& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the Amazon S3 bucket where the disk image is located.</p>
   */
  inline const Aws::String& GetS3Bucket() const { return m_s3Bucket; }
  inline bool S3BucketHasBeenSet() const { return m_s3BucketHasBeenSet; }
  template <typename S3BucketT = Aws::String>
  void SetS3Bucket(S3BucketT&& value) {
    m_s3BucketHasBeenSet = true;
    m_s3Bucket = std::forward<S3BucketT>(value);
  }
  template <typename S3BucketT = Aws::String>
  UserBucket& WithS3Bucket(S3BucketT&& value) {
    SetS3Bucket(std::forward<S3BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The file name of the disk image.</p>
   */
  inline const Aws::String& GetS3Key() const { return m_s3Key; }
  inline bool S3KeyHasBeenSet() const { return m_s3KeyHasBeenSet; }
  template <typename S3KeyT = Aws::String>
  void SetS3Key(S3KeyT&& value) {
    m_s3KeyHasBeenSet = true;
    m_s3Key = std::forward<S3KeyT>(value);
  }
  template <typename S3KeyT = Aws::String>
  UserBucket& WithS3Key(S3KeyT&& value) {
    SetS3Key(std::forward<S3KeyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_s3Bucket;
  bool m_s3BucketHasBeenSet = false;

  Aws::String m_s3Key;
  bool m_s3KeyHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
