/* +------------------------------------------------------------------------+
   |                     Mobile Robot Programming Toolkit (MRPT)            |
   |                          https://www.mrpt.org/                         |
   |                                                                        |
   | Copyright (c) 2005-2024, Individual contributors, see AUTHORS file     |
   | See: https://www.mrpt.org/Authors - All rights reserved.               |
   | Released under BSD License. See: https://www.mrpt.org/License          |
   +------------------------------------------------------------------------+ */

#include "img-precomp.h"  // Precompiled headers
//
#include <mrpt/config.h>
#include <mrpt/img/CImage.h>
#include <mrpt/io/CStream.h>

#include "CImage_impl.h"

// Universal include for all versions of OpenCV
#include <mrpt/3rdparty/do_opencv_includes.h>

using namespace mrpt;
using namespace mrpt::img;

// ---------------------------------------------------------------------------------------
//							START OF JPEG FUNCTIONS PART
// ---------------------------------------------------------------------------------------
/* Expanded data destination object for stdio output */

//#undef INT32
#undef FAR
#define XMD_H

using mrpt::io::CStream;

#if MRPT_HAS_JPEG
#include <jpeglib.h>

#include <cstdio>
#define mrpt_jpeg_source_mgr jpeg_source_mgr

typedef struct
{
  struct jpeg_destination_mgr pub; /* public fields */

  CStream* out;   /* target stream */
  JOCTET* buffer; /* start of buffer */
} mrpt_destination_mgr;

using mrpt_dest_ptr = mrpt_destination_mgr*;

#define OUTPUT_BUF_SIZE 4096 /* choose an efficiently fwrite'able size */

/*
 * Initialize destination --- called by jpeg_start_compress
 * before any data is actually written.
 */

METHODDEF(void)
init_destination(j_compress_ptr cinfo)
{
  auto dest = reinterpret_cast<mrpt_dest_ptr>(cinfo->dest);

  /* Allocate the output buffer --- it will be released when done with image
   */
  dest->buffer = reinterpret_cast<JOCTET*>(((*cinfo->mem->alloc_small)(
      reinterpret_cast<j_common_ptr>(cinfo), JPOOL_IMAGE, OUTPUT_BUF_SIZE * sizeof(JOCTET))));

  dest->pub.next_output_byte = dest->buffer;
  dest->pub.free_in_buffer = OUTPUT_BUF_SIZE;
}

/*
 * Empty the output buffer --- called whenever buffer fills up.
 *
 * In typical applications, this should write the entire output buffer
 * (ignoring the current state of next_output_byte & free_in_buffer),
 * reset the pointer & count to the start of the buffer, and return TRUE
 * indicating that the buffer has been dumped.
 *
 * In applications that need to be able to suspend compression due to output
 * overrun, a FALSE return indicates that the buffer cannot be emptied now.
 * In this situation, the compressor will return to its caller (possibly with
 * an indication that it has not accepted all the supplied scanlines).  The
 * application should resume compression after it has made more room in the
 * output buffer.  Note that there are substantial restrictions on the use of
 * suspension --- see the documentation.
 *
 * When suspending, the compressor will back up to a convenient restart point
 * (typically the start of the current MCU). next_output_byte & free_in_buffer
 * indicate where the restart point will be if the current call returns FALSE.
 * Data beyond this point will be regenerated after resumption, so do not
 * write it out when emptying the buffer externally.
 */

METHODDEF(boolean)
empty_output_buffer(j_compress_ptr cinfo)
{
  auto dest = (mrpt_dest_ptr)cinfo->dest;

  dest->out->Write(dest->buffer, OUTPUT_BUF_SIZE);

  dest->pub.next_output_byte = dest->buffer;
  dest->pub.free_in_buffer = OUTPUT_BUF_SIZE;

  return TRUE;
}

/*
 * Terminate destination --- called by jpeg_finish_compress
 * after all data has been written.  Usually needs to flush buffer.
 *
 * NB: *not* called by jpeg_abort or jpeg_destroy; surrounding
 * application must deal with any cleanup that should happen even
 * for error exit.
 */

METHODDEF(void)
term_destination(j_compress_ptr cinfo)
{
  auto dest = (mrpt_dest_ptr)cinfo->dest;
  size_t datacount = OUTPUT_BUF_SIZE - dest->pub.free_in_buffer;

  /* Write any data remaining in the buffer */
  if (datacount > 0) dest->out->Write(dest->buffer, (int)datacount);
}

GLOBAL(void)
jpeg_stdio_dest(j_compress_ptr cinfo, CStream* out)
{
  mrpt_dest_ptr dest;

  /* The destination object is made permanent so that multiple JPEG images
   * can be written to the same file without re-executing jpeg_stdio_dest.
   * This makes it dangerous to use this manager and a different destination
   * manager serially with the same JPEG object, because their private object
   * sizes may be different.  Caveat programmer.
   */
  if (cinfo->dest == nullptr)
  { /* first time for this JPEG object? */
    cinfo->dest = (jpeg_destination_mgr*)(*cinfo->mem->alloc_small)(
        (j_common_ptr)cinfo, JPOOL_PERMANENT, sizeof(mrpt_destination_mgr));
  }

  dest = (mrpt_dest_ptr)cinfo->dest;
  dest->pub.init_destination = init_destination;
  dest->pub.empty_output_buffer = empty_output_buffer;
  dest->pub.term_destination = term_destination;
  dest->out = out;
}

// -------------------------------------------------------------

/* Expanded data source object for stdio input */

typedef struct
{
  mrpt_jpeg_source_mgr pub; /* public fields */
  CStream* in;              /* source stream */
  JOCTET* buffer;           /* start of buffer */
  boolean start_of_file;    /* have we gotten any data yet? */
} my_source_mgr;

using my_src_ptr = my_source_mgr*;

#define INPUT_BUF_SIZE 4096 /* choose an efficiently fread'able size */

/*
 * Initialize source --- called by jpeg_read_header
 * before any data is actually read.
 */

METHODDEF(void)
init_source(j_decompress_ptr cinfo)
{
  auto src = (my_src_ptr)cinfo->src;

  /* We reset the empty-input-file flag for each image,
   * but we don't clear the input buffer.
   * This is correct behavior for reading a series of images from one source.
   */
  src->start_of_file = TRUE;
}

/*
 * Fill the input buffer --- called whenever buffer is emptied.
 *
 * In typical applications, this should read fresh data into the buffer
 * (ignoring the current state of next_input_byte & bytes_in_buffer),
 * reset the pointer & count to the start of the buffer, and return TRUE
 * indicating that the buffer has been reloaded.  It is not necessary to
 * fill the buffer entirely, only to obtain at least one more byte.
 *
 * There is no such thing as an EOF return.  If the end of the file has been
 * reached, the routine has a choice of ERREXIT() or inserting fake data into
 * the buffer.  In most cases, generating a warning message and inserting a
 * fake EOI marker is the best course of action --- this will allow the
 * decompressor to output however much of the image is there.  However,
 * the resulting error message is misleading if the real problem is an empty
 * input file, so we handle that case specially.
 *
 * In applications that need to be able to suspend compression due to input
 * not being available yet, a FALSE return indicates that no more data can be
 * obtained right now, but more may be forthcoming later.  In this situation,
 * the decompressor will return to its caller (with an indication of the
 * number of scanlines it has read, if any).  The application should resume
 * decompression after it has loaded more data into the input buffer.  Note
 * that there are substantial restrictions on the use of suspension --- see
 * the documentation.
 *
 * When suspending, the decompressor will back up to a convenient restart point
 * (typically the start of the current MCU). next_input_byte & bytes_in_buffer
 * indicate where the restart point will be if the current call returns FALSE.
 * Data beyond this point must be rescanned after resumption, so move it to
 * the front of the buffer rather than discarding it.
 */

METHODDEF(boolean)
fill_input_buffer(j_decompress_ptr cinfo)
{
  auto src = (my_src_ptr)cinfo->src;
  size_t nbytes;

  nbytes = src->in->Read(src->buffer, INPUT_BUF_SIZE);

  if (nbytes <= 0)
  {
    if (src->start_of_file) /* Treat empty input file as fatal error */
    {
      THROW_EXCEPTION("Error looking for JPEG start data!");
    }

    /* Insert a fake EOI marker */
    src->buffer[0] = (JOCTET)0xFF;
    src->buffer[1] = (JOCTET)JPEG_EOI;
    nbytes = 2;
  }

  src->pub.next_input_byte = src->buffer;
  src->pub.bytes_in_buffer = nbytes;
  src->start_of_file = FALSE;

  return TRUE;
}

/*
 * Skip data --- used to skip over a potentially large amount of
 * uninteresting data (such as an APPn marker).
 *
 * Writers of suspendable-input applications must note that skip_input_data
 * is not granted the right to give a suspension return.  If the skip extends
 * beyond the data currently in the buffer, the buffer can be marked empty so
 * that the next read will cause a fill_input_buffer call that can suspend.
 * Arranging for additional bytes to be discarded before reloading the input
 * buffer is the application writer's problem.
 */

METHODDEF(void)
skip_input_data(j_decompress_ptr cinfo, long num_bytes)
{
  auto src = (my_src_ptr)cinfo->src;

  /* Just a dumb implementation for now.  Could use fseek() except
   * it doesn't work on pipes.  Not clear that being smart is worth
   * any trouble anyway --- large skips are infrequent.
   */
  if (num_bytes > 0)
  {
    while (num_bytes > (long)src->pub.bytes_in_buffer)
    {
      num_bytes -= (long)src->pub.bytes_in_buffer;
      (void)fill_input_buffer(cinfo);
      /* note we assume that fill_input_buffer will never return FALSE,
       * so suspension need not be handled.
       */
    }
    src->pub.next_input_byte += (size_t)num_bytes;
    src->pub.bytes_in_buffer -= (size_t)num_bytes;
  }
}

/*
 * An additional method that can be provided by data source modules is the
 * resync_to_restart method for error recovery in the presence of RST markers.
 * For the moment, this source module just uses the default resync method
 * provided by the JPEG library.  That method assumes that no backtracking
 * is possible.
 */

/*
 * Terminate source --- called by jpeg_finish_decompress
 * after all data has been read.  Often a no-op.
 *
 * NB: *not* called by jpeg_abort or jpeg_destroy; surrounding
 * application must deal with any cleanup that should happen even
 * for error exit.
 */

METHODDEF(void)
term_source([[maybe_unused]] j_decompress_ptr cinfo)
{ /* no work necessary here */
}

/*
 * Prepare for input from a stdio stream.
 * The caller must have already opened the stream, and is responsible
 * for closing it after finishing decompression.
 */

GLOBAL(void)
jpeg_stdio_src(j_decompress_ptr cinfo, CStream* in)
{
  my_src_ptr src;

  /* The source object and input buffer are made permanent so that a series
   * of JPEG images can be read from the same file by calling jpeg_stdio_src
   * only before the first one.  (If we discarded the buffer at the end of
   * one image, we'd likely lose the start of the next one.)
   * This makes it unsafe to use this manager and a different source
   * manager serially with the same JPEG object.  Caveat programmer.
   */
  if (cinfo->src == nullptr)
  { /* first time for this JPEG object? */
    cinfo->src = (mrpt_jpeg_source_mgr*)(*cinfo->mem->alloc_small)(
        (j_common_ptr)cinfo, JPOOL_PERMANENT, sizeof(my_source_mgr));
    src = (my_src_ptr)cinfo->src;
    src->buffer = (JOCTET*)(*cinfo->mem->alloc_small)(
        (j_common_ptr)cinfo, JPOOL_PERMANENT, INPUT_BUF_SIZE * sizeof(JOCTET));
  }

  src = (my_src_ptr)cinfo->src;
  src->pub.init_source = init_source;
  src->pub.fill_input_buffer = fill_input_buffer;
  src->pub.skip_input_data = skip_input_data;
  src->pub.resync_to_restart = jpeg_resync_to_restart; /* use default method */
  src->pub.term_source = term_source;
  src->in = in;
  src->pub.bytes_in_buffer = 0;       /* forces fill_input_buffer on first read */
  src->pub.next_input_byte = nullptr; /* until buffer loaded */
}

// ---------------------------------------------------------------------------------------
//							END OF JPEG FUNCTIONS PART
// ---------------------------------------------------------------------------------------
#endif  // MRPT_HAS_JPEG

void CImage::saveToStreamAsJPEG(CStream& out, const int jpeg_quality) const
{
#if MRPT_HAS_OPENCV && MRPT_HAS_JPEG
  MRPT_START

  makeSureImageIsLoaded();  // For delayed loaded images stored externally

  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;

  const auto& img = m_impl->img;

  const unsigned int nCols = img.cols, nRows = img.rows;
  const bool is_color = (img.channels() == 3);

  // Some previous verification:
  ASSERT_(nCols >= 1 && nRows >= 1);
  ASSERT_(img.channels() == 1 || img.channels() == 3);

  // 1) Initialization of the JPEG compresion object:
  // --------------------------------------------------
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&cinfo);

  // 2) Set the destination of jpeg data:
  // --------------------------------------------------
  jpeg_stdio_dest(&cinfo, &out);

  // 3) Set parameters for compression:
  // --------------------------------------------------
  cinfo.image_width = nCols;
  cinfo.image_height = nRows;
  cinfo.input_components = is_color ? 3 : 1;
  cinfo.in_color_space = is_color ? JCS_RGB : JCS_GRAYSCALE;

  jpeg_set_defaults(&cinfo);
  /* Make optional parameter settings here */
  /* Now you can set any non-default parameters you wish to.
   * Here we just illustrate the use of quality (quantization table) scaling:
   */
  jpeg_set_quality(
      &cinfo, jpeg_quality /* quality per cent */, TRUE /* limit to baseline-JPEG values */);

  // 4) Start:
  // --------------------------------------------------
  jpeg_start_compress(&cinfo, TRUE);

  // 5) Write scan lines:
  // --------------------------------------------------
  if (is_color)
  {
    JSAMPROW row_pointer[1]; /* pointer to a single row */
    row_pointer[0] = new uint8_t[img.step[0]];

    for (unsigned int row = 0; row < nRows; row++)
    {
      // Flip RGB bytes order!
      const uint8_t* src = img.ptr<uint8_t>(row);
      uint8_t* target = row_pointer[0];
      for (unsigned int col = 0; col < nCols; col++)
      {
        target[0] = src[2];
        target[1] = src[1];
        target[2] = src[0];

        target += 3;
        src += 3;
      }

      if (1 != jpeg_write_scanlines(&cinfo, row_pointer, 1))
      {
        THROW_EXCEPTION("jpeg_write_scanlines: didn't work!!");
      }
    }

    delete[] row_pointer[0];
  }  // end "color"
  else
  {                          // Is grayscale:
    JSAMPROW row_pointer[1]; /* pointer to a single row */

    for (unsigned int row = 0; row < nRows; row++)
    {
      row_pointer[0] = const_cast<JSAMPROW>(img.ptr<uint8_t>(row));

      // Gray scale:
      if (1 != jpeg_write_scanlines(&cinfo, row_pointer, 1))
      {
        THROW_EXCEPTION("jpeg_write_scanlines: didn't work!!");
      }
    }
  }

  // 6) Compress and finish:
  // --------------------------------------------------
  jpeg_finish_compress(&cinfo);
  jpeg_destroy_compress(&cinfo);

  // DONE!
  MRPT_END
#endif
}

void CImage::loadFromStreamAsJPEG(CStream& in)
{
#if MRPT_HAS_OPENCV && MRPT_HAS_JPEG
  MRPT_START

  // This could have been ported to cv::imdecode(). But on a second thought,
  // it may be not be as easy: imdecode() assumes an input buffer of known
  // size while here we do not know the size to read from the CStream in
  // advance.

  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;

  /* Step 1: allocate and initialize JPEG decompression object */

  /* We set up the normal JPEG error routines, then override error_exit. */
  cinfo.err = jpeg_std_error(&jerr);

  /* Now we can initialize the JPEG decompression object. */
  jpeg_create_decompress(&cinfo);

  /* Step 2: specify data source (eg, a file) */
  jpeg_stdio_src(&cinfo, &in);

  /* Step 3: read file parameters with jpeg_read_header() */
  jpeg_read_header(&cinfo, TRUE);

  /* Step 4: set parameters for decompression */

  /* Step 5: Start decompressor */
  jpeg_start_decompress(&cinfo);

  /* We may need to do some setup of our own at this point before reading
   * the data.  After jpeg_start_decompress() we have the correct scaled
   * output image dimensions available, as well as the output colormap
   * if we asked for color quantization.
   * In this example, we need to make an output work buffer of the right size.
   */
  /* JSAMPLEs per row in output buffer */
  /* physical row width in output buffer */
  const int row_stride = cinfo.output_width * cinfo.output_components;
  /* Make a one-row-high sample array that will go away when done with image
   */
  /* Output row buffer */
  JSAMPARRAY buffer = (*cinfo.mem->alloc_sarray)((j_common_ptr)&cinfo, JPOOL_IMAGE, row_stride, 1);

  // Resize the CImage now:
  this->resize(
      cinfo.output_width, cinfo.output_height, cinfo.out_color_components == 1 ? CH_GRAY : CH_RGB);
  auto& img = m_impl->img;

  /* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */

  /* Here we use the library's state variable cinfo.output_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   */
  const unsigned int nCols = cinfo.output_width, nRows = cinfo.output_height;

  for (unsigned int row = 0; row < nRows; row++)
  {
    /* jpeg_read_scanlines expects an array of pointers to scanlines.
     * Here the array is only one element long, but you could ask for
     * more than one scanline at a time if that's more convenient.
     */
    jpeg_read_scanlines(&cinfo, buffer, 1);

    /* Copy into the CImage object */
    if (isColor())
    {
      // Flip RGB bytes order!
      auto target = img.ptr<uint8_t>(row);
      const auto* src = buffer[0];
      for (unsigned int col = 0; col < nCols; col++)
      {
        target[0] = src[2];
        target[1] = src[1];
        target[2] = src[0];

        target += 3;
        src += 3;
      }
    }
    else
    {
      // Gray scale:
      std::memcpy(img.ptr<uint8_t>(row), buffer[0], row_stride);
    }
  }

  /* Step 7: Finish decompression */

  jpeg_finish_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible
   * with the stdio data source.
   */

  /* Step 8: Release JPEG decompression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_decompress(&cinfo);

  // DONE!
  MRPT_END
#endif
}
