import { select as d3_select, color as d3_color } from '../d3.mjs';
import { browser, settings, internals, constants, isBatchMode, nsSVG,
         isNodeJs, isObject, isFunc, isStr, getDocument } from '../core.mjs';
import { WebGLRenderer, WebGLRenderTarget, CanvasTexture, TextureLoader, Raycaster,
         BufferGeometry, BufferAttribute, Float32BufferAttribute,
         Vector2, Vector3, Color, Points, PointsMaterial,
         LineSegments, LineDashedMaterial, LineBasicMaterial,
         REVISION, DoubleSide, Object3D, Matrix4, Line3, Mesh, MeshBasicMaterial, MeshLambertMaterial,
         Plane, Scene, PerspectiveCamera, OrthographicCamera, ShapeUtils,
         FrontSide, Box3, InstancedMesh, MeshStandardMaterial, MeshNormalMaterial,
         MeshPhysicalMaterial, MeshPhongMaterial, MeshDepthMaterial, MeshMatcapMaterial, MeshToonMaterial,
         Group, PlaneHelper, Euler, Quaternion, BoxGeometry, CircleGeometry, SphereGeometry, Fog,
         AmbientLight, HemisphereLight, DirectionalLight } from '../three.mjs';
import { Font, OrbitControls, SVGRenderer, TextGeometry, EffectComposer, RenderPass, UnrealBloomPass } from '../three_addons.mjs';
import { prSVG, getElementRect, getAbsPosInCanvas, makeTranslate } from './BasePainter.mjs';
import { TAttMarkerHandler } from './TAttMarkerHandler.mjs';
import { getSvgLineStyle } from './TAttLineHandler.mjs';


const originalTHREE = {
   REVISION, DoubleSide, FrontSide, Object3D, Color, Vector2, Vector3, Matrix4, Line3, Raycaster,
   WebGLRenderer, WebGLRenderTarget,
   BufferGeometry, BufferAttribute, Float32BufferAttribute, Mesh, MeshBasicMaterial, MeshLambertMaterial,
   LineSegments, LineDashedMaterial, LineBasicMaterial, Points, PointsMaterial,
   Plane, Scene, PerspectiveCamera, OrthographicCamera, ShapeUtils,
   Box3, InstancedMesh, MeshStandardMaterial, MeshNormalMaterial,
   MeshPhysicalMaterial, MeshPhongMaterial, MeshDepthMaterial, MeshMatcapMaterial, MeshToonMaterial,
   Group, PlaneHelper, Euler, Quaternion, BoxGeometry, CircleGeometry, SphereGeometry, Fog,
   AmbientLight, HemisphereLight, DirectionalLight,
   CanvasTexture, TextureLoader,

   Font, OrbitControls, SVGRenderer, TextGeometry, EffectComposer, RenderPass, UnrealBloomPass
}, THREE = Object.assign({}, originalTHREE);


/** @summary Import proper three.js version
  * @desc in node.js only r162 supports WebGL1 which can be emulated with "gl" package.
  * Therefore only this version can be used for working in node.js
  * @private */
async function importThreeJs(original) {
   if (original) {
      Object.keys(THREE).forEach(key => delete THREE[key]);
      Object.assign(THREE, originalTHREE);
      return THREE;
   }

   if (!isNodeJs() || (THREE.REVISION <= 162))
      return THREE;
   return import('three').then(h1 => {
      Object.assign(THREE, h1);
      return import('three/addons');
   }).then(h2 => {
      Object.assign(THREE, h2);
      return THREE;
   });
}

let _hfont;

/** @summary Create three.js Helvetica Regular Font instance
  * @private */
function getHelveticaFont() {
   if (_hfont && _hfont instanceof THREE.Font)
      return _hfont;

// eslint-disable-next-line
const glyphs={"0":{x_min:73,x_max:715,ha:792,o:"m 394 -29 q 153 129 242 -29 q 73 479 73 272 q 152 829 73 687 q 394 989 241 989 q 634 829 545 989 q 715 479 715 684 q 635 129 715 270 q 394 -29 546 -29 m 394 89 q 546 211 489 89 q 598 479 598 322 q 548 748 598 640 q 394 871 491 871 q 241 748 298 871 q 190 479 190 637 q 239 211 190 319 q 394 89 296 89 "},"1":{x_min:215.671875,x_max:574,ha:792,o:"m 574 0 l 442 0 l 442 697 l 215 697 l 215 796 q 386 833 330 796 q 475 986 447 875 l 574 986 l 574 0 "},"2":{x_min:59,x_max:731,ha:792,o:"m 731 0 l 59 0 q 197 314 59 188 q 457 487 199 315 q 598 691 598 580 q 543 819 598 772 q 411 867 488 867 q 272 811 328 867 q 209 630 209 747 l 81 630 q 182 901 81 805 q 408 986 271 986 q 629 909 536 986 q 731 694 731 826 q 613 449 731 541 q 378 316 495 383 q 201 122 235 234 l 731 122 l 731 0 "},"3":{x_min:54,x_max:737,ha:792,o:"m 737 284 q 635 55 737 141 q 399 -25 541 -25 q 156 52 248 -25 q 54 308 54 140 l 185 308 q 245 147 185 202 q 395 96 302 96 q 539 140 484 96 q 602 280 602 190 q 510 429 602 390 q 324 454 451 454 l 324 565 q 487 584 441 565 q 565 719 565 617 q 515 835 565 791 q 395 879 466 879 q 255 824 307 879 q 203 661 203 769 l 78 661 q 166 909 78 822 q 387 992 250 992 q 603 921 513 992 q 701 723 701 844 q 669 607 701 656 q 578 524 637 558 q 696 434 655 499 q 737 284 737 369 "},"4":{x_min:48,x_max:742.453125,ha:792,o:"m 742 243 l 602 243 l 602 0 l 476 0 l 476 243 l 48 243 l 48 368 l 476 958 l 602 958 l 602 354 l 742 354 l 742 243 m 476 354 l 476 792 l 162 354 l 476 354 "},"5":{x_min:54.171875,x_max:738,ha:792,o:"m 738 314 q 626 60 738 153 q 382 -23 526 -23 q 155 47 248 -23 q 54 256 54 125 l 183 256 q 259 132 204 174 q 382 91 314 91 q 533 149 471 91 q 602 314 602 213 q 538 469 602 411 q 386 528 475 528 q 284 506 332 528 q 197 439 237 484 l 81 439 l 159 958 l 684 958 l 684 840 l 254 840 l 214 579 q 306 627 258 612 q 407 643 354 643 q 636 552 540 643 q 738 314 738 457 "},"6":{x_min:53,x_max:739,ha:792,o:"m 739 312 q 633 62 739 162 q 400 -31 534 -31 q 162 78 257 -31 q 53 439 53 206 q 178 859 53 712 q 441 986 284 986 q 643 912 559 986 q 732 713 732 833 l 601 713 q 544 830 594 786 q 426 875 494 875 q 268 793 331 875 q 193 517 193 697 q 301 597 240 570 q 427 624 362 624 q 643 540 552 624 q 739 312 739 451 m 603 298 q 540 461 603 400 q 404 516 484 516 q 268 461 323 516 q 207 300 207 401 q 269 137 207 198 q 405 83 325 83 q 541 137 486 83 q 603 298 603 197 "},"7":{x_min:58.71875,x_max:730.953125,ha:792,o:"m 730 839 q 469 448 560 641 q 335 0 378 255 l 192 0 q 328 441 235 252 q 593 830 421 630 l 58 830 l 58 958 l 730 958 l 730 839 "},"8":{x_min:55,x_max:736,ha:792,o:"m 571 527 q 694 424 652 491 q 736 280 736 358 q 648 71 736 158 q 395 -26 551 -26 q 142 69 238 -26 q 55 279 55 157 q 96 425 55 359 q 220 527 138 491 q 120 615 153 562 q 88 726 88 668 q 171 904 88 827 q 395 986 261 986 q 618 905 529 986 q 702 727 702 830 q 670 616 702 667 q 571 527 638 565 m 394 565 q 519 610 475 565 q 563 717 563 655 q 521 823 563 781 q 392 872 474 872 q 265 824 312 872 q 224 720 224 783 q 265 613 224 656 q 394 565 312 565 m 395 91 q 545 150 488 91 q 597 280 597 204 q 546 408 597 355 q 395 465 492 465 q 244 408 299 465 q 194 280 194 356 q 244 150 194 203 q 395 91 299 91 "},"9":{x_min:53,x_max:739,ha:792,o:"m 739 524 q 619 94 739 241 q 362 -32 516 -32 q 150 47 242 -32 q 59 244 59 126 l 191 244 q 246 129 191 176 q 373 82 301 82 q 526 161 466 82 q 597 440 597 255 q 363 334 501 334 q 130 432 216 334 q 53 650 53 521 q 134 880 53 786 q 383 986 226 986 q 659 841 566 986 q 739 524 739 719 m 388 449 q 535 514 480 449 q 585 658 585 573 q 535 805 585 744 q 388 873 480 873 q 242 809 294 873 q 191 658 191 745 q 239 514 191 572 q 388 449 292 449 "},"ο":{x_min:0,x_max:712,ha:815,o:"m 356 -25 q 96 88 192 -25 q 0 368 0 201 q 92 642 0 533 q 356 761 192 761 q 617 644 517 761 q 712 368 712 533 q 619 91 712 201 q 356 -25 520 -25 m 356 85 q 527 175 465 85 q 583 369 583 255 q 528 562 583 484 q 356 651 466 651 q 189 560 250 651 q 135 369 135 481 q 187 177 135 257 q 356 85 250 85 "},S:{x_min:0,x_max:788,ha:890,o:"m 788 291 q 662 54 788 144 q 397 -26 550 -26 q 116 68 226 -26 q 0 337 0 168 l 131 337 q 200 152 131 220 q 384 85 269 85 q 557 129 479 85 q 650 270 650 183 q 490 429 650 379 q 194 513 341 470 q 33 739 33 584 q 142 964 33 881 q 388 1041 242 1041 q 644 957 543 1041 q 756 716 756 867 l 625 716 q 561 874 625 816 q 395 933 497 933 q 243 891 309 933 q 164 759 164 841 q 325 609 164 656 q 625 526 475 568 q 788 291 788 454 "},"¦":{x_min:343,x_max:449,ha:792,o:"m 449 462 l 343 462 l 343 986 l 449 986 l 449 462 m 449 -242 l 343 -242 l 343 280 l 449 280 l 449 -242 "},"/":{x_min:183.25,x_max:608.328125,ha:792,o:"m 608 1041 l 266 -129 l 183 -129 l 520 1041 l 608 1041 "},"Τ":{x_min:-0.4375,x_max:777.453125,ha:839,o:"m 777 893 l 458 893 l 458 0 l 319 0 l 319 892 l 0 892 l 0 1013 l 777 1013 l 777 893 "},y:{x_min:0,x_max:684.78125,ha:771,o:"m 684 738 l 388 -83 q 311 -216 356 -167 q 173 -279 252 -279 q 97 -266 133 -279 l 97 -149 q 132 -155 109 -151 q 168 -160 155 -160 q 240 -114 213 -160 q 274 -26 248 -98 l 0 738 l 137 737 l 341 139 l 548 737 l 684 738 "},"Π":{x_min:0,x_max:803,ha:917,o:"m 803 0 l 667 0 l 667 886 l 140 886 l 140 0 l 0 0 l 0 1012 l 803 1012 l 803 0 "},"ΐ":{x_min:-111,x_max:339,ha:361,o:"m 339 800 l 229 800 l 229 925 l 339 925 l 339 800 m -1 800 l -111 800 l -111 925 l -1 925 l -1 800 m 284 3 q 233 -10 258 -5 q 182 -15 207 -15 q 85 26 119 -15 q 42 200 42 79 l 42 737 l 167 737 l 168 215 q 172 141 168 157 q 226 101 183 101 q 248 103 239 101 q 284 112 257 104 l 284 3 m 302 1040 l 113 819 l 30 819 l 165 1040 l 302 1040 "},g:{x_min:0,x_max:686,ha:838,o:"m 686 34 q 586 -213 686 -121 q 331 -306 487 -306 q 131 -252 216 -306 q 31 -84 31 -190 l 155 -84 q 228 -174 166 -138 q 345 -207 284 -207 q 514 -109 454 -207 q 564 89 564 -27 q 461 6 521 36 q 335 -23 401 -23 q 88 100 184 -23 q 0 370 0 215 q 87 634 0 522 q 330 758 183 758 q 457 728 398 758 q 564 644 515 699 l 564 737 l 686 737 l 686 34 m 582 367 q 529 560 582 481 q 358 652 468 652 q 189 561 250 652 q 135 369 135 482 q 189 176 135 255 q 361 85 251 85 q 529 176 468 85 q 582 367 582 255 "},"²":{x_min:0,x_max:442,ha:539,o:"m 442 383 l 0 383 q 91 566 0 492 q 260 668 176 617 q 354 798 354 727 q 315 875 354 845 q 227 905 277 905 q 136 869 173 905 q 99 761 99 833 l 14 761 q 82 922 14 864 q 232 974 141 974 q 379 926 316 974 q 442 797 442 878 q 351 635 442 704 q 183 539 321 611 q 92 455 92 491 l 442 455 l 442 383 "},"–":{x_min:0,x_max:705.5625,ha:803,o:"m 705 334 l 0 334 l 0 410 l 705 410 l 705 334 "},"Κ":{x_min:0,x_max:819.5625,ha:893,o:"m 819 0 l 650 0 l 294 509 l 139 356 l 139 0 l 0 0 l 0 1013 l 139 1013 l 139 526 l 626 1013 l 809 1013 l 395 600 l 819 0 "},"ƒ":{x_min:-46.265625,x_max:392,ha:513,o:"m 392 651 l 259 651 l 79 -279 l -46 -278 l 134 651 l 14 651 l 14 751 l 135 751 q 151 948 135 900 q 304 1041 185 1041 q 334 1040 319 1041 q 392 1034 348 1039 l 392 922 q 337 931 360 931 q 271 883 287 931 q 260 793 260 853 l 260 751 l 392 751 l 392 651 "},e:{x_min:0,x_max:714,ha:813,o:"m 714 326 l 140 326 q 200 157 140 227 q 359 87 260 87 q 488 130 431 87 q 561 245 545 174 l 697 245 q 577 48 670 123 q 358 -26 484 -26 q 97 85 195 -26 q 0 363 0 197 q 94 642 0 529 q 358 765 195 765 q 626 627 529 765 q 714 326 714 503 m 576 429 q 507 583 564 522 q 355 650 445 650 q 206 583 266 650 q 140 429 152 522 l 576 429 "},"ό":{x_min:0,x_max:712,ha:815,o:"m 356 -25 q 94 91 194 -25 q 0 368 0 202 q 92 642 0 533 q 356 761 192 761 q 617 644 517 761 q 712 368 712 533 q 619 91 712 201 q 356 -25 520 -25 m 356 85 q 527 175 465 85 q 583 369 583 255 q 528 562 583 484 q 356 651 466 651 q 189 560 250 651 q 135 369 135 481 q 187 177 135 257 q 356 85 250 85 m 576 1040 l 387 819 l 303 819 l 438 1040 l 576 1040 "},J:{x_min:0,x_max:588,ha:699,o:"m 588 279 q 287 -26 588 -26 q 58 73 126 -26 q 0 327 0 158 l 133 327 q 160 172 133 227 q 288 96 198 96 q 426 171 391 96 q 449 336 449 219 l 449 1013 l 588 1013 l 588 279 "},"»":{x_min:-1,x_max:503,ha:601,o:"m 503 302 l 280 136 l 281 256 l 429 373 l 281 486 l 280 608 l 503 440 l 503 302 m 221 302 l 0 136 l 0 255 l 145 372 l 0 486 l -1 608 l 221 440 l 221 302 "},"©":{x_min:-3,x_max:1008,ha:1106,o:"m 502 -7 q 123 151 263 -7 q -3 501 -3 294 q 123 851 -3 706 q 502 1011 263 1011 q 881 851 739 1011 q 1008 501 1008 708 q 883 151 1008 292 q 502 -7 744 -7 m 502 60 q 830 197 709 60 q 940 501 940 322 q 831 805 940 681 q 502 944 709 944 q 174 805 296 944 q 65 501 65 680 q 173 197 65 320 q 502 60 294 60 m 741 394 q 661 246 731 302 q 496 190 591 190 q 294 285 369 190 q 228 497 228 370 q 295 714 228 625 q 499 813 370 813 q 656 762 588 813 q 733 625 724 711 l 634 625 q 589 704 629 673 q 498 735 550 735 q 377 666 421 735 q 334 504 334 597 q 374 340 334 408 q 490 272 415 272 q 589 304 549 272 q 638 394 628 337 l 741 394 "},"ώ":{x_min:0,x_max:922,ha:1030,o:"m 687 1040 l 498 819 l 415 819 l 549 1040 l 687 1040 m 922 339 q 856 97 922 203 q 650 -26 780 -26 q 538 9 587 -26 q 461 103 489 44 q 387 12 436 46 q 277 -22 339 -22 q 69 97 147 -22 q 0 338 0 202 q 45 551 0 444 q 161 737 84 643 l 302 737 q 175 552 219 647 q 124 336 124 446 q 155 179 124 248 q 275 88 197 88 q 375 163 341 88 q 400 294 400 219 l 400 572 l 524 572 l 524 294 q 561 135 524 192 q 643 88 591 88 q 762 182 719 88 q 797 341 797 257 q 745 555 797 450 q 619 737 705 637 l 760 737 q 874 551 835 640 q 922 339 922 444 "},"^":{x_min:193.0625,x_max:598.609375,ha:792,o:"m 598 772 l 515 772 l 395 931 l 277 772 l 193 772 l 326 1013 l 462 1013 l 598 772 "},"«":{x_min:0,x_max:507.203125,ha:604,o:"m 506 136 l 284 302 l 284 440 l 506 608 l 507 485 l 360 371 l 506 255 l 506 136 m 222 136 l 0 302 l 0 440 l 222 608 l 221 486 l 73 373 l 222 256 l 222 136 "},D:{x_min:0,x_max:828,ha:935,o:"m 389 1013 q 714 867 593 1013 q 828 521 828 729 q 712 161 828 309 q 382 0 587 0 l 0 0 l 0 1013 l 389 1013 m 376 124 q 607 247 523 124 q 681 510 681 355 q 607 771 681 662 q 376 896 522 896 l 139 896 l 139 124 l 376 124 "},"∙":{x_min:0,x_max:142,ha:239,o:"m 142 585 l 0 585 l 0 738 l 142 738 l 142 585 "},"ÿ":{x_min:0,x_max:47,ha:125,o:"m 47 3 q 37 -7 47 -7 q 28 0 30 -7 q 39 -4 32 -4 q 45 3 45 -1 l 37 0 q 28 9 28 0 q 39 19 28 19 l 47 16 l 47 19 l 47 3 m 37 1 q 44 8 44 1 q 37 16 44 16 q 30 8 30 16 q 37 1 30 1 m 26 1 l 23 22 l 14 0 l 3 22 l 3 3 l 0 25 l 13 1 l 22 25 l 26 1 "},w:{x_min:0,x_max:1009.71875,ha:1100,o:"m 1009 738 l 783 0 l 658 0 l 501 567 l 345 0 l 222 0 l 0 738 l 130 738 l 284 174 l 432 737 l 576 738 l 721 173 l 881 737 l 1009 738 "},$:{x_min:0,x_max:700,ha:793,o:"m 664 717 l 542 717 q 490 825 531 785 q 381 872 450 865 l 381 551 q 620 446 540 522 q 700 241 700 370 q 618 45 700 116 q 381 -25 536 -25 l 381 -152 l 307 -152 l 307 -25 q 81 62 162 -25 q 0 297 0 149 l 124 297 q 169 146 124 204 q 307 81 215 89 l 307 441 q 80 536 148 469 q 13 725 13 603 q 96 910 13 839 q 307 982 180 982 l 307 1077 l 381 1077 l 381 982 q 574 917 494 982 q 664 717 664 845 m 307 565 l 307 872 q 187 831 233 872 q 142 724 142 791 q 180 618 142 656 q 307 565 218 580 m 381 76 q 562 237 562 96 q 517 361 562 313 q 381 423 472 409 l 381 76 "},"\\":{x_min:-0.015625,x_max:425.0625,ha:522,o:"m 425 -129 l 337 -129 l 0 1041 l 83 1041 l 425 -129 "},"µ":{x_min:0,x_max:697.21875,ha:747,o:"m 697 -4 q 629 -14 658 -14 q 498 97 513 -14 q 422 9 470 41 q 313 -23 374 -23 q 207 4 258 -23 q 119 81 156 32 l 119 -278 l 0 -278 l 0 738 l 124 738 l 124 343 q 165 173 124 246 q 308 83 216 83 q 452 178 402 83 q 493 359 493 255 l 493 738 l 617 738 l 617 214 q 623 136 617 160 q 673 92 637 92 q 697 96 684 92 l 697 -4 "},"Ι":{x_min:42,x_max:181,ha:297,o:"m 181 0 l 42 0 l 42 1013 l 181 1013 l 181 0 "},"Ύ":{x_min:0,x_max:1144.5,ha:1214,o:"m 1144 1012 l 807 416 l 807 0 l 667 0 l 667 416 l 325 1012 l 465 1012 l 736 533 l 1004 1012 l 1144 1012 m 277 1040 l 83 799 l 0 799 l 140 1040 l 277 1040 "},"’":{x_min:0,x_max:139,ha:236,o:"m 139 851 q 102 737 139 784 q 0 669 65 690 l 0 734 q 59 787 42 741 q 72 873 72 821 l 0 873 l 0 1013 l 139 1013 l 139 851 "},"Ν":{x_min:0,x_max:801,ha:915,o:"m 801 0 l 651 0 l 131 822 l 131 0 l 0 0 l 0 1013 l 151 1013 l 670 191 l 670 1013 l 801 1013 l 801 0 "},"-":{x_min:8.71875,x_max:350.390625,ha:478,o:"m 350 317 l 8 317 l 8 428 l 350 428 l 350 317 "},Q:{x_min:0,x_max:968,ha:1072,o:"m 954 5 l 887 -79 l 744 35 q 622 -11 687 2 q 483 -26 556 -26 q 127 130 262 -26 q 0 504 0 279 q 127 880 0 728 q 484 1041 262 1041 q 841 884 708 1041 q 968 507 968 735 q 933 293 968 398 q 832 104 899 188 l 954 5 m 723 191 q 802 330 777 248 q 828 499 828 412 q 744 790 828 673 q 483 922 650 922 q 228 791 322 922 q 142 505 142 673 q 227 221 142 337 q 487 91 323 91 q 632 123 566 91 l 520 215 l 587 301 l 723 191 "},"ς":{x_min:1,x_max:676.28125,ha:740,o:"m 676 460 l 551 460 q 498 595 542 546 q 365 651 448 651 q 199 578 263 651 q 136 401 136 505 q 266 178 136 241 q 508 106 387 142 q 640 -50 640 62 q 625 -158 640 -105 q 583 -278 611 -211 l 465 -278 q 498 -182 490 -211 q 515 -80 515 -126 q 381 12 515 -15 q 134 91 197 51 q 1 388 1 179 q 100 651 1 542 q 354 761 199 761 q 587 680 498 761 q 676 460 676 599 "},M:{x_min:0,x_max:954,ha:1067,o:"m 954 0 l 819 0 l 819 869 l 537 0 l 405 0 l 128 866 l 128 0 l 0 0 l 0 1013 l 200 1013 l 472 160 l 757 1013 l 954 1013 l 954 0 "},"Ψ":{x_min:0,x_max:1006,ha:1094,o:"m 1006 678 q 914 319 1006 429 q 571 200 814 200 l 571 0 l 433 0 l 433 200 q 92 319 194 200 q 0 678 0 429 l 0 1013 l 139 1013 l 139 679 q 191 417 139 492 q 433 326 255 326 l 433 1013 l 571 1013 l 571 326 l 580 326 q 813 423 747 326 q 868 679 868 502 l 868 1013 l 1006 1013 l 1006 678 "},C:{x_min:0,x_max:886,ha:944,o:"m 886 379 q 760 87 886 201 q 455 -26 634 -26 q 112 136 236 -26 q 0 509 0 283 q 118 882 0 737 q 469 1041 245 1041 q 748 955 630 1041 q 879 708 879 859 l 745 708 q 649 862 724 805 q 473 920 573 920 q 219 791 312 920 q 136 509 136 675 q 217 229 136 344 q 470 99 311 99 q 672 179 591 99 q 753 379 753 259 l 886 379 "},"!":{x_min:0,x_max:138,ha:236,o:"m 138 684 q 116 409 138 629 q 105 244 105 299 l 33 244 q 16 465 33 313 q 0 684 0 616 l 0 1013 l 138 1013 l 138 684 m 138 0 l 0 0 l 0 151 l 138 151 l 138 0 "},"{":{x_min:0,x_max:480.5625,ha:578,o:"m 480 -286 q 237 -213 303 -286 q 187 -45 187 -159 q 194 48 187 -15 q 201 141 201 112 q 164 264 201 225 q 0 314 118 314 l 0 417 q 164 471 119 417 q 201 605 201 514 q 199 665 201 644 q 193 772 193 769 q 241 941 193 887 q 480 1015 308 1015 l 480 915 q 336 866 375 915 q 306 742 306 828 q 310 662 306 717 q 314 577 314 606 q 288 452 314 500 q 176 365 256 391 q 289 275 257 337 q 314 143 314 226 q 313 84 314 107 q 310 -11 310 -5 q 339 -131 310 -94 q 480 -182 377 -182 l 480 -286 "},X:{x_min:-0.015625,x_max:854.15625,ha:940,o:"m 854 0 l 683 0 l 423 409 l 166 0 l 0 0 l 347 519 l 18 1013 l 186 1013 l 428 637 l 675 1013 l 836 1013 l 504 520 l 854 0 "},"#":{x_min:0,x_max:963.890625,ha:1061,o:"m 963 690 l 927 590 l 719 590 l 655 410 l 876 410 l 840 310 l 618 310 l 508 -3 l 393 -2 l 506 309 l 329 310 l 215 -2 l 102 -3 l 212 310 l 0 310 l 36 410 l 248 409 l 312 590 l 86 590 l 120 690 l 347 690 l 459 1006 l 573 1006 l 462 690 l 640 690 l 751 1006 l 865 1006 l 754 690 l 963 690 m 606 590 l 425 590 l 362 410 l 543 410 l 606 590 "},"ι":{x_min:42,x_max:284,ha:361,o:"m 284 3 q 233 -10 258 -5 q 182 -15 207 -15 q 85 26 119 -15 q 42 200 42 79 l 42 738 l 167 738 l 168 215 q 172 141 168 157 q 226 101 183 101 q 248 103 239 101 q 284 112 257 104 l 284 3 "},"Ά":{x_min:0,x_max:906.953125,ha:982,o:"m 283 1040 l 88 799 l 5 799 l 145 1040 l 283 1040 m 906 0 l 756 0 l 650 303 l 251 303 l 143 0 l 0 0 l 376 1012 l 529 1012 l 906 0 m 609 421 l 452 866 l 293 421 l 609 421 "},")":{x_min:0,x_max:318,ha:415,o:"m 318 365 q 257 25 318 191 q 87 -290 197 -141 l 0 -290 q 140 21 93 -128 q 193 360 193 189 q 141 704 193 537 q 0 1024 97 850 l 87 1024 q 257 706 197 871 q 318 365 318 542 "},"ε":{x_min:0,x_max:634.71875,ha:714,o:"m 634 234 q 527 38 634 110 q 300 -25 433 -25 q 98 29 183 -25 q 0 204 0 93 q 37 314 0 265 q 128 390 67 353 q 56 460 82 419 q 26 555 26 505 q 114 712 26 654 q 295 763 191 763 q 499 700 416 763 q 589 515 589 631 l 478 515 q 419 618 464 580 q 307 657 374 657 q 207 630 253 657 q 151 547 151 598 q 238 445 151 469 q 389 434 280 434 l 389 331 l 349 331 q 206 315 255 331 q 125 210 125 287 q 183 107 125 145 q 302 76 233 76 q 436 117 379 76 q 509 234 493 159 l 634 234 "},"Δ":{x_min:0,x_max:952.78125,ha:1028,o:"m 952 0 l 0 0 l 400 1013 l 551 1013 l 952 0 m 762 124 l 476 867 l 187 124 l 762 124 "},"}":{x_min:0,x_max:481,ha:578,o:"m 481 314 q 318 262 364 314 q 282 136 282 222 q 284 65 282 97 q 293 -58 293 -48 q 241 -217 293 -166 q 0 -286 174 -286 l 0 -182 q 143 -130 105 -182 q 171 -2 171 -93 q 168 81 171 22 q 165 144 165 140 q 188 275 165 229 q 306 365 220 339 q 191 455 224 391 q 165 588 165 505 q 168 681 165 624 q 171 742 171 737 q 141 865 171 827 q 0 915 102 915 l 0 1015 q 243 942 176 1015 q 293 773 293 888 q 287 675 293 741 q 282 590 282 608 q 318 466 282 505 q 481 417 364 417 l 481 314 "},"‰":{x_min:-3,x_max:1672,ha:1821,o:"m 846 0 q 664 76 732 0 q 603 244 603 145 q 662 412 603 344 q 846 489 729 489 q 1027 412 959 489 q 1089 244 1089 343 q 1029 76 1089 144 q 846 0 962 0 m 845 103 q 945 143 910 103 q 981 243 981 184 q 947 340 981 301 q 845 385 910 385 q 745 342 782 385 q 709 243 709 300 q 742 147 709 186 q 845 103 781 103 m 888 986 l 284 -25 l 199 -25 l 803 986 l 888 986 m 241 468 q 58 545 126 468 q -3 715 -3 615 q 56 881 -3 813 q 238 958 124 958 q 421 881 353 958 q 483 712 483 813 q 423 544 483 612 q 241 468 356 468 m 241 855 q 137 811 175 855 q 100 710 100 768 q 136 612 100 653 q 240 572 172 572 q 344 614 306 572 q 382 713 382 656 q 347 810 382 771 q 241 855 308 855 m 1428 0 q 1246 76 1314 0 q 1185 244 1185 145 q 1244 412 1185 344 q 1428 489 1311 489 q 1610 412 1542 489 q 1672 244 1672 343 q 1612 76 1672 144 q 1428 0 1545 0 m 1427 103 q 1528 143 1492 103 q 1564 243 1564 184 q 1530 340 1564 301 q 1427 385 1492 385 q 1327 342 1364 385 q 1291 243 1291 300 q 1324 147 1291 186 q 1427 103 1363 103 "},a:{x_min:0,x_max:698.609375,ha:794,o:"m 698 0 q 661 -12 679 -7 q 615 -17 643 -17 q 536 12 564 -17 q 500 96 508 41 q 384 6 456 37 q 236 -25 312 -25 q 65 31 130 -25 q 0 194 0 88 q 118 390 0 334 q 328 435 180 420 q 488 483 476 451 q 495 523 495 504 q 442 619 495 584 q 325 654 389 654 q 209 617 257 654 q 152 513 161 580 l 33 513 q 123 705 33 633 q 332 772 207 772 q 528 712 448 772 q 617 531 617 645 l 617 163 q 624 108 617 126 q 664 90 632 90 l 698 94 l 698 0 m 491 262 l 491 372 q 272 329 350 347 q 128 201 128 294 q 166 113 128 144 q 264 83 205 83 q 414 130 346 83 q 491 262 491 183 "},"—":{x_min:0,x_max:941.671875,ha:1039,o:"m 941 334 l 0 334 l 0 410 l 941 410 l 941 334 "},"=":{x_min:8.71875,x_max:780.953125,ha:792,o:"m 780 510 l 8 510 l 8 606 l 780 606 l 780 510 m 780 235 l 8 235 l 8 332 l 780 332 l 780 235 "},N:{x_min:0,x_max:801,ha:914,o:"m 801 0 l 651 0 l 131 823 l 131 0 l 0 0 l 0 1013 l 151 1013 l 670 193 l 670 1013 l 801 1013 l 801 0 "},"ρ":{x_min:0,x_max:712,ha:797,o:"m 712 369 q 620 94 712 207 q 362 -26 521 -26 q 230 2 292 -26 q 119 83 167 30 l 119 -278 l 0 -278 l 0 362 q 91 643 0 531 q 355 764 190 764 q 617 647 517 764 q 712 369 712 536 m 583 366 q 530 559 583 480 q 359 651 469 651 q 190 562 252 651 q 135 370 135 483 q 189 176 135 257 q 359 85 250 85 q 528 175 466 85 q 583 366 583 254 "},"¯":{x_min:0,x_max:941.671875,ha:938,o:"m 941 1033 l 0 1033 l 0 1109 l 941 1109 l 941 1033 "},Z:{x_min:0,x_max:779,ha:849,o:"m 779 0 l 0 0 l 0 113 l 621 896 l 40 896 l 40 1013 l 779 1013 l 778 887 l 171 124 l 779 124 l 779 0 "},u:{x_min:0,x_max:617,ha:729,o:"m 617 0 l 499 0 l 499 110 q 391 10 460 45 q 246 -25 322 -25 q 61 58 127 -25 q 0 258 0 136 l 0 738 l 125 738 l 125 284 q 156 148 125 202 q 273 82 197 82 q 433 165 369 82 q 493 340 493 243 l 493 738 l 617 738 l 617 0 "},k:{x_min:0,x_max:612.484375,ha:697,o:"m 612 738 l 338 465 l 608 0 l 469 0 l 251 382 l 121 251 l 121 0 l 0 0 l 0 1013 l 121 1013 l 121 402 l 456 738 l 612 738 "},"Η":{x_min:0,x_max:803,ha:917,o:"m 803 0 l 667 0 l 667 475 l 140 475 l 140 0 l 0 0 l 0 1013 l 140 1013 l 140 599 l 667 599 l 667 1013 l 803 1013 l 803 0 "},"Α":{x_min:0,x_max:906.953125,ha:985,o:"m 906 0 l 756 0 l 650 303 l 251 303 l 143 0 l 0 0 l 376 1013 l 529 1013 l 906 0 m 609 421 l 452 866 l 293 421 l 609 421 "},s:{x_min:0,x_max:604,ha:697,o:"m 604 217 q 501 36 604 104 q 292 -23 411 -23 q 86 43 166 -23 q 0 238 0 114 l 121 237 q 175 122 121 164 q 300 85 223 85 q 415 112 363 85 q 479 207 479 147 q 361 309 479 276 q 140 372 141 370 q 21 544 21 426 q 111 708 21 647 q 298 761 190 761 q 492 705 413 761 q 583 531 583 643 l 462 531 q 412 625 462 594 q 298 657 363 657 q 199 636 242 657 q 143 558 143 608 q 262 454 143 486 q 484 394 479 397 q 604 217 604 341 "},B:{x_min:0,x_max:778,ha:876,o:"m 580 546 q 724 469 670 535 q 778 311 778 403 q 673 83 778 171 q 432 0 575 0 l 0 0 l 0 1013 l 411 1013 q 629 957 541 1013 q 732 768 732 892 q 691 633 732 693 q 580 546 650 572 m 393 899 l 139 899 l 139 588 l 379 588 q 521 624 462 588 q 592 744 592 667 q 531 859 592 819 q 393 899 471 899 m 419 124 q 566 169 504 124 q 635 303 635 219 q 559 436 635 389 q 402 477 494 477 l 139 477 l 139 124 l 419 124 "},"…":{x_min:0,x_max:614,ha:708,o:"m 142 0 l 0 0 l 0 151 l 142 151 l 142 0 m 378 0 l 236 0 l 236 151 l 378 151 l 378 0 m 614 0 l 472 0 l 472 151 l 614 151 l 614 0 "},"?":{x_min:0,x_max:607,ha:704,o:"m 607 777 q 543 599 607 674 q 422 474 482 537 q 357 272 357 391 l 236 272 q 297 487 236 395 q 411 619 298 490 q 474 762 474 691 q 422 885 474 838 q 301 933 371 933 q 179 880 228 933 q 124 706 124 819 l 0 706 q 94 963 0 872 q 302 1044 177 1044 q 511 973 423 1044 q 607 777 607 895 m 370 0 l 230 0 l 230 151 l 370 151 l 370 0 "},H:{x_min:0,x_max:803,ha:915,o:"m 803 0 l 667 0 l 667 475 l 140 475 l 140 0 l 0 0 l 0 1013 l 140 1013 l 140 599 l 667 599 l 667 1013 l 803 1013 l 803 0 "},"ν":{x_min:0,x_max:675,ha:761,o:"m 675 738 l 404 0 l 272 0 l 0 738 l 133 738 l 340 147 l 541 738 l 675 738 "},c:{x_min:1,x_max:701.390625,ha:775,o:"m 701 264 q 584 53 681 133 q 353 -26 487 -26 q 91 91 188 -26 q 1 370 1 201 q 92 645 1 537 q 353 761 190 761 q 572 688 479 761 q 690 493 666 615 l 556 493 q 487 606 545 562 q 356 650 428 650 q 186 563 246 650 q 134 372 134 487 q 188 179 134 258 q 359 88 250 88 q 492 136 437 88 q 566 264 548 185 l 701 264 "},"¶":{x_min:0,x_max:566.671875,ha:678,o:"m 21 892 l 52 892 l 98 761 l 145 892 l 176 892 l 178 741 l 157 741 l 157 867 l 108 741 l 88 741 l 40 871 l 40 741 l 21 741 l 21 892 m 308 854 l 308 731 q 252 691 308 691 q 227 691 240 691 q 207 696 213 695 l 207 712 l 253 706 q 288 733 288 706 l 288 763 q 244 741 279 741 q 193 797 193 741 q 261 860 193 860 q 287 860 273 860 q 308 854 302 855 m 288 842 l 263 843 q 213 796 213 843 q 248 756 213 756 q 288 796 288 756 l 288 842 m 566 988 l 502 988 l 502 -1 l 439 -1 l 439 988 l 317 988 l 317 -1 l 252 -1 l 252 602 q 81 653 155 602 q 0 805 0 711 q 101 989 0 918 q 309 1053 194 1053 l 566 1053 l 566 988 "},"β":{x_min:0,x_max:660,ha:745,o:"m 471 550 q 610 450 561 522 q 660 280 660 378 q 578 64 660 151 q 367 -22 497 -22 q 239 5 299 -22 q 126 82 178 32 l 126 -278 l 0 -278 l 0 593 q 54 903 0 801 q 318 1042 127 1042 q 519 964 436 1042 q 603 771 603 887 q 567 644 603 701 q 471 550 532 586 m 337 79 q 476 138 418 79 q 535 279 535 198 q 427 437 535 386 q 226 477 344 477 l 226 583 q 398 620 329 583 q 486 762 486 668 q 435 884 486 833 q 312 935 384 935 q 169 861 219 935 q 126 698 126 797 l 126 362 q 170 169 126 242 q 337 79 224 79 "},"Μ":{x_min:0,x_max:954,ha:1068,o:"m 954 0 l 819 0 l 819 868 l 537 0 l 405 0 l 128 865 l 128 0 l 0 0 l 0 1013 l 199 1013 l 472 158 l 758 1013 l 954 1013 l 954 0 "},"Ό":{x_min:0.109375,x_max:1120,ha:1217,o:"m 1120 505 q 994 132 1120 282 q 642 -29 861 -29 q 290 130 422 -29 q 167 505 167 280 q 294 883 167 730 q 650 1046 430 1046 q 999 882 868 1046 q 1120 505 1120 730 m 977 504 q 896 784 977 669 q 644 915 804 915 q 391 785 484 915 q 307 504 307 669 q 391 224 307 339 q 644 95 486 95 q 894 224 803 95 q 977 504 977 339 m 277 1040 l 83 799 l 0 799 l 140 1040 l 277 1040 "},"Ή":{x_min:0,x_max:1158,ha:1275,o:"m 1158 0 l 1022 0 l 1022 475 l 496 475 l 496 0 l 356 0 l 356 1012 l 496 1012 l 496 599 l 1022 599 l 1022 1012 l 1158 1012 l 1158 0 m 277 1040 l 83 799 l 0 799 l 140 1040 l 277 1040 "},"•":{x_min:0,x_max:663.890625,ha:775,o:"m 663 529 q 566 293 663 391 q 331 196 469 196 q 97 294 194 196 q 0 529 0 393 q 96 763 0 665 q 331 861 193 861 q 566 763 469 861 q 663 529 663 665 "},"¥":{x_min:0.1875,x_max:819.546875,ha:886,o:"m 563 561 l 697 561 l 696 487 l 520 487 l 482 416 l 482 380 l 697 380 l 695 308 l 482 308 l 482 0 l 342 0 l 342 308 l 125 308 l 125 380 l 342 380 l 342 417 l 303 487 l 125 487 l 125 561 l 258 561 l 0 1013 l 140 1013 l 411 533 l 679 1013 l 819 1013 l 563 561 "},"(":{x_min:0,x_max:318.0625,ha:415,o:"m 318 -290 l 230 -290 q 61 23 122 -142 q 0 365 0 190 q 62 712 0 540 q 230 1024 119 869 l 318 1024 q 175 705 219 853 q 125 360 125 542 q 176 22 125 187 q 318 -290 223 -127 "},U:{x_min:0,x_max:796,ha:904,o:"m 796 393 q 681 93 796 212 q 386 -25 566 -25 q 101 95 208 -25 q 0 393 0 211 l 0 1013 l 138 1013 l 138 391 q 204 191 138 270 q 394 107 276 107 q 586 191 512 107 q 656 391 656 270 l 656 1013 l 796 1013 l 796 393 "},"γ":{x_min:0.5,x_max:744.953125,ha:822,o:"m 744 737 l 463 54 l 463 -278 l 338 -278 l 338 54 l 154 495 q 104 597 124 569 q 13 651 67 651 l 0 651 l 0 751 l 39 753 q 168 711 121 753 q 242 594 207 676 l 403 208 l 617 737 l 744 737 "},"α":{x_min:0,x_max:765.5625,ha:809,o:"m 765 -4 q 698 -14 726 -14 q 564 97 586 -14 q 466 7 525 40 q 337 -26 407 -26 q 88 98 186 -26 q 0 369 0 212 q 88 637 0 525 q 337 760 184 760 q 465 728 407 760 q 563 637 524 696 l 563 739 l 685 739 l 685 222 q 693 141 685 168 q 748 94 708 94 q 765 96 760 94 l 765 -4 m 584 371 q 531 562 584 485 q 360 653 470 653 q 192 566 254 653 q 135 379 135 489 q 186 181 135 261 q 358 84 247 84 q 528 176 465 84 q 584 371 584 260 "},F:{x_min:0,x_max:683.328125,ha:717,o:"m 683 888 l 140 888 l 140 583 l 613 583 l 613 458 l 140 458 l 140 0 l 0 0 l 0 1013 l 683 1013 l 683 888 "},"­":{x_min:0,x_max:705.5625,ha:803,o:"m 705 334 l 0 334 l 0 410 l 705 410 l 705 334 "},":":{x_min:0,x_max:142,ha:239,o:"m 142 585 l 0 585 l 0 738 l 142 738 l 142 585 m 142 0 l 0 0 l 0 151 l 142 151 l 142 0 "},"Χ":{x_min:0,x_max:854.171875,ha:935,o:"m 854 0 l 683 0 l 423 409 l 166 0 l 0 0 l 347 519 l 18 1013 l 186 1013 l 427 637 l 675 1013 l 836 1013 l 504 521 l 854 0 "},"*":{x_min:116,x_max:674,ha:792,o:"m 674 768 l 475 713 l 610 544 l 517 477 l 394 652 l 272 478 l 178 544 l 314 713 l 116 766 l 153 876 l 341 812 l 342 1013 l 446 1013 l 446 811 l 635 874 l 674 768 "},"†":{x_min:0,x_max:777,ha:835,o:"m 458 804 l 777 804 l 777 683 l 458 683 l 458 0 l 319 0 l 319 681 l 0 683 l 0 804 l 319 804 l 319 1015 l 458 1013 l 458 804 "},"°":{x_min:0,x_max:347,ha:444,o:"m 173 802 q 43 856 91 802 q 0 977 0 905 q 45 1101 0 1049 q 173 1153 90 1153 q 303 1098 255 1153 q 347 977 347 1049 q 303 856 347 905 q 173 802 256 802 m 173 884 q 238 910 214 884 q 262 973 262 937 q 239 1038 262 1012 q 173 1064 217 1064 q 108 1037 132 1064 q 85 973 85 1010 q 108 910 85 937 q 173 884 132 884 "},V:{x_min:0,x_max:862.71875,ha:940,o:"m 862 1013 l 505 0 l 361 0 l 0 1013 l 143 1013 l 434 165 l 718 1012 l 862 1013 "},"Ξ":{x_min:0,x_max:734.71875,ha:763,o:"m 723 889 l 9 889 l 9 1013 l 723 1013 l 723 889 m 673 463 l 61 463 l 61 589 l 673 589 l 673 463 m 734 0 l 0 0 l 0 124 l 734 124 l 734 0 "}," ":{x_min:0,x_max:0,ha:853},"Ϋ":{x_min:0.328125,x_max:819.515625,ha:889,o:"m 588 1046 l 460 1046 l 460 1189 l 588 1189 l 588 1046 m 360 1046 l 232 1046 l 232 1189 l 360 1189 l 360 1046 m 819 1012 l 482 416 l 482 0 l 342 0 l 342 416 l 0 1012 l 140 1012 l 411 533 l 679 1012 l 819 1012 "},"”":{x_min:0,x_max:347,ha:454,o:"m 139 851 q 102 737 139 784 q 0 669 65 690 l 0 734 q 59 787 42 741 q 72 873 72 821 l 0 873 l 0 1013 l 139 1013 l 139 851 m 347 851 q 310 737 347 784 q 208 669 273 690 l 208 734 q 267 787 250 741 q 280 873 280 821 l 208 873 l 208 1013 l 347 1013 l 347 851 "},"@":{x_min:0,x_max:1260,ha:1357,o:"m 1098 -45 q 877 -160 1001 -117 q 633 -203 752 -203 q 155 -29 327 -203 q 0 360 0 127 q 176 802 0 616 q 687 1008 372 1008 q 1123 854 969 1008 q 1260 517 1260 718 q 1155 216 1260 341 q 868 82 1044 82 q 772 106 801 82 q 737 202 737 135 q 647 113 700 144 q 527 82 594 82 q 367 147 420 82 q 314 312 314 212 q 401 565 314 452 q 639 690 498 690 q 810 588 760 690 l 849 668 l 938 668 q 877 441 900 532 q 833 226 833 268 q 853 182 833 198 q 902 167 873 167 q 1088 272 1012 167 q 1159 512 1159 372 q 1051 793 1159 681 q 687 925 925 925 q 248 747 415 925 q 97 361 97 586 q 226 26 97 159 q 627 -122 370 -122 q 856 -87 737 -122 q 1061 8 976 -53 l 1098 -45 m 786 488 q 738 580 777 545 q 643 615 700 615 q 483 517 548 615 q 425 322 425 430 q 457 203 425 250 q 552 156 490 156 q 722 273 665 156 q 786 488 738 309 "},"Ί":{x_min:0,x_max:499,ha:613,o:"m 277 1040 l 83 799 l 0 799 l 140 1040 l 277 1040 m 499 0 l 360 0 l 360 1012 l 499 1012 l 499 0 "},i:{x_min:14,x_max:136,ha:275,o:"m 136 873 l 14 873 l 14 1013 l 136 1013 l 136 873 m 136 0 l 14 0 l 14 737 l 136 737 l 136 0 "},"Β":{x_min:0,x_max:778,ha:877,o:"m 580 545 q 724 468 671 534 q 778 310 778 402 q 673 83 778 170 q 432 0 575 0 l 0 0 l 0 1013 l 411 1013 q 629 957 541 1013 q 732 768 732 891 q 691 632 732 692 q 580 545 650 571 m 393 899 l 139 899 l 139 587 l 379 587 q 521 623 462 587 q 592 744 592 666 q 531 859 592 819 q 393 899 471 899 m 419 124 q 566 169 504 124 q 635 302 635 219 q 559 435 635 388 q 402 476 494 476 l 139 476 l 139 124 l 419 124 "},"υ":{x_min:0,x_max:617,ha:725,o:"m 617 352 q 540 94 617 199 q 308 -24 455 -24 q 76 94 161 -24 q 0 352 0 199 l 0 739 l 126 739 l 126 355 q 169 185 126 257 q 312 98 220 98 q 451 185 402 98 q 492 355 492 257 l 492 739 l 617 739 l 617 352 "},"]":{x_min:0,x_max:275,ha:372,o:"m 275 -281 l 0 -281 l 0 -187 l 151 -187 l 151 920 l 0 920 l 0 1013 l 275 1013 l 275 -281 "},m:{x_min:0,x_max:1019,ha:1128,o:"m 1019 0 l 897 0 l 897 454 q 860 591 897 536 q 739 660 816 660 q 613 586 659 660 q 573 436 573 522 l 573 0 l 447 0 l 447 455 q 412 591 447 535 q 294 657 372 657 q 165 586 213 657 q 122 437 122 521 l 122 0 l 0 0 l 0 738 l 117 738 l 117 640 q 202 730 150 697 q 316 763 254 763 q 437 730 381 763 q 525 642 494 697 q 621 731 559 700 q 753 763 682 763 q 943 694 867 763 q 1019 512 1019 625 l 1019 0 "},"χ":{x_min:8.328125,x_max:780.5625,ha:815,o:"m 780 -278 q 715 -294 747 -294 q 616 -257 663 -294 q 548 -175 576 -227 l 379 133 l 143 -277 l 9 -277 l 313 254 l 163 522 q 127 586 131 580 q 36 640 91 640 q 8 637 27 640 l 8 752 l 52 757 q 162 719 113 757 q 236 627 200 690 l 383 372 l 594 737 l 726 737 l 448 250 l 625 -69 q 670 -153 647 -110 q 743 -188 695 -188 q 780 -184 759 -188 l 780 -278 "},"ί":{x_min:42,x_max:326.71875,ha:361,o:"m 284 3 q 233 -10 258 -5 q 182 -15 207 -15 q 85 26 119 -15 q 42 200 42 79 l 42 737 l 167 737 l 168 215 q 172 141 168 157 q 226 101 183 101 q 248 102 239 101 q 284 112 257 104 l 284 3 m 326 1040 l 137 819 l 54 819 l 189 1040 l 326 1040 "},"Ζ":{x_min:0,x_max:779.171875,ha:850,o:"m 779 0 l 0 0 l 0 113 l 620 896 l 40 896 l 40 1013 l 779 1013 l 779 887 l 170 124 l 779 124 l 779 0 "},R:{x_min:0,x_max:781.953125,ha:907,o:"m 781 0 l 623 0 q 587 242 590 52 q 407 433 585 433 l 138 433 l 138 0 l 0 0 l 0 1013 l 396 1013 q 636 946 539 1013 q 749 731 749 868 q 711 597 749 659 q 608 502 674 534 q 718 370 696 474 q 729 207 722 352 q 781 26 736 62 l 781 0 m 373 551 q 533 594 465 551 q 614 731 614 645 q 532 859 614 815 q 373 896 465 896 l 138 896 l 138 551 l 373 551 "},o:{x_min:0,x_max:713,ha:821,o:"m 357 -25 q 94 91 194 -25 q 0 368 0 202 q 93 642 0 533 q 357 761 193 761 q 618 644 518 761 q 713 368 713 533 q 619 91 713 201 q 357 -25 521 -25 m 357 85 q 528 175 465 85 q 584 369 584 255 q 529 562 584 484 q 357 651 467 651 q 189 560 250 651 q 135 369 135 481 q 187 177 135 257 q 357 85 250 85 "},K:{x_min:0,x_max:819.46875,ha:906,o:"m 819 0 l 649 0 l 294 509 l 139 355 l 139 0 l 0 0 l 0 1013 l 139 1013 l 139 526 l 626 1013 l 809 1013 l 395 600 l 819 0 "},",":{x_min:0,x_max:142,ha:239,o:"m 142 -12 q 105 -132 142 -82 q 0 -205 68 -182 l 0 -138 q 57 -82 40 -124 q 70 0 70 -51 l 0 0 l 0 151 l 142 151 l 142 -12 "},d:{x_min:0,x_max:683,ha:796,o:"m 683 0 l 564 0 l 564 93 q 456 6 516 38 q 327 -25 395 -25 q 87 100 181 -25 q 0 365 0 215 q 90 639 0 525 q 343 763 187 763 q 564 647 486 763 l 564 1013 l 683 1013 l 683 0 m 582 373 q 529 562 582 484 q 361 653 468 653 q 190 561 253 653 q 135 365 135 479 q 189 175 135 254 q 358 85 251 85 q 529 178 468 85 q 582 373 582 258 "},"¨":{x_min:-109,x_max:247,ha:232,o:"m 247 1046 l 119 1046 l 119 1189 l 247 1189 l 247 1046 m 19 1046 l -109 1046 l -109 1189 l 19 1189 l 19 1046 "},E:{x_min:0,x_max:736.109375,ha:789,o:"m 736 0 l 0 0 l 0 1013 l 725 1013 l 725 889 l 139 889 l 139 585 l 677 585 l 677 467 l 139 467 l 139 125 l 736 125 l 736 0 "},Y:{x_min:0,x_max:820,ha:886,o:"m 820 1013 l 482 416 l 482 0 l 342 0 l 342 416 l 0 1013 l 140 1013 l 411 534 l 679 1012 l 820 1013 "},"\"":{x_min:0,x_max:299,ha:396,o:"m 299 606 l 203 606 l 203 988 l 299 988 l 299 606 m 96 606 l 0 606 l 0 988 l 96 988 l 96 606 "},"‹":{x_min:17.984375,x_max:773.609375,ha:792,o:"m 773 40 l 18 376 l 17 465 l 773 799 l 773 692 l 159 420 l 773 149 l 773 40 "},"„":{x_min:0,x_max:364,ha:467,o:"m 141 -12 q 104 -132 141 -82 q 0 -205 67 -182 l 0 -138 q 56 -82 40 -124 q 69 0 69 -51 l 0 0 l 0 151 l 141 151 l 141 -12 m 364 -12 q 327 -132 364 -82 q 222 -205 290 -182 l 222 -138 q 279 -82 262 -124 q 292 0 292 -51 l 222 0 l 222 151 l 364 151 l 364 -12 "},"δ":{x_min:1,x_max:710,ha:810,o:"m 710 360 q 616 87 710 196 q 356 -28 518 -28 q 99 82 197 -28 q 1 356 1 192 q 100 606 1 509 q 355 703 199 703 q 180 829 288 754 q 70 903 124 866 l 70 1012 l 643 1012 l 643 901 l 258 901 q 462 763 422 794 q 636 592 577 677 q 710 360 710 485 m 584 365 q 552 501 584 447 q 451 602 521 555 q 372 611 411 611 q 197 541 258 611 q 136 355 136 472 q 190 171 136 245 q 358 85 252 85 q 528 173 465 85 q 584 365 584 252 "},"έ":{x_min:0,x_max:634.71875,ha:714,o:"m 634 234 q 527 38 634 110 q 300 -25 433 -25 q 98 29 183 -25 q 0 204 0 93 q 37 313 0 265 q 128 390 67 352 q 56 459 82 419 q 26 555 26 505 q 114 712 26 654 q 295 763 191 763 q 499 700 416 763 q 589 515 589 631 l 478 515 q 419 618 464 580 q 307 657 374 657 q 207 630 253 657 q 151 547 151 598 q 238 445 151 469 q 389 434 280 434 l 389 331 l 349 331 q 206 315 255 331 q 125 210 125 287 q 183 107 125 145 q 302 76 233 76 q 436 117 379 76 q 509 234 493 159 l 634 234 m 520 1040 l 331 819 l 248 819 l 383 1040 l 520 1040 "},"ω":{x_min:0,x_max:922,ha:1031,o:"m 922 339 q 856 97 922 203 q 650 -26 780 -26 q 538 9 587 -26 q 461 103 489 44 q 387 12 436 46 q 277 -22 339 -22 q 69 97 147 -22 q 0 339 0 203 q 45 551 0 444 q 161 738 84 643 l 302 738 q 175 553 219 647 q 124 336 124 446 q 155 179 124 249 q 275 88 197 88 q 375 163 341 88 q 400 294 400 219 l 400 572 l 524 572 l 524 294 q 561 135 524 192 q 643 88 591 88 q 762 182 719 88 q 797 342 797 257 q 745 556 797 450 q 619 738 705 638 l 760 738 q 874 551 835 640 q 922 339 922 444 "},"´":{x_min:0,x_max:96,ha:251,o:"m 96 606 l 0 606 l 0 988 l 96 988 l 96 606 "},"±":{x_min:11,x_max:781,ha:792,o:"m 781 490 l 446 490 l 446 255 l 349 255 l 349 490 l 11 490 l 11 586 l 349 586 l 349 819 l 446 819 l 446 586 l 781 586 l 781 490 m 781 21 l 11 21 l 11 115 l 781 115 l 781 21 "},"|":{x_min:343,x_max:449,ha:792,o:"m 449 462 l 343 462 l 343 986 l 449 986 l 449 462 m 449 -242 l 343 -242 l 343 280 l 449 280 l 449 -242 "},"ϋ":{x_min:0,x_max:617,ha:725,o:"m 482 800 l 372 800 l 372 925 l 482 925 l 482 800 m 239 800 l 129 800 l 129 925 l 239 925 l 239 800 m 617 352 q 540 93 617 199 q 308 -24 455 -24 q 76 93 161 -24 q 0 352 0 199 l 0 738 l 126 738 l 126 354 q 169 185 126 257 q 312 98 220 98 q 451 185 402 98 q 492 354 492 257 l 492 738 l 617 738 l 617 352 "},"§":{x_min:0,x_max:593,ha:690,o:"m 593 425 q 554 312 593 369 q 467 233 516 254 q 537 83 537 172 q 459 -74 537 -12 q 288 -133 387 -133 q 115 -69 184 -133 q 47 96 47 -6 l 166 96 q 199 7 166 40 q 288 -26 232 -26 q 371 -5 332 -26 q 420 60 420 21 q 311 201 420 139 q 108 309 210 255 q 0 490 0 383 q 33 602 0 551 q 124 687 66 654 q 75 743 93 712 q 58 812 58 773 q 133 984 58 920 q 300 1043 201 1043 q 458 987 394 1043 q 529 814 529 925 l 411 814 q 370 908 404 877 q 289 939 336 939 q 213 911 246 939 q 180 841 180 883 q 286 720 180 779 q 484 612 480 615 q 593 425 593 534 m 467 409 q 355 544 467 473 q 196 630 228 612 q 146 587 162 609 q 124 525 124 558 q 239 387 124 462 q 398 298 369 315 q 448 345 429 316 q 467 409 467 375 "},b:{x_min:0,x_max:685,ha:783,o:"m 685 372 q 597 99 685 213 q 347 -25 501 -25 q 219 5 277 -25 q 121 93 161 36 l 121 0 l 0 0 l 0 1013 l 121 1013 l 121 634 q 214 723 157 692 q 341 754 272 754 q 591 637 493 754 q 685 372 685 526 m 554 356 q 499 550 554 470 q 328 644 437 644 q 162 556 223 644 q 108 369 108 478 q 160 176 108 256 q 330 83 221 83 q 498 169 435 83 q 554 356 554 245 "},q:{x_min:0,x_max:683,ha:876,o:"m 683 -278 l 564 -278 l 564 97 q 474 8 533 39 q 345 -23 415 -23 q 91 93 188 -23 q 0 364 0 203 q 87 635 0 522 q 337 760 184 760 q 466 727 408 760 q 564 637 523 695 l 564 737 l 683 737 l 683 -278 m 582 375 q 527 564 582 488 q 358 652 466 652 q 190 565 253 652 q 135 377 135 488 q 189 179 135 261 q 361 84 251 84 q 530 179 469 84 q 582 375 582 260 "},"Ω":{x_min:-0.171875,x_max:969.5625,ha:1068,o:"m 969 0 l 555 0 l 555 123 q 744 308 675 194 q 814 558 814 423 q 726 812 814 709 q 484 922 633 922 q 244 820 334 922 q 154 567 154 719 q 223 316 154 433 q 412 123 292 199 l 412 0 l 0 0 l 0 124 l 217 124 q 68 327 122 210 q 15 572 15 444 q 144 911 15 781 q 484 1041 274 1041 q 822 909 691 1041 q 953 569 953 777 q 899 326 953 443 q 750 124 846 210 l 969 124 l 969 0 "},"ύ":{x_min:0,x_max:617,ha:725,o:"m 617 352 q 540 93 617 199 q 308 -24 455 -24 q 76 93 161 -24 q 0 352 0 199 l 0 738 l 126 738 l 126 354 q 169 185 126 257 q 312 98 220 98 q 451 185 402 98 q 492 354 492 257 l 492 738 l 617 738 l 617 352 m 535 1040 l 346 819 l 262 819 l 397 1040 l 535 1040 "},z:{x_min:-0.015625,x_max:613.890625,ha:697,o:"m 613 0 l 0 0 l 0 100 l 433 630 l 20 630 l 20 738 l 594 738 l 593 636 l 163 110 l 613 110 l 613 0 "},"™":{x_min:0,x_max:894,ha:1000,o:"m 389 951 l 229 951 l 229 503 l 160 503 l 160 951 l 0 951 l 0 1011 l 389 1011 l 389 951 m 894 503 l 827 503 l 827 939 l 685 503 l 620 503 l 481 937 l 481 503 l 417 503 l 417 1011 l 517 1011 l 653 580 l 796 1010 l 894 1011 l 894 503 "},"ή":{x_min:0.78125,x_max:697,ha:810,o:"m 697 -278 l 572 -278 l 572 454 q 540 587 572 536 q 425 650 501 650 q 271 579 337 650 q 206 420 206 509 l 206 0 l 81 0 l 81 489 q 73 588 81 562 q 0 644 56 644 l 0 741 q 68 755 38 755 q 158 721 124 755 q 200 630 193 687 q 297 726 234 692 q 434 761 359 761 q 620 692 544 761 q 697 516 697 624 l 697 -278 m 479 1040 l 290 819 l 207 819 l 341 1040 l 479 1040 "},"Θ":{x_min:0,x_max:960,ha:1056,o:"m 960 507 q 833 129 960 280 q 476 -32 698 -32 q 123 129 255 -32 q 0 507 0 280 q 123 883 0 732 q 476 1045 255 1045 q 832 883 696 1045 q 960 507 960 732 m 817 500 q 733 789 817 669 q 476 924 639 924 q 223 792 317 924 q 142 507 142 675 q 222 222 142 339 q 476 89 315 89 q 730 218 636 89 q 817 500 817 334 m 716 449 l 243 449 l 243 571 l 716 571 l 716 449 "},"®":{x_min:-3,x_max:1008,ha:1106,o:"m 503 532 q 614 562 566 532 q 672 658 672 598 q 614 747 672 716 q 503 772 569 772 l 338 772 l 338 532 l 503 532 m 502 -7 q 123 151 263 -7 q -3 501 -3 294 q 123 851 -3 706 q 502 1011 263 1011 q 881 851 739 1011 q 1008 501 1008 708 q 883 151 1008 292 q 502 -7 744 -7 m 502 60 q 830 197 709 60 q 940 501 940 322 q 831 805 940 681 q 502 944 709 944 q 174 805 296 944 q 65 501 65 680 q 173 197 65 320 q 502 60 294 60 m 788 146 l 678 146 q 653 316 655 183 q 527 449 652 449 l 338 449 l 338 146 l 241 146 l 241 854 l 518 854 q 688 808 621 854 q 766 658 766 755 q 739 563 766 607 q 668 497 713 519 q 751 331 747 472 q 788 164 756 190 l 788 146 "},"~":{x_min:0,x_max:833,ha:931,o:"m 833 958 q 778 753 833 831 q 594 665 716 665 q 402 761 502 665 q 240 857 302 857 q 131 795 166 857 q 104 665 104 745 l 0 665 q 54 867 0 789 q 237 958 116 958 q 429 861 331 958 q 594 765 527 765 q 704 827 670 765 q 729 958 729 874 l 833 958 "},"Ε":{x_min:0,x_max:736.21875,ha:778,o:"m 736 0 l 0 0 l 0 1013 l 725 1013 l 725 889 l 139 889 l 139 585 l 677 585 l 677 467 l 139 467 l 139 125 l 736 125 l 736 0 "},"³":{x_min:0,x_max:450,ha:547,o:"m 450 552 q 379 413 450 464 q 220 366 313 366 q 69 414 130 366 q 0 567 0 470 l 85 567 q 126 470 85 504 q 225 437 168 437 q 320 467 280 437 q 360 552 360 498 q 318 632 360 608 q 213 657 276 657 q 195 657 203 657 q 176 657 181 657 l 176 722 q 279 733 249 722 q 334 815 334 752 q 300 881 334 856 q 220 907 267 907 q 133 875 169 907 q 97 781 97 844 l 15 781 q 78 926 15 875 q 220 972 135 972 q 364 930 303 972 q 426 817 426 888 q 344 697 426 733 q 421 642 392 681 q 450 552 450 603 "},"[":{x_min:0,x_max:273.609375,ha:371,o:"m 273 -281 l 0 -281 l 0 1013 l 273 1013 l 273 920 l 124 920 l 124 -187 l 273 -187 l 273 -281 "},L:{x_min:0,x_max:645.828125,ha:696,o:"m 645 0 l 0 0 l 0 1013 l 140 1013 l 140 126 l 645 126 l 645 0 "},"σ":{x_min:0,x_max:803.390625,ha:894,o:"m 803 628 l 633 628 q 713 368 713 512 q 618 93 713 204 q 357 -25 518 -25 q 94 91 194 -25 q 0 368 0 201 q 94 644 0 533 q 356 761 194 761 q 481 750 398 761 q 608 739 564 739 l 803 739 l 803 628 m 360 85 q 529 180 467 85 q 584 374 584 262 q 527 566 584 490 q 352 651 463 651 q 187 559 247 651 q 135 368 135 478 q 189 175 135 254 q 360 85 251 85 "},"ζ":{x_min:0,x_max:573,ha:642,o:"m 573 -40 q 553 -162 573 -97 q 510 -278 543 -193 l 400 -278 q 441 -187 428 -219 q 462 -90 462 -132 q 378 -14 462 -14 q 108 45 197 -14 q 0 290 0 117 q 108 631 0 462 q 353 901 194 767 l 55 901 l 55 1012 l 561 1012 l 561 924 q 261 669 382 831 q 128 301 128 489 q 243 117 128 149 q 458 98 350 108 q 573 -40 573 80 "},"θ":{x_min:0,x_max:674,ha:778,o:"m 674 496 q 601 160 674 304 q 336 -26 508 -26 q 73 153 165 -26 q 0 485 0 296 q 72 840 0 683 q 343 1045 166 1045 q 605 844 516 1045 q 674 496 674 692 m 546 579 q 498 798 546 691 q 336 935 437 935 q 178 798 237 935 q 126 579 137 701 l 546 579 m 546 475 l 126 475 q 170 233 126 348 q 338 80 230 80 q 504 233 447 80 q 546 475 546 346 "},"Ο":{x_min:0,x_max:958,ha:1054,o:"m 485 1042 q 834 883 703 1042 q 958 511 958 735 q 834 136 958 287 q 481 -26 701 -26 q 126 130 261 -26 q 0 504 0 279 q 127 880 0 729 q 485 1042 263 1042 m 480 98 q 731 225 638 98 q 815 504 815 340 q 733 783 815 670 q 480 913 640 913 q 226 785 321 913 q 142 504 142 671 q 226 224 142 339 q 480 98 319 98 "},"Γ":{x_min:0,x_max:705.28125,ha:749,o:"m 705 886 l 140 886 l 140 0 l 0 0 l 0 1012 l 705 1012 l 705 886 "}," ":{x_min:0,x_max:0,ha:375},"%":{x_min:-3,x_max:1089,ha:1186,o:"m 845 0 q 663 76 731 0 q 602 244 602 145 q 661 412 602 344 q 845 489 728 489 q 1027 412 959 489 q 1089 244 1089 343 q 1029 76 1089 144 q 845 0 962 0 m 844 103 q 945 143 909 103 q 981 243 981 184 q 947 340 981 301 q 844 385 909 385 q 744 342 781 385 q 708 243 708 300 q 741 147 708 186 q 844 103 780 103 m 888 986 l 284 -25 l 199 -25 l 803 986 l 888 986 m 241 468 q 58 545 126 468 q -3 715 -3 615 q 56 881 -3 813 q 238 958 124 958 q 421 881 353 958 q 483 712 483 813 q 423 544 483 612 q 241 468 356 468 m 241 855 q 137 811 175 855 q 100 710 100 768 q 136 612 100 653 q 240 572 172 572 q 344 614 306 572 q 382 713 382 656 q 347 810 382 771 q 241 855 308 855 "},P:{x_min:0,x_max:726,ha:806,o:"m 424 1013 q 640 931 555 1013 q 726 719 726 850 q 637 506 726 587 q 413 426 548 426 l 140 426 l 140 0 l 0 0 l 0 1013 l 424 1013 m 379 889 l 140 889 l 140 548 l 372 548 q 522 589 459 548 q 593 720 593 637 q 528 845 593 801 q 379 889 463 889 "},"Έ":{x_min:0,x_max:1078.21875,ha:1118,o:"m 1078 0 l 342 0 l 342 1013 l 1067 1013 l 1067 889 l 481 889 l 481 585 l 1019 585 l 1019 467 l 481 467 l 481 125 l 1078 125 l 1078 0 m 277 1040 l 83 799 l 0 799 l 140 1040 l 277 1040 "},"Ώ":{x_min:0.125,x_max:1136.546875,ha:1235,o:"m 1136 0 l 722 0 l 722 123 q 911 309 842 194 q 981 558 981 423 q 893 813 981 710 q 651 923 800 923 q 411 821 501 923 q 321 568 321 720 q 390 316 321 433 q 579 123 459 200 l 579 0 l 166 0 l 166 124 l 384 124 q 235 327 289 210 q 182 572 182 444 q 311 912 182 782 q 651 1042 441 1042 q 989 910 858 1042 q 1120 569 1120 778 q 1066 326 1120 443 q 917 124 1013 210 l 1136 124 l 1136 0 m 277 1040 l 83 800 l 0 800 l 140 1041 l 277 1040 "},_:{x_min:0,x_max:705.5625,ha:803,o:"m 705 -334 l 0 -334 l 0 -234 l 705 -234 l 705 -334 "},"Ϊ":{x_min:-110,x_max:246,ha:275,o:"m 246 1046 l 118 1046 l 118 1189 l 246 1189 l 246 1046 m 18 1046 l -110 1046 l -110 1189 l 18 1189 l 18 1046 m 136 0 l 0 0 l 0 1012 l 136 1012 l 136 0 "},"+":{x_min:23,x_max:768,ha:792,o:"m 768 372 l 444 372 l 444 0 l 347 0 l 347 372 l 23 372 l 23 468 l 347 468 l 347 840 l 444 840 l 444 468 l 768 468 l 768 372 "},"½":{x_min:0,x_max:1050,ha:1149,o:"m 1050 0 l 625 0 q 712 178 625 108 q 878 277 722 187 q 967 385 967 328 q 932 456 967 429 q 850 484 897 484 q 759 450 798 484 q 721 352 721 416 l 640 352 q 706 502 640 448 q 851 551 766 551 q 987 509 931 551 q 1050 385 1050 462 q 976 251 1050 301 q 829 179 902 215 q 717 68 740 133 l 1050 68 l 1050 0 m 834 985 l 215 -28 l 130 -28 l 750 984 l 834 985 m 224 422 l 142 422 l 142 811 l 0 811 l 0 867 q 104 889 62 867 q 164 973 157 916 l 224 973 l 224 422 "},"Ρ":{x_min:0,x_max:720,ha:783,o:"m 424 1013 q 637 933 554 1013 q 720 723 720 853 q 633 508 720 591 q 413 426 546 426 l 140 426 l 140 0 l 0 0 l 0 1013 l 424 1013 m 378 889 l 140 889 l 140 548 l 371 548 q 521 589 458 548 q 592 720 592 637 q 527 845 592 801 q 378 889 463 889 "},"'":{x_min:0,x_max:139,ha:236,o:"m 139 851 q 102 737 139 784 q 0 669 65 690 l 0 734 q 59 787 42 741 q 72 873 72 821 l 0 873 l 0 1013 l 139 1013 l 139 851 "},"ª":{x_min:0,x_max:350,ha:397,o:"m 350 625 q 307 616 328 616 q 266 631 281 616 q 247 673 251 645 q 190 628 225 644 q 116 613 156 613 q 32 641 64 613 q 0 722 0 669 q 72 826 0 800 q 247 866 159 846 l 247 887 q 220 934 247 916 q 162 953 194 953 q 104 934 129 953 q 76 882 80 915 l 16 882 q 60 976 16 941 q 166 1011 104 1011 q 266 979 224 1011 q 308 891 308 948 l 308 706 q 311 679 308 688 q 331 670 315 670 l 350 672 l 350 625 m 247 757 l 247 811 q 136 790 175 798 q 64 726 64 773 q 83 682 64 697 q 132 667 103 667 q 207 690 174 667 q 247 757 247 718 "},"΅":{x_min:0,x_max:450,ha:553,o:"m 450 800 l 340 800 l 340 925 l 450 925 l 450 800 m 406 1040 l 212 800 l 129 800 l 269 1040 l 406 1040 m 110 800 l 0 800 l 0 925 l 110 925 l 110 800 "},T:{x_min:0,x_max:777,ha:835,o:"m 777 894 l 458 894 l 458 0 l 319 0 l 319 894 l 0 894 l 0 1013 l 777 1013 l 777 894 "},"Φ":{x_min:0,x_max:915,ha:997,o:"m 527 0 l 389 0 l 389 122 q 110 231 220 122 q 0 509 0 340 q 110 785 0 677 q 389 893 220 893 l 389 1013 l 527 1013 l 527 893 q 804 786 693 893 q 915 509 915 679 q 805 231 915 341 q 527 122 696 122 l 527 0 m 527 226 q 712 310 641 226 q 779 507 779 389 q 712 705 779 627 q 527 787 641 787 l 527 226 m 389 226 l 389 787 q 205 698 275 775 q 136 505 136 620 q 206 308 136 391 q 389 226 276 226 "},"⁋":{x_min:0,x_max:0,ha:694},j:{x_min:-77.78125,x_max:167,ha:349,o:"m 167 871 l 42 871 l 42 1013 l 167 1013 l 167 871 m 167 -80 q 121 -231 167 -184 q -26 -278 76 -278 l -77 -278 l -77 -164 l -41 -164 q 26 -143 11 -164 q 42 -65 42 -122 l 42 737 l 167 737 l 167 -80 "},"Σ":{x_min:0,x_max:756.953125,ha:819,o:"m 756 0 l 0 0 l 0 107 l 395 523 l 22 904 l 22 1013 l 745 1013 l 745 889 l 209 889 l 566 523 l 187 125 l 756 125 l 756 0 "},"›":{x_min:18.0625,x_max:774,ha:792,o:"m 774 376 l 18 40 l 18 149 l 631 421 l 18 692 l 18 799 l 774 465 l 774 376 "},"<":{x_min:17.984375,x_max:773.609375,ha:792,o:"m 773 40 l 18 376 l 17 465 l 773 799 l 773 692 l 159 420 l 773 149 l 773 40 "},"£":{x_min:0,x_max:704.484375,ha:801,o:"m 704 41 q 623 -10 664 5 q 543 -26 583 -26 q 359 15 501 -26 q 243 36 288 36 q 158 23 197 36 q 73 -21 119 10 l 6 76 q 125 195 90 150 q 175 331 175 262 q 147 443 175 383 l 0 443 l 0 512 l 108 512 q 43 734 43 623 q 120 929 43 854 q 358 1010 204 1010 q 579 936 487 1010 q 678 729 678 857 l 678 684 l 552 684 q 504 838 552 780 q 362 896 457 896 q 216 852 263 896 q 176 747 176 815 q 199 627 176 697 q 248 512 217 574 l 468 512 l 468 443 l 279 443 q 297 356 297 398 q 230 194 297 279 q 153 107 211 170 q 227 133 190 125 q 293 142 264 142 q 410 119 339 142 q 516 96 482 96 q 579 105 550 96 q 648 142 608 115 l 704 41 "},t:{x_min:0,x_max:367,ha:458,o:"m 367 0 q 312 -5 339 -2 q 262 -8 284 -8 q 145 28 183 -8 q 108 143 108 64 l 108 638 l 0 638 l 0 738 l 108 738 l 108 944 l 232 944 l 232 738 l 367 738 l 367 638 l 232 638 l 232 185 q 248 121 232 140 q 307 102 264 102 q 345 104 330 102 q 367 107 360 107 l 367 0 "},"¬":{x_min:0,x_max:706,ha:803,o:"m 706 411 l 706 158 l 630 158 l 630 335 l 0 335 l 0 411 l 706 411 "},"λ":{x_min:0,x_max:750,ha:803,o:"m 750 -7 q 679 -15 716 -15 q 538 59 591 -15 q 466 214 512 97 l 336 551 l 126 0 l 0 0 l 270 705 q 223 837 247 770 q 116 899 190 899 q 90 898 100 899 l 90 1004 q 152 1011 125 1011 q 298 938 244 1011 q 373 783 326 901 l 605 192 q 649 115 629 136 q 716 95 669 95 l 736 95 q 750 97 745 97 l 750 -7 "},W:{x_min:0,x_max:1263.890625,ha:1351,o:"m 1263 1013 l 995 0 l 859 0 l 627 837 l 405 0 l 265 0 l 0 1013 l 136 1013 l 342 202 l 556 1013 l 701 1013 l 921 207 l 1133 1012 l 1263 1013 "},">":{x_min:18.0625,x_max:774,ha:792,o:"m 774 376 l 18 40 l 18 149 l 631 421 l 18 692 l 18 799 l 774 465 l 774 376 "},v:{x_min:0,x_max:675.15625,ha:761,o:"m 675 738 l 404 0 l 272 0 l 0 738 l 133 737 l 340 147 l 541 737 l 675 738 "},"τ":{x_min:0.28125,x_max:644.5,ha:703,o:"m 644 628 l 382 628 l 382 179 q 388 120 382 137 q 436 91 401 91 q 474 94 447 91 q 504 97 501 97 l 504 0 q 454 -9 482 -5 q 401 -14 426 -14 q 278 67 308 -14 q 260 233 260 118 l 260 628 l 0 628 l 0 739 l 644 739 l 644 628 "},"ξ":{x_min:0,x_max:624.9375,ha:699,o:"m 624 -37 q 608 -153 624 -96 q 563 -278 593 -211 l 454 -278 q 491 -183 486 -200 q 511 -83 511 -126 q 484 -23 511 -44 q 370 1 452 1 q 323 0 354 1 q 283 -1 293 -1 q 84 76 169 -1 q 0 266 0 154 q 56 431 0 358 q 197 538 108 498 q 94 613 134 562 q 54 730 54 665 q 77 823 54 780 q 143 901 101 867 l 27 901 l 27 1012 l 576 1012 l 576 901 l 380 901 q 244 863 303 901 q 178 745 178 820 q 312 600 178 636 q 532 582 380 582 l 532 479 q 276 455 361 479 q 118 281 118 410 q 165 173 118 217 q 274 120 208 133 q 494 101 384 110 q 624 -37 624 76 "},"&":{x_min:-3,x_max:894.25,ha:992,o:"m 894 0 l 725 0 l 624 123 q 471 0 553 40 q 306 -41 390 -41 q 168 -7 231 -41 q 62 92 105 26 q 14 187 31 139 q -3 276 -3 235 q 55 433 -3 358 q 248 581 114 508 q 170 689 196 640 q 137 817 137 751 q 214 985 137 922 q 384 1041 284 1041 q 548 988 483 1041 q 622 824 622 928 q 563 666 622 739 q 431 556 516 608 l 621 326 q 649 407 639 361 q 663 493 653 426 l 781 493 q 703 229 781 352 l 894 0 m 504 818 q 468 908 504 877 q 384 940 433 940 q 293 907 331 940 q 255 818 255 875 q 289 714 255 767 q 363 628 313 678 q 477 729 446 682 q 504 818 504 771 m 556 209 l 314 499 q 179 395 223 449 q 135 283 135 341 q 146 222 135 253 q 183 158 158 192 q 333 80 241 80 q 556 209 448 80 "},"Λ":{x_min:0,x_max:862.5,ha:942,o:"m 862 0 l 719 0 l 426 847 l 143 0 l 0 0 l 356 1013 l 501 1013 l 862 0 "},I:{x_min:41,x_max:180,ha:293,o:"m 180 0 l 41 0 l 41 1013 l 180 1013 l 180 0 "},G:{x_min:0,x_max:921,ha:1011,o:"m 921 0 l 832 0 l 801 136 q 655 15 741 58 q 470 -28 568 -28 q 126 133 259 -28 q 0 499 0 284 q 125 881 0 731 q 486 1043 259 1043 q 763 957 647 1043 q 905 709 890 864 l 772 709 q 668 866 747 807 q 486 926 589 926 q 228 795 322 926 q 142 507 142 677 q 228 224 142 342 q 483 94 323 94 q 712 195 625 94 q 796 435 796 291 l 477 435 l 477 549 l 921 549 l 921 0 "},"ΰ":{x_min:0,x_max:617,ha:725,o:"m 524 800 l 414 800 l 414 925 l 524 925 l 524 800 m 183 800 l 73 800 l 73 925 l 183 925 l 183 800 m 617 352 q 540 93 617 199 q 308 -24 455 -24 q 76 93 161 -24 q 0 352 0 199 l 0 738 l 126 738 l 126 354 q 169 185 126 257 q 312 98 220 98 q 451 185 402 98 q 492 354 492 257 l 492 738 l 617 738 l 617 352 m 489 1040 l 300 819 l 216 819 l 351 1040 l 489 1040 "},"`":{x_min:0,x_max:138.890625,ha:236,o:"m 138 699 l 0 699 l 0 861 q 36 974 0 929 q 138 1041 72 1020 l 138 977 q 82 931 95 969 q 69 839 69 893 l 138 839 l 138 699 "},"·":{x_min:0,x_max:142,ha:239,o:"m 142 585 l 0 585 l 0 738 l 142 738 l 142 585 "},"Υ":{x_min:0.328125,x_max:819.515625,ha:889,o:"m 819 1013 l 482 416 l 482 0 l 342 0 l 342 416 l 0 1013 l 140 1013 l 411 533 l 679 1013 l 819 1013 "},r:{x_min:0,x_max:355.5625,ha:432,o:"m 355 621 l 343 621 q 179 569 236 621 q 122 411 122 518 l 122 0 l 0 0 l 0 737 l 117 737 l 117 604 q 204 719 146 686 q 355 753 262 753 l 355 621 "},x:{x_min:0,x_max:675,ha:764,o:"m 675 0 l 525 0 l 331 286 l 144 0 l 0 0 l 256 379 l 12 738 l 157 737 l 336 473 l 516 738 l 661 738 l 412 380 l 675 0 "},"μ":{x_min:0,x_max:696.609375,ha:747,o:"m 696 -4 q 628 -14 657 -14 q 498 97 513 -14 q 422 8 470 41 q 313 -24 374 -24 q 207 3 258 -24 q 120 80 157 31 l 120 -278 l 0 -278 l 0 738 l 124 738 l 124 343 q 165 172 124 246 q 308 82 216 82 q 451 177 402 82 q 492 358 492 254 l 492 738 l 616 738 l 616 214 q 623 136 616 160 q 673 92 636 92 q 696 95 684 92 l 696 -4 "},h:{x_min:0,x_max:615,ha:724,o:"m 615 472 l 615 0 l 490 0 l 490 454 q 456 590 490 535 q 338 654 416 654 q 186 588 251 654 q 122 436 122 522 l 122 0 l 0 0 l 0 1013 l 122 1013 l 122 633 q 218 727 149 694 q 362 760 287 760 q 552 676 484 760 q 615 472 615 600 "},".":{x_min:0,x_max:142,ha:239,o:"m 142 0 l 0 0 l 0 151 l 142 151 l 142 0 "},"φ":{x_min:-2,x_max:878,ha:974,o:"m 496 -279 l 378 -279 l 378 -17 q 101 88 204 -17 q -2 367 -2 194 q 68 626 -2 510 q 283 758 151 758 l 283 646 q 167 537 209 626 q 133 373 133 462 q 192 177 133 254 q 378 93 259 93 l 378 758 q 445 764 426 763 q 476 765 464 765 q 765 659 653 765 q 878 377 878 553 q 771 96 878 209 q 496 -17 665 -17 l 496 -279 m 496 93 l 514 93 q 687 183 623 93 q 746 380 746 265 q 691 569 746 491 q 522 658 629 658 l 496 656 l 496 93 "},";":{x_min:0,x_max:142,ha:239,o:"m 142 585 l 0 585 l 0 738 l 142 738 l 142 585 m 142 -12 q 105 -132 142 -82 q 0 -206 68 -182 l 0 -138 q 58 -82 43 -123 q 68 0 68 -56 l 0 0 l 0 151 l 142 151 l 142 -12 "},f:{x_min:0,x_max:378,ha:472,o:"m 378 638 l 246 638 l 246 0 l 121 0 l 121 638 l 0 638 l 0 738 l 121 738 q 137 935 121 887 q 290 1028 171 1028 q 320 1027 305 1028 q 378 1021 334 1026 l 378 908 q 323 918 346 918 q 257 870 273 918 q 246 780 246 840 l 246 738 l 378 738 l 378 638 "},"“":{x_min:1,x_max:348.21875,ha:454,o:"m 140 670 l 1 670 l 1 830 q 37 943 1 897 q 140 1011 74 990 l 140 947 q 82 900 97 940 q 68 810 68 861 l 140 810 l 140 670 m 348 670 l 209 670 l 209 830 q 245 943 209 897 q 348 1011 282 990 l 348 947 q 290 900 305 940 q 276 810 276 861 l 348 810 l 348 670 "},A:{x_min:0.03125,x_max:906.953125,ha:1008,o:"m 906 0 l 756 0 l 648 303 l 251 303 l 142 0 l 0 0 l 376 1013 l 529 1013 l 906 0 m 610 421 l 452 867 l 293 421 l 610 421 "},"‘":{x_min:1,x_max:139.890625,ha:236,o:"m 139 670 l 1 670 l 1 830 q 37 943 1 897 q 139 1011 74 990 l 139 947 q 82 900 97 940 q 68 810 68 861 l 139 810 l 139 670 "},"ϊ":{x_min:-70,x_max:283,ha:361,o:"m 283 800 l 173 800 l 173 925 l 283 925 l 283 800 m 40 800 l -70 800 l -70 925 l 40 925 l 40 800 m 283 3 q 232 -10 257 -5 q 181 -15 206 -15 q 84 26 118 -15 q 41 200 41 79 l 41 737 l 166 737 l 167 215 q 171 141 167 157 q 225 101 182 101 q 247 103 238 101 q 283 112 256 104 l 283 3 "},"π":{x_min:-0.21875,x_max:773.21875,ha:857,o:"m 773 -7 l 707 -11 q 575 40 607 -11 q 552 174 552 77 l 552 226 l 552 626 l 222 626 l 222 0 l 97 0 l 97 626 l 0 626 l 0 737 l 773 737 l 773 626 l 676 626 l 676 171 q 695 103 676 117 q 773 90 714 90 l 773 -7 "},"ά":{x_min:0,x_max:765.5625,ha:809,o:"m 765 -4 q 698 -14 726 -14 q 564 97 586 -14 q 466 7 525 40 q 337 -26 407 -26 q 88 98 186 -26 q 0 369 0 212 q 88 637 0 525 q 337 760 184 760 q 465 727 407 760 q 563 637 524 695 l 563 738 l 685 738 l 685 222 q 693 141 685 168 q 748 94 708 94 q 765 95 760 94 l 765 -4 m 584 371 q 531 562 584 485 q 360 653 470 653 q 192 566 254 653 q 135 379 135 489 q 186 181 135 261 q 358 84 247 84 q 528 176 465 84 q 584 371 584 260 m 604 1040 l 415 819 l 332 819 l 466 1040 l 604 1040 "},O:{x_min:0,x_max:958,ha:1057,o:"m 485 1041 q 834 882 702 1041 q 958 512 958 734 q 834 136 958 287 q 481 -26 702 -26 q 126 130 261 -26 q 0 504 0 279 q 127 880 0 728 q 485 1041 263 1041 m 480 98 q 731 225 638 98 q 815 504 815 340 q 733 783 815 669 q 480 912 640 912 q 226 784 321 912 q 142 504 142 670 q 226 224 142 339 q 480 98 319 98 "},n:{x_min:0,x_max:615,ha:724,o:"m 615 463 l 615 0 l 490 0 l 490 454 q 453 592 490 537 q 331 656 410 656 q 178 585 240 656 q 117 421 117 514 l 117 0 l 0 0 l 0 738 l 117 738 l 117 630 q 218 728 150 693 q 359 764 286 764 q 552 675 484 764 q 615 463 615 593 "},l:{x_min:41,x_max:166,ha:279,o:"m 166 0 l 41 0 l 41 1013 l 166 1013 l 166 0 "},"¤":{x_min:40.09375,x_max:728.796875,ha:825,o:"m 728 304 l 649 224 l 512 363 q 383 331 458 331 q 256 363 310 331 l 119 224 l 40 304 l 177 441 q 150 553 150 493 q 184 673 150 621 l 40 818 l 119 898 l 267 749 q 321 766 291 759 q 384 773 351 773 q 447 766 417 773 q 501 749 477 759 l 649 898 l 728 818 l 585 675 q 612 618 604 648 q 621 553 621 587 q 591 441 621 491 l 728 304 m 384 682 q 280 643 318 682 q 243 551 243 604 q 279 461 243 499 q 383 423 316 423 q 487 461 449 423 q 525 553 525 500 q 490 641 525 605 q 384 682 451 682 "},"κ":{x_min:0,x_max:632.328125,ha:679,o:"m 632 0 l 482 0 l 225 384 l 124 288 l 124 0 l 0 0 l 0 738 l 124 738 l 124 446 l 433 738 l 596 738 l 312 466 l 632 0 "},p:{x_min:0,x_max:685,ha:786,o:"m 685 364 q 598 96 685 205 q 350 -23 504 -23 q 121 89 205 -23 l 121 -278 l 0 -278 l 0 738 l 121 738 l 121 633 q 220 726 159 691 q 351 761 280 761 q 598 636 504 761 q 685 364 685 522 m 557 371 q 501 560 557 481 q 330 651 437 651 q 162 559 223 651 q 108 366 108 479 q 162 177 108 254 q 333 87 224 87 q 502 178 441 87 q 557 371 557 258 "},"‡":{x_min:0,x_max:777,ha:835,o:"m 458 238 l 458 0 l 319 0 l 319 238 l 0 238 l 0 360 l 319 360 l 319 681 l 0 683 l 0 804 l 319 804 l 319 1015 l 458 1013 l 458 804 l 777 804 l 777 683 l 458 683 l 458 360 l 777 360 l 777 238 l 458 238 "},"ψ":{x_min:0,x_max:808,ha:907,o:"m 465 -278 l 341 -278 l 341 -15 q 87 102 180 -15 q 0 378 0 210 l 0 739 l 133 739 l 133 379 q 182 195 133 275 q 341 98 242 98 l 341 922 l 465 922 l 465 98 q 623 195 563 98 q 675 382 675 278 l 675 742 l 808 742 l 808 381 q 720 104 808 213 q 466 -13 627 -13 l 465 -278 "},"η":{x_min:0.78125,x_max:697,ha:810,o:"m 697 -278 l 572 -278 l 572 454 q 540 587 572 536 q 425 650 501 650 q 271 579 337 650 q 206 420 206 509 l 206 0 l 81 0 l 81 489 q 73 588 81 562 q 0 644 56 644 l 0 741 q 68 755 38 755 q 158 720 124 755 q 200 630 193 686 q 297 726 234 692 q 434 761 359 761 q 620 692 544 761 q 697 516 697 624 l 697 -278 "}};
// eslint-disable-next-line
const cssFontWeight='normal', ascender=1189, underlinePosition=-100, cssFontStyle='normal', boundingBox={yMin:-334,xMin:-111,yMax:1189,xMax:1672}, resolution = 1000, original_font_information={postscript_name:"Helvetiker-Regular",version_string:"Version 1.00 2004 initial release",vendor_url:"http://www.magenta.gr/",full_font_name:"Helvetiker",font_family_name:"Helvetiker",copyright:"Copyright (c) Μagenta ltd, 2004",description:"",trademark:"",designer:"",designer_url:"",unique_font_identifier:"Μagenta ltd:Helvetiker:22-10-104",license_url:"http://www.ellak.gr/fonts/MgOpen/license.html",license_description:"Copyright (c) 2004 by MAGENTA Ltd. All Rights Reserved.\r\n\r\nPermission is hereby granted, free of charge, to any person obtaining a copy of the fonts accompanying this license (\"Fonts\") and associated documentation files (the \"Font Software\"), to reproduce and distribute the Font Software, including without limitation the rights to use, copy, merge, publish, distribute, and/or sell copies of the Font Software, and to permit persons to whom the Font Software is furnished to do so, subject to the following conditions: \r\n\r\nThe above copyright and this permission notice shall be included in all copies of one or more of the Font Software typefaces.\r\n\r\nThe Font Software may be modified, altered, or added to, and in particular the designs of glyphs or characters in the Fonts may be modified and additional glyphs or characters may be added to the Fonts, only if the fonts are renamed to names not containing the word \"MgOpen\", or if the modifications are accepted for inclusion in the Font Software itself by the each appointed Administrator.\r\n\r\nThis License becomes null and void to the extent applicable to Fonts or Font Software that has been modified and is distributed under the \"MgOpen\" name.\r\n\r\nThe Font Software may be sold as part of a larger software package but no copy of one or more of the Font Software typefaces may be sold by itself. \r\n\r\nTHE FONT SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF COPYRIGHT, PATENT, TRADEMARK, OR OTHER RIGHT. IN NO EVENT SHALL MAGENTA OR PERSONS OR BODIES IN CHARGE OF ADMINISTRATION AND MAINTENANCE OF THE FONT SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, INCLUDING ANY GENERAL, SPECIAL, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF THE USE OR INABILITY TO USE THE FONT SOFTWARE OR FROM OTHER DEALINGS IN THE FONT SOFTWARE.",manufacturer_name:"Μagenta ltd",font_sub_family_name:"Regular"}, descender = -334, familyName = 'Helvetiker', lineHeight = 1522, underlineThickness = 50, helvetiker_regular_typeface = {glyphs, cssFontWeight, ascender, underlinePosition, cssFontStyle, boundingBox, resolution,original_font_information, descender, familyName, lineHeight, underlineThickness};

   _hfont = new THREE.Font({
      ascender, boundingBox, cssFontStyle, cssFontWeight, default: helvetiker_regular_typeface, descender, familyName, glyphs,
      lineHeight, original_font_information, resolution, underlinePosition, underlineThickness
   });

   return _hfont;
}

/** @summary Create three.js Color instance, handles optional opacity
  * @private */
function getMaterialArgs(color, args) {
   if (!args || !isObject(args)) args = {};

   if (isStr(color) && (((color[0] === '#') && (color.length === 9)) || (color.indexOf('rgba') >= 0))) {
      const col = d3_color(color);
      args.color = new THREE.Color(col.r, col.g, col.b);
      args.opacity = col.opacity ?? 1;
      args.transparent = args.opacity < 1;
   } else
      args.color = new THREE.Color(color);
   return args;
}

function createSVGRenderer(as_is, precision, doc) {
   if (as_is) {
      if (doc !== undefined)
         globalThis.docuemnt = doc;
      const rndr = new THREE.SVGRenderer();
      rndr.setPrecision(precision);
      return rndr;
   }

   const excl_style1 = ';stroke-opacity:1;stroke-width:1;stroke-linecap:round',
         excl_style2 = ';fill-opacity:1',
   doc_wrapper = {
     svg_attr: {},
     svg_style: {},
     path_attr: {},
     accPath: '',
     createElementNS(ns, kind) {
        if (kind === 'path') {
           return {
              _wrapper: this,
              setAttribute(name, value) {
                 // cut useless fill-opacity:1 at the end of many SVG attributes
                 if ((name === 'style') && value) {
                    const pos1 = value.indexOf(excl_style1);
                    if ((pos1 >= 0) && (pos1 === value.length - excl_style1.length))
                       value = value.slice(0, value.length - excl_style1.length);
                    const pos2 = value.indexOf(excl_style2);
                    if ((pos2 >= 0) && (pos2 === value.length - excl_style2.length))
                       value = value.slice(0, value.length - excl_style2.length);
                 }
                 this._wrapper.path_attr[name] = value;
              }
           };
        }

        if (kind !== 'svg') {
           console.error(`not supported element for SVGRenderer ${kind}`);
           return null;
        }

        return {
           _wrapper: this,
           childNodes: [], // may be accessed - make dummy
           style: this.svg_style, // for background color
           setAttribute(name, value) {
              this._wrapper.svg_attr[name] = value;
           },
           appendChild(/* node */) {
              this._wrapper.accPath += `<path style="${this._wrapper.path_attr.style}" d="${this._wrapper.path_attr.d}"/>`;
              this._wrapper.path_attr = {};
           },
           removeChild(/* node */) {
              this.childNodes = [];
           }
        };
     }
   };

   let originalDocument;

   if (isNodeJs()) {
      originalDocument = globalThis.document;
      globalThis.document = doc_wrapper;
   }

   const rndr = new THREE.SVGRenderer();

   if (isNodeJs())
      globalThis.document = originalDocument;

   rndr.doc_wrapper = doc_wrapper; // use it to get final SVG code

   rndr.originalRender = rndr.render;

   rndr.render = function(scene, camera) {
      const _doc = globalThis.document;
      if (isNodeJs())
         globalThis.document = this.doc_wrapper;

      this.originalRender(scene, camera);

      if (isNodeJs())
         globalThis.document = _doc;
   };

   rndr.clearHTML = function() {
      this.doc_wrapper.accPath = '';
   };

   rndr.makeOuterHTML = function() {
      const wrap = this.doc_wrapper,
           _textSizeAttr = `viewBox="${wrap.svg_attr.viewBox}" width="${wrap.svg_attr.width}" height="${wrap.svg_attr.height}"`,
           _textClearAttr = wrap.svg_style.backgroundColor ? ` style="background:${wrap.svg_style.backgroundColor}"` : '';

      return `<svg xmlns="${nsSVG}" ${_textSizeAttr}${_textClearAttr}>${wrap.accPath}</svg>`;
   };

   rndr.fillTargetSVG = function(svg) {
      if (isNodeJs()) {
         const wrap = this.doc_wrapper;

         svg.setAttribute('viewBox', wrap.svg_attr.viewBox);
         svg.setAttribute('width', wrap.svg_attr.width);
         svg.setAttribute('height', wrap.svg_attr.height);
         svg.style.background = wrap.svg_style.backgroundColor || '';

         svg.innerHTML = wrap.accPath;
      } else {
         const src = this.domElement;

         svg.setAttribute('viewBox', src.getAttribute('viewBox'));
         svg.setAttribute('width', src.getAttribute('width'));
         svg.setAttribute('height', src.getAttribute('height'));
         svg.style.background = src.style.backgroundColor;

         while (src.firstChild) {
            const elem = src.firstChild;
            src.removeChild(elem);
            svg.appendChild(elem);
         }
      }
   };

   rndr.setPrecision(precision);

   return rndr;
}


/** @summary Define rendering kind which will be used for rendering of 3D elements
  * @param {value} [render3d] - preconfigured value, will be used if applicable
  * @param {value} [is_batch] - is batch mode is configured
  * @return {value} - rendering kind, see constants.Render3D
  * @private */
function getRender3DKind(render3d, is_batch) {
   if (is_batch === undefined)
      is_batch = isBatchMode();

   if (!render3d) render3d = is_batch ? settings.Render3DBatch : settings.Render3D;
   const rc = constants.Render3D;

   if (render3d === rc.Default) render3d = is_batch ? rc.WebGLImage : rc.WebGL;
   if (is_batch && (render3d === rc.WebGL)) render3d = rc.WebGLImage;

   return render3d;
}

const Handling3DDrawings = {

   /** @summary Access current 3d mode
     * @param {string} [new_value] - when specified, set new 3d mode
     * @return current value
     * @private */
   access3dKind(new_value) {
      const svg = this.getPadSvg();
      if (svg.empty())
         return -1;

      // returns kind of currently created 3d canvas
      const kind = svg.property('can3d');
      if (new_value !== undefined)
         svg.property('can3d', new_value);
      return ((kind === null) || (kind === undefined)) ? -1 : kind;
   },

   /** @summary Returns size which availble for 3D drawing.
     * @desc One uses frame sizes for the 3D drawing - like TH2/TH3 objects
     * @private */
   getSizeFor3d(can3d /* , render3d */) {
      if (can3d === undefined) {
         // analyze which render/embed mode can be used
         can3d = getRender3DKind();
         // all non-webgl elements can be embedded into SVG as is
         if (can3d !== constants.Render3D.WebGL)
            can3d = constants.Embed3D.EmbedSVG;
         else if (settings.Embed3D !== constants.Embed3D.Default)
            can3d = settings.Embed3D;
         else if (browser.isFirefox)
            can3d = constants.Embed3D.Embed;
         else if (browser.chromeVersion > 95)
            // version 96 works partially, 97 works fine
            can3d = constants.Embed3D.Embed;
         else
            can3d = constants.Embed3D.Overlay;
      }

      const pad = this.getPadSvg(),
            clname = 'draw3d_' + (this.getPadName() || 'canvas');

      if (pad.empty()) {
         // this is a case when object drawn without canvas

         const rect = getElementRect(this.selectDom());
         if ((rect.height < 10) && (rect.width > 10)) {
            rect.height = Math.round(0.66 * rect.width);
            this.selectDom().style('height', rect.height + 'px');
         }
         rect.x = 0; rect.y = 0; rect.clname = clname; rect.can3d = -1;
         return rect;
      }

      const fp = this.getFramePainter(), pp = this.getPadPainter();
      let size;

      if (fp?.mode3d && (can3d > 0))
         size = fp.getFrameRect();
      else {
         let elem = (can3d > 0) ? pad : this.getCanvSvg();
         size = { x: 0, y: 0, width: elem.property('draw_width'), height: elem.property('draw_height') };
         if (Number.isNaN(size.width) || Number.isNaN(size.height)) {
            size.width = pp.getPadWidth();
            size.height = pp.getPadHeight();
         } else if (fp && !fp.mode3d) {
            elem = this.getFrameSvg();
            size.x = elem.property('draw_x');
            size.y = elem.property('draw_y');
         }
      }

      size.clname = clname;
      size.can3d = can3d;

      const rect = pp?.getPadRect();
      if (rect) {
         // while 3D canvas uses area also for the axis labels, extend area relative to normal frame
         const dx = Math.round(size.width*0.07), dy = Math.round(size.height*0.05);

         size.x = Math.max(0, size.x - dx);
         size.y = Math.max(0, size.y - dy);
         size.width = Math.min(size.width + 2*dx, rect.width - size.x);
         size.height = Math.min(size.height + 2*dy, rect.height - size.y);
      }

      if (can3d === constants.Embed3D.Overlay) {
         size = getAbsPosInCanvas(this.getPadSvg(), size);
         const scale = this.getCanvPainter().getPadScale();
         if (scale && scale !== 1) {
            size.x /= scale;
            size.y /= scale;
            size.width /= scale;
            size.height /= scale;
         }
      }

      return size;
   },

   /** @summary Clear all 3D drawings
     * @return can3d value - how webgl canvas was placed
     * @private */
   clear3dCanvas() {
      const can3d = this.access3dKind(null);
      if (can3d < 0) {
         // remove first child from main element - if it is canvas
         const main = this.selectDom().node();
         let chld = main?.firstChild;

         if (chld && !chld.$jsroot)
            chld = chld.nextSibling;

         if (chld?.$jsroot) {
            delete chld.painter;
            main.removeChild(chld);
         }
         return can3d;
      }

      const size = this.getSizeFor3d(can3d);
      if (size.can3d === 0) {
         d3_select(this.getCanvSvg().node().nextSibling).remove(); // remove html5 canvas
         this.getCanvSvg().style('display', null); // show SVG canvas
      } else {
         if (this.getPadSvg().empty()) return;

         this.apply3dSize(size).remove();

         this.getFrameSvg().style('display', null);  // clear display property
      }
      return can3d;
   },

   /** @summary Add 3D canvas
     * @private */
   add3dCanvas(size, canv, webgl) {
      if (!canv || (size.can3d < -1)) return;

      if (size.can3d === -1) {
         // case when 3D object drawn without canvas

         const main = this.selectDom().node();
         if (main !== null) {
            main.appendChild(canv);
            canv.painter = this;
            canv.$jsroot = true; // mark canvas as added by jsroot
         }

         return;
      }

      if ((size.can3d > 0) && !webgl)
         size.can3d = constants.Embed3D.EmbedSVG;

      this.access3dKind(size.can3d);

      if (size.can3d === 0) {
         this.getCanvSvg().style('display', 'none'); // hide SVG canvas

         this.getCanvSvg().node().parentNode.appendChild(canv); // add directly
      } else {
         if (this.getPadSvg().empty()) return;

         // first hide normal frame
         this.getFrameSvg().style('display', 'none');

         const elem = this.apply3dSize(size);
         elem.attr('title', '').node().appendChild(canv);
      }
   },

   /** @summary Apply size to 3D elements
     * @private */
   apply3dSize(size, onlyget) {
      if (size.can3d < 0)
         return d3_select(null);

      let elem;

      if (size.can3d > 1) {
         elem = this.getLayerSvg(size.clname);
         if (onlyget)
            return elem;

         const svg = this.getPadSvg();

         if (size.can3d === constants.Embed3D.EmbedSVG) {
            // this is SVG mode or image mode - just create group to hold element

            if (elem.empty())
               elem = svg.insert('g', '.primitives_layer').attr('class', size.clname);

            makeTranslate(elem, size.x, size.y);
         } else {
            if (elem.empty())
               elem = svg.insert('foreignObject', '.primitives_layer').attr('class', size.clname);

            elem.attr('x', size.x)
                .attr('y', size.y)
                .attr('width', size.width)
                .attr('height', size.height)
                .attr('viewBox', `0 0 ${size.width} ${size.height}`)
                .attr('preserveAspectRatio', 'xMidYMid');
         }
      } else {
         let prnt = this.getCanvSvg().node().parentNode;

         elem = d3_select(prnt).select('.' + size.clname);
         if (onlyget)
            return elem;

         // force redraw by resize
         this.getCanvSvg().property('redraw_by_resize', true);

         if (elem.empty()) {
            elem = d3_select(prnt).append('div').attr('class', size.clname)
                                  .style('user-select', 'none');
         }

         // our position inside canvas, but to set 'absolute' position we should use
         // canvas element offset relative to first parent with non-static position
         // now try to use getBoundingClientRect - it should be more precise

         const pos0 = prnt.getBoundingClientRect(), doc = getDocument();

         while (prnt) {
            if (prnt === doc) {
               prnt = null;
               break;
            }
            try {
               if (getComputedStyle(prnt).position !== 'static') break;
            } catch {
               break;
            }
            prnt = prnt.parentNode;
         }

         const pos1 = prnt?.getBoundingClientRect() ?? { top: 0, left: 0 },
               offx = Math.round(pos0.left - pos1.left),
               offy = Math.round(pos0.top - pos1.top);

         elem.style('position', 'absolute').style('left', (size.x + offx) + 'px').style('top', (size.y + offy) + 'px').style('width', size.width + 'px').style('height', size.height + 'px');
      }

      return elem;
   }

}; // Handling3DDrawings


/** @summary Assigns method to handle 3D drawings inside SVG
  * @private */
function assign3DHandler(painter) {
   Object.assign(painter, Handling3DDrawings);
}


/** @summary Creates renderer for the 3D drawings
  * @param {value} width - rendering width
  * @param {value} height - rendering height
  * @param {value} render3d - render type, see {@link constants.Render3D}
  * @param {object} args - different arguments for creating 3D renderer
  * @return {Promise} with renderer object
  * @private */
async function createRender3D(width, height, render3d, args) {
   const rc = constants.Render3D, doc = getDocument();

   render3d = getRender3DKind(render3d);

   if (!args) args = { antialias: true, alpha: true };

   let promise;

   if (render3d === rc.SVG) {
      // SVG rendering
      const r = createSVGRenderer(false, 0, doc);
      r.jsroot_dom = doc.createElementNS(nsSVG, 'svg');
      promise = Promise.resolve(r);
   } else if (isNodeJs()) {
      // try to use WebGL inside node.js - need to create headless context
      promise = import('canvas').then(node_canvas => {
         args.canvas = node_canvas.default.createCanvas(width, height);
         args.canvas.addEventListener = () => {}; // dummy
         args.canvas.removeEventListener = () => {}; // dummy
         args.canvas.style = {};
         return internals._node_gl || import('gl');
      }).then(node_gl => {
         internals._node_gl = node_gl;
         const gl = node_gl?.default(width, height, { preserveDrawingBuffer: true });
         if (!gl) throw Error('Fail to create headless-gl');
         args.context = gl;
         gl.canvas = args.canvas;

         const r = new THREE.WebGLRenderer(args);
         r.jsroot_output = new THREE.WebGLRenderTarget(width, height);
         r.setRenderTarget(r.jsroot_output);
         r.jsroot_dom = doc.createElementNS(nsSVG, 'image');
         return r;
      }).catch(() => {
         console.log('gl module is not available - fallback to SVGRenderer');
         render3d = rc.SVG;
         const r = createSVGRenderer(false, 0, doc);
         r.jsroot_dom = doc.createElementNS(nsSVG, 'svg');
         return r;
      });
   } else if (render3d === rc.WebGL) {
      // interactive WebGL Rendering
      promise = Promise.resolve(new THREE.WebGLRenderer(args));
   } else {
      // rendering with WebGL directly into svg image
      const r = new THREE.WebGLRenderer(args);
      r.jsroot_dom = doc.createElementNS(nsSVG, 'image');
      promise = Promise.resolve(r);
   }

   return promise.then(renderer => {
      if (!renderer.jsroot_dom)
         renderer.jsroot_dom = renderer.domElement;
      else
         renderer.jsroot_custom_dom = true;

      // res.renderer.setClearColor('#000000', 1);
      // res.renderer.setClearColor(0x0, 0);
      renderer.jsroot_render3d = render3d;

      // which format used to convert into images
      renderer.jsroot_image_format = 'png';

      renderer.originalSetSize = renderer.setSize;

      // apply size to dom element
      renderer.setSize = function(w, h, updateStyle) {
         if (this.jsroot_custom_dom) {
            this.jsroot_dom.setAttribute('width', w);
            this.jsroot_dom.setAttribute('height', h);
         }

         this.originalSetSize(w, h, updateStyle);
      };

      renderer.setSize(width, height);

      return renderer;
   });
}


/** @summary Cleanup created renderer object
  * @private */
function cleanupRender3D(renderer) {
   if (!renderer) return;

   if (isNodeJs()) {
      const ctxt = isFunc(renderer.getContext) ? renderer.getContext() : null,
            ext = ctxt?.getExtension('STACKGL_destroy_context');
      if (isFunc(ext?.destroy))
          ext.destroy();
   } else {
      // suppress warnings in Chrome about lost webgl context, not required in firefox
      if (browser.isChrome && isFunc(renderer.forceContextLoss))
         renderer.forceContextLoss();

      if (isFunc(renderer.dispose))
         renderer.dispose();
   }
}

/** @summary Cleanup previous renderings before doing next one
  * @desc used together with SVG
  * @private */
function beforeRender3D(renderer) {
   if (isFunc(renderer.clearHTML))
      renderer.clearHTML();
}

/** @summary Post-process result of rendering
  * @desc used together with SVG or node.js image rendering
  * @private */
function afterRender3D(renderer) {
   const rc = constants.Render3D;

   if (renderer.jsroot_render3d === rc.WebGL)
      return;

   if (renderer.jsroot_render3d === rc.SVG) {
      // case of SVGRenderer
      renderer.fillTargetSVG(renderer.jsroot_dom);
   } else if (isNodeJs()) {
      // this is WebGL rendering in node.js
      const canvas = renderer.domElement,
            context = canvas.getContext('2d'),
            pixels = new Uint8Array(4 * canvas.width * canvas.height);

      renderer.readRenderTargetPixels(renderer.jsroot_output, 0, 0, canvas.width, canvas.height, pixels);

      // small code to flip Y scale
      let indx1 = 0, indx2 = (canvas.height - 1) * 4 * canvas.width, k, d;
      while (indx1 < indx2) {
         for (k = 0; k < 4 * canvas.width; ++k) {
            d = pixels[indx1 + k]; pixels[indx1 + k] = pixels[indx2 + k]; pixels[indx2 + k] = d;
         }
         indx1 += 4 * canvas.width;
         indx2 -= 4 * canvas.width;
      }

      const imageData = context.createImageData(canvas.width, canvas.height);
      imageData.data.set(pixels);
      context.putImageData(imageData, 0, 0);

      const format = 'image/' + renderer.jsroot_image_format,
            dataUrl = canvas.toDataURL(format);

      renderer.jsroot_dom.setAttribute('href', dataUrl);
   } else {
      const dataUrl = renderer.domElement.toDataURL('image/' + renderer.jsroot_image_format);
      renderer.jsroot_dom.setAttribute('href', dataUrl);
   }
}

// ========================================================================================================

/**
 * @summary Tooltip handler for 3D drawings
 *
 * @private
 */

class TooltipFor3D {

   /** @summary constructor
     * @param {object} dom - DOM element
     * @param {object} canvas - canvas for 3D rendering */
   constructor(prnt, canvas) {
      this.tt = null;
      this.cont = null;
      this.lastlbl = '';
      this.parent = prnt || getDocument().body;
      this.canvas = canvas; // we need canvas to recalculate mouse events
      this.abspos = !prnt;
      this.scale = 1;
   }

   /** @summary check parent */
   checkParent(prnt) {
      if (prnt && (this.parent !== prnt)) {
         this.hide();
         this.parent = prnt;
      }
   }

   /** @summary set scaling factor */
   setScale(v) {
      this.scale = v;
   }

   /** @summary extract position from event
     * @desc can be used to process it later when event is gone */
   extract_pos(e) {
      if (isObject(e) && (e.u !== undefined) && (e.l !== undefined))
         return e;
      const res = { u: 0, l: 0 };
      if (this.abspos) {
         res.l = e.pageX;
         res.u = e.pageY;
      } else {
         res.l = e.offsetX;
         res.u = e.offsetY;
      }
      res.l /= this.scale;
      res.u /= this.scale;
      return res;
   }

   /** @summary Method used to define position of next tooltip
     * @desc event is delivered from canvas,
     * but position should be calculated relative to the element where tooltip is placed */
   pos(e) {
      if (!this.tt)
         return;

      const pos = this.extract_pos(e);
      if (!this.abspos) {
         const rect1 = this.parent.getBoundingClientRect(),
               rect2 = this.canvas.getBoundingClientRect();

         if ((rect1.left !== undefined) && (rect2.left!== undefined))
            pos.l += (rect2.left-rect1.left);

         if ((rect1.top !== undefined) && (rect2.top!== undefined))
            pos.u += rect2.top-rect1.top;

         if (pos.l + this.tt.offsetWidth + 3 >= this.parent.offsetWidth)
            pos.l = this.parent.offsetWidth - this.tt.offsetWidth - 3;

         if (pos.u + this.tt.offsetHeight + 15 >= this.parent.offsetHeight)
            pos.u = this.parent.offsetHeight - this.tt.offsetHeight - 15;

         // one should find parent with non-static position,
         // all absolute coordinates calculated relative to such node
         let abs_parent = this.parent;
         while (abs_parent) {
            const style = getComputedStyle(abs_parent);
            if (!style || (style.position !== 'static')) break;
            if (!abs_parent.parentNode || (abs_parent.parentNode.nodeType !== 1)) break;
            abs_parent = abs_parent.parentNode;
         }

         if (abs_parent && (abs_parent !== this.parent)) {
            const rect0 = abs_parent.getBoundingClientRect();
            pos.l += (rect1.left - rect0.left);
            pos.u += (rect1.top - rect0.top);
         }
      }

      this.tt.style.top = `${pos.u+15}px`;
      this.tt.style.left = `${pos.l+3}px`;
   }

   /** @summary Show tooltip */
   show(v /* , mouse_pos, status_func */) {
      if (!v)
         return this.hide();

      if (isObject(v) && (v.lines || v.line)) {
         if (v.only_status)
            return this.hide();

         if (v.line)
            v = v.line;
         else {
            let res = v.lines[0];
            for (let n = 1; n < v.lines.length; ++n)
               res += '<br/>' + v.lines[n];
            v = res;
         }
      }

      if (this.tt === null) {
         const doc = getDocument();
         this.tt = doc.createElement('div');
         this.tt.setAttribute('style', 'opacity: 1; filter: alpha(opacity=1); position: absolute; display: block; overflow: hidden; z-index: 101;');
         this.cont = doc.createElement('div');
         this.cont.setAttribute('style', 'display: block; padding: 2px 12px 3px 7px; margin-left: 5px; font-size: 11px; background: #777; color: #fff;');
         this.tt.appendChild(this.cont);
         this.parent.appendChild(this.tt);
      }

      if (this.lastlbl !== v) {
         this.cont.innerHTML = v;
         this.lastlbl = v;
         this.tt.style.width = 'auto'; // let it be automatically resizing...
      }
   }

   /** @summary Hide tooltip */
   hide() {
      if (this.tt !== null)
         this.parent.removeChild(this.tt);

      this.tt = null;
      this.lastlbl = '';
   }

} // class TooltipFor3D

/** @summary Create OrbitControls for painter
  * @private */
function createOrbitControl(painter, camera, scene, renderer, lookat) {
   const enable_zoom = settings.Zooming && settings.ZoomMouse,
         enable_select = isFunc(painter.processMouseClick);

   let control = null;

   function control_mousedown(evnt) {
      if (!control) return;

      // function used to hide some events from orbit control and redirect them to zooming rect
      if (control.mouse_zoom_mesh) {
         evnt.stopImmediatePropagation();
         evnt.stopPropagation();
         return;
      }

      // only left-button is considered
      if ((evnt.button!==undefined) && (evnt.button !== 0)) return;
      if ((evnt.buttons!==undefined) && (evnt.buttons !== 1)) return;

      if (control.enable_zoom) {
         control.mouse_zoom_mesh = control.detectZoomMesh(evnt);
         if (control.mouse_zoom_mesh) {
            // just block orbit control
            evnt.stopImmediatePropagation();
            evnt.stopPropagation();
            return;
         }
      }

      if (control.enable_select)
         control.mouse_select_pnt = control.getMousePos(evnt, {});
   }

   function control_mouseup(evnt) {
      if (!control) return;

      if (control.mouse_zoom_mesh && control.mouse_zoom_mesh.point2 && control.painter.get3dZoomCoord) {
         let kind = control.mouse_zoom_mesh.object.zoom,
             pos1 = control.painter.get3dZoomCoord(control.mouse_zoom_mesh.point, kind),
             pos2 = control.painter.get3dZoomCoord(control.mouse_zoom_mesh.point2, kind);

         if (pos1 > pos2)
            [pos1, pos2] = [pos2, pos1];

         if ((kind === 'z') && control.mouse_zoom_mesh.object.use_y_for_z) kind = 'y';

         // try to zoom
         if ((pos1 < pos2) && control.painter.zoom(kind, pos1, pos2))
            control.mouse_zoom_mesh = null;
      }

      // if selection was drawn, it should be removed and picture rendered again
      if (control.enable_zoom)
         control.removeZoomMesh();

      // only left-button is considered
      // if ((evnt.button!==undefined) && (evnt.button !== 0)) return;
      // if ((evnt.buttons!==undefined) && (evnt.buttons !== 1)) return;

      if (control.enable_select && control.mouse_select_pnt) {
         const pnt = control.getMousePos(evnt, {}),
               same_pnt = (pnt.x === control.mouse_select_pnt.x) && (pnt.y === control.mouse_select_pnt.y);
         delete control.mouse_select_pnt;

         if (same_pnt) {
            const intersects = control.getMouseIntersects(pnt);
            control.painter.processMouseClick(pnt, intersects, evnt);
         }
      }
   }

   function render3DFired(_painter) {
      if (_painter?.renderer === undefined)
         return false;
      // when timeout configured, object is prepared for rendering
      return _painter.render_tmout !== undefined;
   }

   function control_mousewheel(evnt) {
      if (!control) return;

      // try to handle zoom extra
      if (render3DFired(control.painter) || control.mouse_zoom_mesh) {
         evnt.preventDefault();
         evnt.stopPropagation();
         evnt.stopImmediatePropagation();
         return; // already fired redraw, do not react on the mouse wheel
      }

      const intersect = control.detectZoomMesh(evnt);
      if (!intersect) return;

      evnt.preventDefault();
      evnt.stopPropagation();
      evnt.stopImmediatePropagation();

      if (isFunc(control.painter?.analyzeMouseWheelEvent)) {
         let kind = intersect.object.zoom,
             position = intersect.point[kind];
         const item = { name: kind, ignore: false };

         // z changes from 0..2*size_z3d, others -size_x3d..+size_x3d
         switch (kind) {
            case 'x': position = (position + control.painter.size_x3d)/2/control.painter.size_x3d; break;
            case 'y': position = (position + control.painter.size_y3d)/2/control.painter.size_y3d; break;
            case 'z': position = position/2/control.painter.size_z3d; break;
         }

         control.painter.analyzeMouseWheelEvent(evnt, item, position, false);

         if ((kind === 'z') && intersect.object.use_y_for_z) kind = 'y';

         control.painter.zoom(kind, item.min, item.max);
      }
   }

   // assign own handler before creating OrbitControl

   if (settings.Zooming && settings.ZoomWheel)
      renderer.domElement.addEventListener('wheel', control_mousewheel);

   if (enable_zoom || enable_select) {
      renderer.domElement.addEventListener('pointerdown', control_mousedown);
      renderer.domElement.addEventListener('pointerup', control_mouseup);
   }

   control = new THREE.OrbitControls(camera, renderer.domElement);

   control.enableDamping = false;
   control.dampingFactor = 1.0;
   control.enableZoom = true;
   control.enableKeys = settings.HandleKeys;

   if (lookat) {
      control.target.copy(lookat);
      control.target0.copy(lookat);
      control.update();
   }

   control.tooltip = new TooltipFor3D(painter.selectDom().node(), renderer.domElement);

   control.painter = painter;
   control.camera = camera;
   control.scene = scene;
   control.renderer = renderer;
   control.raycaster = new THREE.Raycaster();
   control.raycaster.params.Line.threshold = 10;
   control.raycaster.params.Points.threshold = 5;
   control.mouse_zoom_mesh = null; // zoom mesh, currently used in the zooming
   control.block_ctxt = false; // require to block context menu command appearing after control ends, required in chrome which inject contextmenu when key released
   control.block_mousemove = false; // when true, tooltip or cursor will not react on mouse move
   control.cursor_changed = false;
   control.control_changed = false;
   control.control_active = false;
   control.mouse_ctxt = { x: 0, y: 0, on: false };
   control.enable_zoom = enable_zoom;
   control.enable_select = enable_select;

   control.cleanup = function() {
      if (settings.Zooming && settings.ZoomWheel)
         this.domElement.removeEventListener('wheel', control_mousewheel);
      if (this.enable_zoom || this.enable_select) {
         this.domElement.removeEventListener('pointerdown', control_mousedown);
         this.domElement.removeEventListener('pointerup', control_mouseup);
      }

      this.domElement.removeEventListener('click', this.lstn_click);
      this.domElement.removeEventListener('dblclick', this.lstn_dblclick);
      this.domElement.removeEventListener('contextmenu', this.lstn_contextmenu);
      this.domElement.removeEventListener('mousemove', this.lstn_mousemove);
      this.domElement.removeEventListener('mouseleave', this.lstn_mouseleave);

      this.dispose(); // this is from OrbitControl itself

      this.tooltip.hide();
      delete this.tooltip;
      delete this.painter;
      delete this.camera;
      delete this.scene;
      delete this.renderer;
      delete this.raycaster;
      delete this.mouse_zoom_mesh;
   };

   control.hideTooltip = function() {
      this.tooltip.hide();
   };

   control.getMousePos = function(evnt, mouse) {
      mouse.x = ('offsetX' in evnt) ? evnt.offsetX : evnt.layerX;
      mouse.y = ('offsetY' in evnt) ? evnt.offsetY : evnt.layerY;
      mouse.clientX = evnt.clientX;
      mouse.clientY = evnt.clientY;
      return mouse;
   };

   control.getOriginDirectionIntersects = function(origin, direction) {
      this.raycaster.set(origin, direction);
      let intersects = this.raycaster.intersectObjects(this.scene.children, true);
      // painter may want to filter intersects
      if (isFunc(this.painter.filterIntersects))
         intersects = this.painter.filterIntersects(intersects);
      return intersects;
   };

   control.getMouseIntersects = function(mouse) {
      // domElement gives correct coordinate with canvas render, but isn't always right for webgl renderer
      if (!this.renderer)
         return [];

      const sz = (this.renderer instanceof THREE.SVGRenderer) ? this.renderer.domElement : this.renderer.getSize(new THREE.Vector2()),
            pnt = { x: mouse.x / sz.width * 2 - 1, y: -mouse.y / sz.height * 2 + 1 };

      this.camera.updateMatrix();
      this.camera.updateMatrixWorld();
      this.raycaster.setFromCamera(pnt, this.camera);
      let intersects = this.raycaster.intersectObjects(this.scene.children, true);

      // painter may want to filter intersects
      if (isFunc(this.painter.filterIntersects))
         intersects = this.painter.filterIntersects(intersects);

      return intersects;
   };

   control.detectZoomMesh = function(evnt) {
      const mouse = this.getMousePos(evnt, {}),
            intersects = this.getMouseIntersects(mouse);
      if (intersects) {
         for (let n = 0; n < intersects.length; ++n) {
            if (intersects[n].object.zoom && !intersects[n].object.zoom_disabled)
               return intersects[n];
         }
      }

      return null;
   };

   control.getInfoAtMousePosition = function(mouse_pos) {
      const intersects = this.getMouseIntersects(mouse_pos);
      let tip = null, _painter = null;

      for (let i = 0; i < intersects.length; ++i) {
         if (intersects[i].object.tooltip) {
            tip = intersects[i].object.tooltip(intersects[i]);
            _painter = intersects[i].object.painter;
            break;
         }
      }

      if (tip && _painter) {
         return { obj: _painter.getObject(),
                  name: _painter.getObject().fName,
                  bin: tip.bin, cont: tip.value,
                  binx: tip.ix, biny: tip.iy, binz: tip.iz,
                  grx: (tip.x1+tip.x2)/2, gry: (tip.y1+tip.y2)/2, grz: (tip.z1+tip.z2)/2 };
      }
   };

   control.processDblClick = function(evnt) {
      // first check if zoom mesh clicked
      const zoom_intersect = this.detectZoomMesh(evnt);
      if (zoom_intersect && this.painter) {
         this.painter.unzoom(zoom_intersect.object.use_y_for_z ? 'y' : zoom_intersect.object.zoom);
         return;
      }

      // then check if double-click handler assigned
      const fp = this.painter?.getFramePainter();
      if (isFunc(fp?._dblclick_handler)) {
         const info = this.getInfoAtMousePosition(this.getMousePos(evnt, {}));
         if (info) {
            fp._dblclick_handler(info);
            return;
         }
      }

      this.reset();
   };

   control.changeEvent = function() {
      this.mouse_ctxt.on = false; // disable context menu if any changes where done by orbit control
      this.painter.render3D(0);
      this.control_changed = true;
   };

   control.startEvent = function() {
      this.control_active = true;
      this.block_ctxt = false;
      this.mouse_ctxt.on = false;

      this.tooltip.hide();

      // do not reset here, problem of events sequence in orbitcontrol
      // it issue change/start/stop event when do zooming
      // control.control_changed = false;
   };

   control.endEvent = function() {
      this.control_active = false;
      if (this.mouse_ctxt.on) {
         this.mouse_ctxt.on = false;
         this.contextMenu(this.mouse_ctxt, this.getMouseIntersects(this.mouse_ctxt));
      } /* else if (this.control_changed) {
         // react on camera change when required
      } */
      this.control_changed = false;
   };

   control.mainProcessContextMenu = function(evnt) {
      evnt.preventDefault();
      this.getMousePos(evnt, this.mouse_ctxt);
      if (this.control_active)
         this.mouse_ctxt.on = true;
      else if (this.block_ctxt)
         this.block_ctxt = false;
      else
         this.contextMenu(this.mouse_ctxt, this.getMouseIntersects(this.mouse_ctxt));
   };

   control.contextMenu = function(/* pos, intersects */) {
      // do nothing, function called when context menu want to be activated
   };

   control.setTooltipEnabled = function(on) {
      this.block_mousemove = !on;
      if (on === false) {
         this.tooltip.hide();
         this.removeZoomMesh();
      }
   };

   control.removeZoomMesh = function() {
      if (this.mouse_zoom_mesh?.object.showSelection())
         this.painter.render3D();
      this.mouse_zoom_mesh = null; // in any case clear mesh, enable orbit control again
   };

   control.mainProcessMouseMove = function(evnt) {
      if (!this.painter)
         return; // protect when cleanup

      if (this.control_active && evnt.buttons && (evnt.buttons & 2))
         this.block_ctxt = true; // if right button in control was active, block next context menu

      if (this.control_active || this.block_mousemove || !isFunc(this.processMouseMove))
         return;

      if (this.mouse_zoom_mesh) {
         // when working with zoom mesh, need special handling

         const zoom2 = this.detectZoomMesh(evnt),
               pnt2 = (zoom2?.object === this.mouse_zoom_mesh.object) ? zoom2.point : this.mouse_zoom_mesh.object.globalIntersect(this.raycaster);

         if (pnt2) this.mouse_zoom_mesh.point2 = pnt2;

         if (pnt2 && this.painter.enable_highlight) {
            if (this.mouse_zoom_mesh.object.showSelection(this.mouse_zoom_mesh.point, pnt2))
               this.painter.render3D(0);
         }

         this.tooltip.hide();
         return;
      }

      evnt.preventDefault();

      // extract mouse position
      this.tmout_mouse = this.getMousePos(evnt, {});
      this.tmout_ttpos = this.tooltip?.extract_pos(evnt);

      if (this.tmout_handle) {
         clearTimeout(this.tmout_handle);
         delete this.tmout_handle;
      }

      if (!this.mouse_tmout)
         this.delayedProcessMouseMove();
      else
         this.tmout_handle = setTimeout(() => this.delayedProcessMouseMove(), this.mouse_tmout);
   };

   control.delayedProcessMouseMove = function() {
      // remove handle - allow to trigger new timeout
      delete this.tmout_handle;
      if (!this.painter) return; // protect when cleanup

      const mouse = this.tmout_mouse,
            intersects = this.getMouseIntersects(mouse),
            tip = this.processMouseMove(intersects);

      if (tip) {
         let name = '', title = '', info = '';
         const coord = mouse ? mouse.x.toFixed(0) + ',' + mouse.y.toFixed(0) : '';
         if (isStr(tip))
            info = tip;
         else {
            name = tip.name;
            title = tip.title;
            if (tip.line)
               info = tip.line;
            else if (tip.lines) {
               info = tip.lines.slice(1).join(' ');
               name = tip.lines[0];
            }
         }
         this.painter.showObjectStatus(name, title, info, coord);
      }

      this.cursor_changed = false;
      if (tip && this.painter?.isTooltipAllowed()) {
         this.tooltip.checkParent(this.painter.selectDom().node());
         this.tooltip.show(tip, mouse);
         this.tooltip.pos(this.tmout_ttpos);
      } else {
         this.tooltip.hide();
         if (intersects) {
            for (let n = 0; n < intersects.length; ++n) {
               if (intersects[n].object.zoom && !intersects[n].object.zoom_disabled)
                  this.cursor_changed = true;
            }
         }
      }

      getDocument().body.style.cursor = this.cursor_changed ? 'pointer' : 'auto';
   };

   control.mainProcessMouseLeave = function() {
      if (!this.painter)
         return; // protect when cleanup

      // do not enter main event at all
      if (this.tmout_handle) {
         clearTimeout(this.tmout_handle);
         delete this.tmout_handle;
      }
      this.tooltip.hide();
      if (isFunc(this.processMouseLeave))
         this.processMouseLeave();
      if (this.cursor_changed) {
         getDocument().body.style.cursor = 'auto';
         this.cursor_changed = false;
      }
   };

   control.mainProcessDblClick = function(evnt) {
      // suppress simple click handler if double click detected
      if (this.single_click_tm) {
         clearTimeout(this.single_click_tm);
         delete this.single_click_tm;
      }
      this.processDblClick(evnt);
   };

   control.processClick = function(mouse_pos, kind) {
      delete this.single_click_tm;

      if (kind === 1) {
         const fp = this.painter?.getFramePainter();
         if (isFunc(fp?._click_handler)) {
            const info = this.getInfoAtMousePosition(mouse_pos);
            if (info) {
               fp._click_handler(info);
               return;
            }
         }
      }

      // method assigned in the Eve7 and used for object selection
      if ((kind === 2) && isFunc(this.processSingleClick)) {
         const intersects = this.getMouseIntersects(mouse_pos);
         this.processSingleClick(intersects);
      }

      if (kind === 3) {
         const intersects = this.getMouseIntersects(mouse_pos);
         let objpainter = null;
         for (let i = 0; !objpainter && (i < intersects.length); ++i) {
            const obj3d = intersects[i].object;
            objpainter = obj3d.painter || obj3d.parent?.painter; // check one top level
         }
         if (objpainter) {
            // while axis painter not directly appears in the list of primitives, pad and canvas take from frame
            const padp = this.painter?.getPadPainter(),
                  canvp = this.painter?.getCanvPainter();
            canvp?.producePadEvent('select', padp, objpainter);
         }
      }
   };

   control.lstn_click = function(evnt) {
      // ignore right-mouse click
      if (evnt.detail === 2)
         return;

      if (this.single_click_tm) {
         clearTimeout(this.single_click_tm);
         delete this.single_click_tm;
      }

      let kind = 0;
      if (isFunc(this.painter?.getFramePainter()?._click_handler))
         kind = 1;  // user click handler
      else if (this.processSingleClick && this.painter?.options?.mouse_click)
         kind = 2;  // eve7 click handler
      else if (this.painter?.getCanvPainter())
         kind = 3;  // select event for GED

      // if normal event, set longer timeout waiting if double click not detected
      if (kind)
         this.single_click_tm = setTimeout(this.processClick.bind(this, this.getMousePos(evnt, {}), kind), 300);
   }.bind(control);

   control.addEventListener('change', () => control.changeEvent());
   control.addEventListener('start', () => control.startEvent());
   control.addEventListener('end', () => control.endEvent());

   control.lstn_contextmenu = evnt => control.mainProcessContextMenu(evnt);
   control.lstn_dblclick = evnt => control.mainProcessDblClick(evnt);
   control.lstn_mousemove = evnt => control.mainProcessMouseMove(evnt);
   control.lstn_mouseleave = () => control.mainProcessMouseLeave();

   renderer.domElement.addEventListener('click', control.lstn_click);
   renderer.domElement.addEventListener('dblclick', control.lstn_dblclick);
   renderer.domElement.addEventListener('contextmenu', control.lstn_contextmenu);
   renderer.domElement.addEventListener('mousemove', control.lstn_mousemove);
   renderer.domElement.addEventListener('mouseleave', control.lstn_mouseleave);

   return control;
}

/** @summary Method cleanup three.js object as much as possible.
  * @desc Simplify JS engine to remove it from memory
  * @private */
function disposeThreejsObject(obj, only_childs) {
   if (!obj)
      return;

   if (obj.children) {
      for (let i = 0; i < obj.children.length; i++)
         disposeThreejsObject(obj.children[i]);
   }

   if (only_childs) {
      obj.children = [];
      return;
   }

   obj.children = undefined;

   if (obj.geometry) {
      obj.geometry.dispose();
      obj.geometry = undefined;
   }
   if (obj.material) {
      if (obj.material.map) {
         obj.material.map.dispose();
         obj.material.map = undefined;
      }
      obj.material.dispose();
      obj.material = undefined;
   }

   // cleanup jsroot fields to simplify browser cleanup job
   delete obj.painter;
   delete obj.bins_index;
   delete obj.tooltip;
   delete obj.stack; // used in geom painter
   delete obj.drawn_highlight; // special highlight object
}


/** @summary Create LineSegments mesh (or only geometry)
  * @desc If required, calculates lineDistance attribute for dashed geometries
  * @private */
function createLineSegments(arr, material, index = undefined, only_geometry = false) {
   const geom = new THREE.BufferGeometry();

   geom.setAttribute('position', arr instanceof Float32Array ? new THREE.BufferAttribute(arr, 3) : new THREE.Float32BufferAttribute(arr, 3));
   if (index) geom.setIndex(new THREE.BufferAttribute(index, 1));

   if (material.isLineDashedMaterial) {
      const v1 = new THREE.Vector3(),
            v2 = new THREE.Vector3();
      let d = 0, distances;

      if (index) {
         distances = new Float32Array(index.length);
         for (let n = 0; n < index.length; n += 2) {
            const i1 = index[n], i2 = index[n+1];
            v1.set(arr[i1], arr[i1+1], arr[i1+2]);
            v2.set(arr[i2], arr[i2+1], arr[i2+2]);
            distances[n] = d;
            d += v2.distanceTo(v1);
            distances[n+1] = d;
         }
      } else {
         distances = new Float32Array(arr.length/3);
         for (let n = 0; n < arr.length; n += 6) {
            v1.set(arr[n], arr[n+1], arr[n+2]);
            v2.set(arr[n+3], arr[n+4], arr[n+5]);
            distances[n/3] = d;
            d += v2.distanceTo(v1);
            distances[n/3+1] = d;
         }
      }
      geom.setAttribute('lineDistance', new THREE.BufferAttribute(distances, 1));
   }

   return only_geometry ? geom : new THREE.LineSegments(geom, material);
}

/** @summary Help structures for calculating Box mesh
  * @private */
const Box3D = {
    Vertices: [new THREE.Vector3(1, 1, 1), new THREE.Vector3(1, 1, 0),
               new THREE.Vector3(1, 0, 1), new THREE.Vector3(1, 0, 0),
               new THREE.Vector3(0, 1, 0), new THREE.Vector3(0, 1, 1),
               new THREE.Vector3(0, 0, 0), new THREE.Vector3(0, 0, 1)],
    Indexes: [0, 2, 1, 2, 3, 1, 4, 6, 5, 6, 7, 5, 4, 5, 1, 5, 0, 1,
              7, 6, 2, 6, 3, 2, 5, 7, 0, 7, 2, 0, 1, 3, 4, 3, 6, 4],
    Normals: [1, 0, 0, -1, 0, 0, 0, 1, 0, 0, -1, 0, 0, 0, 1, 0, 0, -1],
    Segments: [0, 2, 2, 7, 7, 5, 5, 0, 1, 3, 3, 6, 6, 4, 4, 1, 1, 0, 3, 2, 6, 7, 4, 5],  // segments addresses Vertices
    Crosses: [0, 7, 2, 5, 0, 3, 1, 2, 7, 3, 2, 6, 5, 6, 4, 7, 5, 1, 0, 4, 3, 4, 1, 6], // addresses Vertices
    MeshSegments: undefined
};

// these segments address vertices from the mesh, we can use positions from box mesh
Box3D.MeshSegments = (function() {
   const arr = new Int32Array(Box3D.Segments.length);

   for (let n = 0; n < arr.length; ++n) {
      for (let k = 0; k < Box3D.Indexes.length; ++k) {
         if (Box3D.Segments[n] === Box3D.Indexes[k]) {
            arr[n] = k;
            break;
         }
      }
   }
   return arr;
})();


/**
 * @summary Abstract interactive control interface for 3D objects
 *
 * @abstract
 * @private
 */

class InteractiveControl {

   cleanup() {}
   extractIndex(/* intersect */) {}
   setSelected(/* col, indx */) {}
   setHighlight(/* col, indx */) {}
   checkHighlightIndex(/* indx */) {}

} // class InteractiveControl


/**
 * @summary Special class to control highliht and selection of single points, used in geo painter
 *
 * @private
 */

class PointsControl extends InteractiveControl {

   /** @summary constructor
     * @param {object} mesh - draw object */
   constructor(mesh) {
      super();
      this.mesh = mesh;
   }

   /** @summary cleanup object */
   cleanup() {
      if (!this.mesh) return;
      delete this.mesh.is_selected;
      this.createSpecial(null);
      delete this.mesh;
   }

   /** @summary extract intersect index */
   extractIndex(intersect) {
      return intersect && intersect.index!==undefined ? intersect.index : undefined;
   }

   /** @summary set selection */
   setSelected(col, indx) {
      const m = this.mesh;
      if ((m.select_col === col) && (m.select_indx === indx)) {
         col = null;
         indx = undefined;
      }
      m.select_col = col;
      m.select_indx = indx;
      this.createSpecial(col, indx);
      return true;
   }

   /** @summary set highlight */
   setHighlight(col, indx) {
      const m = this.mesh;
      m.h_index = indx;
      if (col)
         this.createSpecial(col, indx);
      else
         this.createSpecial(m.select_col, m.select_indx);
      return true;
   }

   /** @summary create special object */
   createSpecial(color, index) {
      const m = this.mesh;
      if (!color) {
         if (m.js_special) {
            m.remove(m.js_special);
            disposeThreejsObject(m.js_special);
            delete m.js_special;
         }
         return;
      }

      if (!m.js_special) {
         const geom = new THREE.BufferGeometry();
         geom.setAttribute('position', m.geometry.getAttribute('position'));
         const material = new THREE.PointsMaterial({ size: m.material.size*2, color });
         material.sizeAttenuation = m.material.sizeAttenuation;

         m.js_special = new THREE.Points(geom, material);
         m.js_special.jsroot_special = true; // special object, exclude from intersections
         m.add(m.js_special);
      }

      m.js_special.material.color = new THREE.Color(color);
      if (index !== undefined) m.js_special.geometry.setDrawRange(index, 1);
   }

} // class PointsControl


/**
 * @summary Class for creation of 3D points
 *
 * @private
 */

class PointsCreator {

   /** @summary constructor
     * @param {number} number - number of points
     * @param {boolean} [iswebgl] - if WebGL is used
     * @param {number} [scale] - scale factor */
   constructor(number, iswebgl = true, scale = 1) {
      this.webgl = iswebgl;
      this.scale = scale || 1;

      this.pos = new Float32Array(number*3);
      this.geom = new THREE.BufferGeometry();
      this.geom.setAttribute('position', new THREE.BufferAttribute(this.pos, 3));
      this.indx = 0;
   }

   /** @summary Add point */
   addPoint(x, y, z) {
      this.pos[this.indx] = x;
      this.pos[this.indx+1] = y;
      this.pos[this.indx+2] = z;
      this.indx += 3;
   }

   /** @summary Create points */
   createPoints(args) {
      if (!isObject(args))
         args = { color: args };
      if (!args.color)
         args.color = 'black';

      let k = 1;

      // special dots
      if (!args.style)
         k = 1.1;
      else if (args.style === 1)
         k = 0.3;
      else if (args.style === 6)
         k = 0.5;
      else if (args.style === 7)
         k = 0.7;

      const makePoints = texture => {
         const material_args = { size: 3*this.scale*k };
         if (texture) {
            material_args.map = texture;
            material_args.transparent = true;
         } else
            material_args.color = args.color || 'black';

         const pnts = new THREE.Points(this.geom, new THREE.PointsMaterial(material_args));
         pnts.nvertex = 1;
         return pnts;
      };

      // this is plain creation of points, no need for texture loading

      if (k !== 1) {
         const res = makePoints();
         return this.noPromise ? res : Promise.resolve(res);
      }

      const handler = new TAttMarkerHandler({ style: args.style, color: args.color, size: 7 }),
            w = handler.fill ? 1 : 7,
            imgdata = `<svg width="64" height="64" xmlns="${nsSVG}">` +
                      `<path d="${handler.create(32, 32)}" style="stroke: ${handler.getStrokeColor()}; stroke-width: ${w}; fill: ${args.fill ?? handler.getFillColor()}"></path>`+
                      '</svg>',
            dataUrl = prSVG + (isNodeJs() ? imgdata : encodeURIComponent(imgdata));
      let promise;

      if (isNodeJs()) {
         promise = import('canvas').then(handle => handle.default.loadImage(dataUrl).then(img => {
               const canvas = handle.default.createCanvas(64, 64),
                     ctx = canvas.getContext('2d');
               ctx.drawImage(img, 0, 0, 64, 64);
               return new THREE.CanvasTexture(canvas);
            }));
      } else if (this.noPromise) {
         // only for v6 support
         return makePoints(new THREE.TextureLoader().load(dataUrl));
      } else {
         promise = new Promise((resolveFunc, rejectFunc) => {
            const loader = new THREE.TextureLoader();
            loader.load(dataUrl, res => resolveFunc(res), undefined, () => rejectFunc(Error(`Fail to load ${dataUrl}`)));
         });
      }

      return promise.then(makePoints);
   }

} // class PointsCreator


/** @summary Create material for 3D line
  * @desc Takes into account dashed properties
  * @private */
function create3DLineMaterial(painter, arg, is_v7 = false) {
   if (!painter || !arg) return null;

   let color, lstyle, lwidth;
   if (isStr(arg) || is_v7) {
      color = painter.v7EvalColor(arg+'color', 'black');
      lstyle = parseInt(painter.v7EvalAttr(arg+'style', 0));
      lwidth = parseInt(painter.v7EvalAttr(arg+'width', 1));
   } else {
      color = painter.getColor(arg.fLineColor);
      lstyle = arg.fLineStyle;
      lwidth = arg.fLineWidth;
   }

   const style = lstyle ? getSvgLineStyle(lstyle) : '',
         dash = style ? style.split(',') : [],
         material = (dash && dash.length >= 2)
            ? new THREE.LineDashedMaterial({ color, dashSize: parseInt(dash[0]), gapSize: parseInt(dash[1]) })
            : new THREE.LineBasicMaterial({ color });

   if (lwidth && (lwidth > 1)) material.linewidth = lwidth;

   return material;
}

/** @summary Create plain text geometry
  * @private */
function createTextGeometry(lbl, size) {
   const geom_args = { font: getHelveticaFont(), size, height: 0, curveSegments: 5 };
   if (THREE.REVISION > 162)
      geom_args.depth = 0;
   else
      geom_args.height = 0;

   return new THREE.TextGeometry(lbl, geom_args);
}


export { THREE, importThreeJs, assign3DHandler, disposeThreejsObject, createOrbitControl,
         createLineSegments, create3DLineMaterial, Box3D, getMaterialArgs,
         createRender3D, beforeRender3D, afterRender3D, getRender3DKind, cleanupRender3D,
         getHelveticaFont, createTextGeometry, InteractiveControl, PointsControl, PointsCreator, createSVGRenderer };
