/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*                  This file is part of the program and library             */
/*         SCIP --- Solving Constraint Integer Programs                      */
/*                                                                           */
/*  Copyright (c) 2002-2026 Zuse Institute Berlin (ZIB)                      */
/*                                                                           */
/*  Licensed under the Apache License, Version 2.0 (the "License");          */
/*  you may not use this file except in compliance with the License.         */
/*  You may obtain a copy of the License at                                  */
/*                                                                           */
/*      http://www.apache.org/licenses/LICENSE-2.0                           */
/*                                                                           */
/*  Unless required by applicable law or agreed to in writing, software      */
/*  distributed under the License is distributed on an "AS IS" BASIS,        */
/*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. */
/*  See the License for the specific language governing permissions and      */
/*  limitations under the License.                                           */
/*                                                                           */
/*  You should have received a copy of the Apache-2.0 license                */
/*  along with SCIP; see the file LICENSE. If not visit scipopt.org.         */
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**@file   prop_symmetry.h
 * @ingroup PROPAGATORS
 * @brief  propagator for symmetry handling
 * @author Marc Pfetsch
 * @author Thomas Rehn
 * @author Christopher Hojny
 */

/*---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2*/

#ifndef __SCIP_PROP_SYMMETRY_H_
#define __SCIP_PROP_SYMMETRY_H_

#include "scip/scip.h"
#include "symmetry/type_symmetry.h"

#ifdef __cplusplus
extern "C" {
#endif

/** include symmetry propagator */
SCIP_EXPORT
SCIP_RETCODE SCIPincludePropSymmetry(
   SCIP*                 scip                /**< SCIP data structure */
   );

/** return currently available symmetry group information */
SCIP_EXPORT
SCIP_RETCODE SCIPgetSymmetry(
   SCIP*                 scip,               /**< SCIP data structure */
   int*                  npermvars,          /**< pointer to store number of variables for permutations */
   SCIP_VAR***           permvars,           /**< pointer to store variables on which permutations act */
   SCIP_HASHMAP**        permvarmap,         /**< pointer to store hash map of permvars (or NULL) */
   int*                  nperms,             /**< pointer to store number of permutations */
   int***                perms,              /**< pointer to store permutation generators as (nperms x npermvars) matrix (or NULL)*/
   int***                permstrans,         /**< pointer to store permutation generators as (npermvars x nperms) matrix (or NULL)*/
   SCIP_Real*            log10groupsize,     /**< pointer to store log10 of group size (or NULL) */
   SCIP_Bool*            binvaraffected,     /**< pointer to store whether binary variables are affected */
   int**                 components,         /**< pointer to store components of symmetry group (or NULL) */
   int**                 componentbegins,    /**< pointer to store begin positions of components in components array (or NULL) */
   int**                 vartocomponent,     /**< pointer to store assignment from variable to its component (or NULL) */
   int*                  ncomponents         /**< pointer to store number of components (or NULL) */
   );

/** return number of the symmetry group's generators */
SCIP_EXPORT
int SCIPgetSymmetryNGenerators(
   SCIP*                 scip                /**< SCIP data structure */
   );

/** displays generators of symmetry group, if available */
SCIP_EXPORT
SCIP_RETCODE SCIPdisplaySymmetryGenerators(
   SCIP*                 scip,               /**< SCIP data structure */
   SCIP_PROP*            prop                /**< symmetry propagator or NULL */
   );

/** creates new operator node type (used for symmetry detection) and returns its representation
 *
 *  If the operator node already exists, the function terminates with SCIP_INVALIDDATA.
 */
SCIP_EXPORT
SCIP_RETCODE SCIPcreateSymOpNodeType(
   SCIP*                 scip,               /**< SCIP pointer */
   const char*           opnodename,         /**< name of new operator node type */
   int*                  nodetype            /**< pointer to store the new node type */
   );

/** returns representation of an operator node type.
 *
 *  If the node type does not already exist, a new node type will be created.
 */
SCIP_EXPORT
SCIP_RETCODE SCIPgetSymOpNodeType(
   SCIP*                 scip,               /**< SCIP pointer */
   const char*           opnodename,         /**< name of new operator node type */
   int*                  nodetype            /**< pointer to store the node type */
   );

#ifdef __cplusplus
}
#endif

#endif
