﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/controltower/ControlTowerRequest.h>
#include <aws/controltower/ControlTower_EXPORTS.h>
#include <aws/controltower/model/EnabledControlFilter.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ControlTower {
namespace Model {

/**
 */
class ListEnabledControlsRequest : public ControlTowerRequest {
 public:
  AWS_CONTROLTOWER_API ListEnabledControlsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListEnabledControls"; }

  AWS_CONTROLTOWER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ARN of the organizational unit. For information on how to find the
   * <code>targetIdentifier</code>, see <a
   * href="https://docs.aws.amazon.com/controltower/latest/APIReference/Welcome.html">the
   * overview page</a>.</p>
   */
  inline const Aws::String& GetTargetIdentifier() const { return m_targetIdentifier; }
  inline bool TargetIdentifierHasBeenSet() const { return m_targetIdentifierHasBeenSet; }
  template <typename TargetIdentifierT = Aws::String>
  void SetTargetIdentifier(TargetIdentifierT&& value) {
    m_targetIdentifierHasBeenSet = true;
    m_targetIdentifier = std::forward<TargetIdentifierT>(value);
  }
  template <typename TargetIdentifierT = Aws::String>
  ListEnabledControlsRequest& WithTargetIdentifier(TargetIdentifierT&& value) {
    SetTargetIdentifier(std::forward<TargetIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to continue the list from a previous API call with the same
   * parameters.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListEnabledControlsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>How many results to return per API call.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListEnabledControlsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An input filter for the <code>ListEnabledControls</code> API that lets you
   * select the types of control operations to view.</p>
   */
  inline const EnabledControlFilter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = EnabledControlFilter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = EnabledControlFilter>
  ListEnabledControlsRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A boolean value that determines whether to include enabled controls from
   * child organizational units in the response.</p>
   */
  inline bool GetIncludeChildren() const { return m_includeChildren; }
  inline bool IncludeChildrenHasBeenSet() const { return m_includeChildrenHasBeenSet; }
  inline void SetIncludeChildren(bool value) {
    m_includeChildrenHasBeenSet = true;
    m_includeChildren = value;
  }
  inline ListEnabledControlsRequest& WithIncludeChildren(bool value) {
    SetIncludeChildren(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_targetIdentifier;
  bool m_targetIdentifierHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  EnabledControlFilter m_filter;
  bool m_filterHasBeenSet = false;

  bool m_includeChildren{false};
  bool m_includeChildrenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ControlTower
}  // namespace Aws
