﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/directory-service-data/DirectoryServiceDataRequest.h>
#include <aws/directory-service-data/DirectoryServiceData_EXPORTS.h>
#include <aws/directory-service-data/model/AttributeValue.h>
#include <aws/directory-service-data/model/GroupScope.h>
#include <aws/directory-service-data/model/GroupType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DirectoryServiceData {
namespace Model {

/**
 */
class CreateGroupRequest : public DirectoryServiceDataRequest {
 public:
  AWS_DIRECTORYSERVICEDATA_API CreateGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateGroup"; }

  AWS_DIRECTORYSERVICEDATA_API Aws::String SerializePayload() const override;

  AWS_DIRECTORYSERVICEDATA_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> A unique and case-sensitive identifier that you provide to make sure the
   * idempotency of the request, so multiple identical calls have the same effect as
   * one single call. </p> <p> A client token is valid for 8 hours after the first
   * request that uses it completes. After 8 hours, any request with the same client
   * token is treated as a new request. If the request succeeds, any future uses of
   * that token will be idempotent for another 8 hours. </p> <p> If you submit a
   * request with the same client token but change one of the other parameters within
   * the 8-hour idempotency window, Directory Service Data returns an
   * <code>ConflictException</code>. </p>  <p> This parameter is optional when
   * using the CLI or SDK. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateGroupRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The identifier (ID) of the directory that's associated with the group. </p>
   */
  inline const Aws::String& GetDirectoryId() const { return m_directoryId; }
  inline bool DirectoryIdHasBeenSet() const { return m_directoryIdHasBeenSet; }
  template <typename DirectoryIdT = Aws::String>
  void SetDirectoryId(DirectoryIdT&& value) {
    m_directoryIdHasBeenSet = true;
    m_directoryId = std::forward<DirectoryIdT>(value);
  }
  template <typename DirectoryIdT = Aws::String>
  CreateGroupRequest& WithDirectoryId(DirectoryIdT&& value) {
    SetDirectoryId(std::forward<DirectoryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The scope of the AD group. For details, see <a
   * href="https://learn.microsoft.com/en-us/windows-server/identity/ad-ds/manage/understand-security-groups#group-scope">Active
   * Directory security group scope</a>. </p>
   */
  inline GroupScope GetGroupScope() const { return m_groupScope; }
  inline bool GroupScopeHasBeenSet() const { return m_groupScopeHasBeenSet; }
  inline void SetGroupScope(GroupScope value) {
    m_groupScopeHasBeenSet = true;
    m_groupScope = value;
  }
  inline CreateGroupRequest& WithGroupScope(GroupScope value) {
    SetGroupScope(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The AD group type. For details, see <a
   * href="https://learn.microsoft.com/en-us/windows-server/identity/ad-ds/manage/understand-security-groups#how-active-directory-security-groups-work">Active
   * Directory security group type</a>.</p>
   */
  inline GroupType GetGroupType() const { return m_groupType; }
  inline bool GroupTypeHasBeenSet() const { return m_groupTypeHasBeenSet; }
  inline void SetGroupType(GroupType value) {
    m_groupTypeHasBeenSet = true;
    m_groupType = value;
  }
  inline CreateGroupRequest& WithGroupType(GroupType value) {
    SetGroupType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> An expression that defines one or more attributes with the data type and
   * value of each attribute. </p>
   */
  inline const Aws::Map<Aws::String, AttributeValue>& GetOtherAttributes() const { return m_otherAttributes; }
  inline bool OtherAttributesHasBeenSet() const { return m_otherAttributesHasBeenSet; }
  template <typename OtherAttributesT = Aws::Map<Aws::String, AttributeValue>>
  void SetOtherAttributes(OtherAttributesT&& value) {
    m_otherAttributesHasBeenSet = true;
    m_otherAttributes = std::forward<OtherAttributesT>(value);
  }
  template <typename OtherAttributesT = Aws::Map<Aws::String, AttributeValue>>
  CreateGroupRequest& WithOtherAttributes(OtherAttributesT&& value) {
    SetOtherAttributes(std::forward<OtherAttributesT>(value));
    return *this;
  }
  template <typename OtherAttributesKeyT = Aws::String, typename OtherAttributesValueT = AttributeValue>
  CreateGroupRequest& AddOtherAttributes(OtherAttributesKeyT&& key, OtherAttributesValueT&& value) {
    m_otherAttributesHasBeenSet = true;
    m_otherAttributes.emplace(std::forward<OtherAttributesKeyT>(key), std::forward<OtherAttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the group. </p>
   */
  inline const Aws::String& GetSAMAccountName() const { return m_sAMAccountName; }
  inline bool SAMAccountNameHasBeenSet() const { return m_sAMAccountNameHasBeenSet; }
  template <typename SAMAccountNameT = Aws::String>
  void SetSAMAccountName(SAMAccountNameT&& value) {
    m_sAMAccountNameHasBeenSet = true;
    m_sAMAccountName = std::forward<SAMAccountNameT>(value);
  }
  template <typename SAMAccountNameT = Aws::String>
  CreateGroupRequest& WithSAMAccountName(SAMAccountNameT&& value) {
    SetSAMAccountName(std::forward<SAMAccountNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_directoryId;
  bool m_directoryIdHasBeenSet = false;

  GroupScope m_groupScope{GroupScope::NOT_SET};
  bool m_groupScopeHasBeenSet = false;

  GroupType m_groupType{GroupType::NOT_SET};
  bool m_groupTypeHasBeenSet = false;

  Aws::Map<Aws::String, AttributeValue> m_otherAttributes;
  bool m_otherAttributesHasBeenSet = false;

  Aws::String m_sAMAccountName;
  bool m_sAMAccountNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectoryServiceData
}  // namespace Aws
