﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dynamodb/DynamoDBRequest.h>
#include <aws/dynamodb/DynamoDB_EXPORTS.h>
#include <aws/dynamodb/model/AttributeValue.h>
#include <aws/dynamodb/model/ReturnConsumedCapacity.h>

#include <utility>

namespace Aws {
namespace DynamoDB {
namespace Model {

/**
 * <p>Represents the input of a <code>GetItem</code> operation.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/GetItemInput">AWS
 * API Reference</a></p>
 */
class GetItemRequest : public DynamoDBRequest {
 public:
  AWS_DYNAMODB_API GetItemRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetItem"; }

  AWS_DYNAMODB_API Aws::String SerializePayload() const override;

  AWS_DYNAMODB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_DYNAMODB_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The name of the table containing the requested item. You can also provide the
   * Amazon Resource Name (ARN) of the table in this parameter.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  GetItemRequest& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of attribute names to <code>AttributeValue</code> objects, representing
   * the primary key of the item to retrieve.</p> <p>For the primary key, you must
   * provide all of the attributes. For example, with a simple primary key, you only
   * need to provide a value for the partition key. For a composite primary key, you
   * must provide values for both the partition key and the sort key.</p>
   */
  inline const Aws::Map<Aws::String, AttributeValue>& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::Map<Aws::String, AttributeValue>>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::Map<Aws::String, AttributeValue>>
  GetItemRequest& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  template <typename KeyKeyT = Aws::String, typename KeyValueT = AttributeValue>
  GetItemRequest& AddKey(KeyKeyT&& key, KeyValueT&& value) {
    m_keyHasBeenSet = true;
    m_key.emplace(std::forward<KeyKeyT>(key), std::forward<KeyValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This is a legacy parameter. Use <code>ProjectionExpression</code> instead.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.AttributesToGet.html">AttributesToGet</a>
   * in the <i>Amazon DynamoDB Developer Guide</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAttributesToGet() const { return m_attributesToGet; }
  inline bool AttributesToGetHasBeenSet() const { return m_attributesToGetHasBeenSet; }
  template <typename AttributesToGetT = Aws::Vector<Aws::String>>
  void SetAttributesToGet(AttributesToGetT&& value) {
    m_attributesToGetHasBeenSet = true;
    m_attributesToGet = std::forward<AttributesToGetT>(value);
  }
  template <typename AttributesToGetT = Aws::Vector<Aws::String>>
  GetItemRequest& WithAttributesToGet(AttributesToGetT&& value) {
    SetAttributesToGet(std::forward<AttributesToGetT>(value));
    return *this;
  }
  template <typename AttributesToGetT = Aws::String>
  GetItemRequest& AddAttributesToGet(AttributesToGetT&& value) {
    m_attributesToGetHasBeenSet = true;
    m_attributesToGet.emplace_back(std::forward<AttributesToGetT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines the read consistency model: If set to <code>true</code>, then the
   * operation uses strongly consistent reads; otherwise, the operation uses
   * eventually consistent reads.</p>
   */
  inline bool GetConsistentRead() const { return m_consistentRead; }
  inline bool ConsistentReadHasBeenSet() const { return m_consistentReadHasBeenSet; }
  inline void SetConsistentRead(bool value) {
    m_consistentReadHasBeenSet = true;
    m_consistentRead = value;
  }
  inline GetItemRequest& WithConsistentRead(bool value) {
    SetConsistentRead(value);
    return *this;
  }
  ///@}

  ///@{

  inline ReturnConsumedCapacity GetReturnConsumedCapacity() const { return m_returnConsumedCapacity; }
  inline bool ReturnConsumedCapacityHasBeenSet() const { return m_returnConsumedCapacityHasBeenSet; }
  inline void SetReturnConsumedCapacity(ReturnConsumedCapacity value) {
    m_returnConsumedCapacityHasBeenSet = true;
    m_returnConsumedCapacity = value;
  }
  inline GetItemRequest& WithReturnConsumedCapacity(ReturnConsumedCapacity value) {
    SetReturnConsumedCapacity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string that identifies one or more attributes to retrieve from the table.
   * These attributes can include scalars, sets, or elements of a JSON document. The
   * attributes in the expression must be separated by commas.</p> <p>If no attribute
   * names are specified, then all attributes are returned. If any of the requested
   * attributes are not found, they do not appear in the result.</p> <p>For more
   * information, see <a
   * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html">Specifying
   * Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.</p>
   */
  inline const Aws::String& GetProjectionExpression() const { return m_projectionExpression; }
  inline bool ProjectionExpressionHasBeenSet() const { return m_projectionExpressionHasBeenSet; }
  template <typename ProjectionExpressionT = Aws::String>
  void SetProjectionExpression(ProjectionExpressionT&& value) {
    m_projectionExpressionHasBeenSet = true;
    m_projectionExpression = std::forward<ProjectionExpressionT>(value);
  }
  template <typename ProjectionExpressionT = Aws::String>
  GetItemRequest& WithProjectionExpression(ProjectionExpressionT&& value) {
    SetProjectionExpression(std::forward<ProjectionExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more substitution tokens for attribute names in an expression. The
   * following are some use cases for using
   * <code>ExpressionAttributeNames</code>:</p> <ul> <li> <p>To access an attribute
   * whose name conflicts with a DynamoDB reserved word.</p> </li> <li> <p>To create
   * a placeholder for repeating occurrences of an attribute name in an
   * expression.</p> </li> <li> <p>To prevent special characters in an attribute name
   * from being misinterpreted in an expression.</p> </li> </ul> <p>Use the <b>#</b>
   * character in an expression to dereference an attribute name. For example,
   * consider the following attribute name:</p> <ul> <li> <p> <code>Percentile</code>
   * </p> </li> </ul> <p>The name of this attribute conflicts with a reserved word,
   * so it cannot be used directly in an expression. (For the complete list of
   * reserved words, see <a
   * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ReservedWords.html">Reserved
   * Words</a> in the <i>Amazon DynamoDB Developer Guide</i>). To work around this,
   * you could specify the following for <code>ExpressionAttributeNames</code>:</p>
   * <ul> <li> <p> <code>{"#P":"Percentile"}</code> </p> </li> </ul> <p>You could
   * then use this substitution in an expression, as in this example:</p> <ul> <li>
   * <p> <code>#P = :val</code> </p> </li> </ul>  <p>Tokens that begin with the
   * <b>:</b> character are <i>expression attribute values</i>, which are
   * placeholders for the actual value at runtime.</p>  <p>For more
   * information on expression attribute names, see <a
   * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.AccessingItemAttributes.html">Specifying
   * Item Attributes</a> in the <i>Amazon DynamoDB Developer Guide</i>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetExpressionAttributeNames() const { return m_expressionAttributeNames; }
  inline bool ExpressionAttributeNamesHasBeenSet() const { return m_expressionAttributeNamesHasBeenSet; }
  template <typename ExpressionAttributeNamesT = Aws::Map<Aws::String, Aws::String>>
  void SetExpressionAttributeNames(ExpressionAttributeNamesT&& value) {
    m_expressionAttributeNamesHasBeenSet = true;
    m_expressionAttributeNames = std::forward<ExpressionAttributeNamesT>(value);
  }
  template <typename ExpressionAttributeNamesT = Aws::Map<Aws::String, Aws::String>>
  GetItemRequest& WithExpressionAttributeNames(ExpressionAttributeNamesT&& value) {
    SetExpressionAttributeNames(std::forward<ExpressionAttributeNamesT>(value));
    return *this;
  }
  template <typename ExpressionAttributeNamesKeyT = Aws::String, typename ExpressionAttributeNamesValueT = Aws::String>
  GetItemRequest& AddExpressionAttributeNames(ExpressionAttributeNamesKeyT&& key, ExpressionAttributeNamesValueT&& value) {
    m_expressionAttributeNamesHasBeenSet = true;
    m_expressionAttributeNames.emplace(std::forward<ExpressionAttributeNamesKeyT>(key),
                                       std::forward<ExpressionAttributeNamesValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_tableName;
  bool m_tableNameHasBeenSet = false;

  Aws::Map<Aws::String, AttributeValue> m_key;
  bool m_keyHasBeenSet = false;

  Aws::Vector<Aws::String> m_attributesToGet;
  bool m_attributesToGetHasBeenSet = false;

  bool m_consistentRead{false};
  bool m_consistentReadHasBeenSet = false;

  ReturnConsumedCapacity m_returnConsumedCapacity{ReturnConsumedCapacity::NOT_SET};
  bool m_returnConsumedCapacityHasBeenSet = false;

  Aws::String m_projectionExpression;
  bool m_projectionExpressionHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_expressionAttributeNames;
  bool m_expressionAttributeNamesHasBeenSet = false;
};

}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
