﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/emr-containers/EMRContainers_EXPORTS.h>
#include <aws/emr-containers/model/Certificate.h>
#include <aws/emr-containers/model/ConfigurationOverrides.h>
#include <aws/emr-containers/model/EndpointState.h>
#include <aws/emr-containers/model/FailureReason.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace EMRContainers {
namespace Model {

/**
 * <p>This entity represents the endpoint that is managed by Amazon EMR on
 * EKS.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/Endpoint">AWS
 * API Reference</a></p>
 */
class Endpoint {
 public:
  AWS_EMRCONTAINERS_API Endpoint() = default;
  AWS_EMRCONTAINERS_API Endpoint(Aws::Utils::Json::JsonView jsonValue);
  AWS_EMRCONTAINERS_API Endpoint& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_EMRCONTAINERS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID of the endpoint.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  Endpoint& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the endpoint.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Endpoint& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the endpoint.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Endpoint& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the endpoint's virtual cluster.</p>
   */
  inline const Aws::String& GetVirtualClusterId() const { return m_virtualClusterId; }
  inline bool VirtualClusterIdHasBeenSet() const { return m_virtualClusterIdHasBeenSet; }
  template <typename VirtualClusterIdT = Aws::String>
  void SetVirtualClusterId(VirtualClusterIdT&& value) {
    m_virtualClusterIdHasBeenSet = true;
    m_virtualClusterId = std::forward<VirtualClusterIdT>(value);
  }
  template <typename VirtualClusterIdT = Aws::String>
  Endpoint& WithVirtualClusterId(VirtualClusterIdT&& value) {
    SetVirtualClusterId(std::forward<VirtualClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the endpoint.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  Endpoint& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the endpoint.</p>
   */
  inline EndpointState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(EndpointState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline Endpoint& WithState(EndpointState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The EMR release version to be used for the endpoint.</p>
   */
  inline const Aws::String& GetReleaseLabel() const { return m_releaseLabel; }
  inline bool ReleaseLabelHasBeenSet() const { return m_releaseLabelHasBeenSet; }
  template <typename ReleaseLabelT = Aws::String>
  void SetReleaseLabel(ReleaseLabelT&& value) {
    m_releaseLabelHasBeenSet = true;
    m_releaseLabel = std::forward<ReleaseLabelT>(value);
  }
  template <typename ReleaseLabelT = Aws::String>
  Endpoint& WithReleaseLabel(ReleaseLabelT&& value) {
    SetReleaseLabel(std::forward<ReleaseLabelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The execution role ARN of the endpoint.</p>
   */
  inline const Aws::String& GetExecutionRoleArn() const { return m_executionRoleArn; }
  inline bool ExecutionRoleArnHasBeenSet() const { return m_executionRoleArnHasBeenSet; }
  template <typename ExecutionRoleArnT = Aws::String>
  void SetExecutionRoleArn(ExecutionRoleArnT&& value) {
    m_executionRoleArnHasBeenSet = true;
    m_executionRoleArn = std::forward<ExecutionRoleArnT>(value);
  }
  template <typename ExecutionRoleArnT = Aws::String>
  Endpoint& WithExecutionRoleArn(ExecutionRoleArnT&& value) {
    SetExecutionRoleArn(std::forward<ExecutionRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The certificate generated by emr control plane on customer behalf to secure
   * the managed endpoint.</p>
   */
  inline const Certificate& GetCertificateAuthority() const { return m_certificateAuthority; }
  inline bool CertificateAuthorityHasBeenSet() const { return m_certificateAuthorityHasBeenSet; }
  template <typename CertificateAuthorityT = Certificate>
  void SetCertificateAuthority(CertificateAuthorityT&& value) {
    m_certificateAuthorityHasBeenSet = true;
    m_certificateAuthority = std::forward<CertificateAuthorityT>(value);
  }
  template <typename CertificateAuthorityT = Certificate>
  Endpoint& WithCertificateAuthority(CertificateAuthorityT&& value) {
    SetCertificateAuthority(std::forward<CertificateAuthorityT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration settings that are used to override existing configurations
   * for endpoints.</p>
   */
  inline const ConfigurationOverrides& GetConfigurationOverrides() const { return m_configurationOverrides; }
  inline bool ConfigurationOverridesHasBeenSet() const { return m_configurationOverridesHasBeenSet; }
  template <typename ConfigurationOverridesT = ConfigurationOverrides>
  void SetConfigurationOverrides(ConfigurationOverridesT&& value) {
    m_configurationOverridesHasBeenSet = true;
    m_configurationOverrides = std::forward<ConfigurationOverridesT>(value);
  }
  template <typename ConfigurationOverridesT = ConfigurationOverrides>
  Endpoint& WithConfigurationOverrides(ConfigurationOverridesT&& value) {
    SetConfigurationOverrides(std::forward<ConfigurationOverridesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The server URL of the endpoint.</p>
   */
  inline const Aws::String& GetServerUrl() const { return m_serverUrl; }
  inline bool ServerUrlHasBeenSet() const { return m_serverUrlHasBeenSet; }
  template <typename ServerUrlT = Aws::String>
  void SetServerUrl(ServerUrlT&& value) {
    m_serverUrlHasBeenSet = true;
    m_serverUrl = std::forward<ServerUrlT>(value);
  }
  template <typename ServerUrlT = Aws::String>
  Endpoint& WithServerUrl(ServerUrlT&& value) {
    SetServerUrl(std::forward<ServerUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the endpoint was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  Endpoint& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The security group configuration of the endpoint. </p>
   */
  inline const Aws::String& GetSecurityGroup() const { return m_securityGroup; }
  inline bool SecurityGroupHasBeenSet() const { return m_securityGroupHasBeenSet; }
  template <typename SecurityGroupT = Aws::String>
  void SetSecurityGroup(SecurityGroupT&& value) {
    m_securityGroupHasBeenSet = true;
    m_securityGroup = std::forward<SecurityGroupT>(value);
  }
  template <typename SecurityGroupT = Aws::String>
  Endpoint& WithSecurityGroup(SecurityGroupT&& value) {
    SetSecurityGroup(std::forward<SecurityGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subnet IDs of the endpoint. </p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnetIds() const { return m_subnetIds; }
  inline bool SubnetIdsHasBeenSet() const { return m_subnetIdsHasBeenSet; }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  void SetSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds = std::forward<SubnetIdsT>(value);
  }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  Endpoint& WithSubnetIds(SubnetIdsT&& value) {
    SetSubnetIds(std::forward<SubnetIdsT>(value));
    return *this;
  }
  template <typename SubnetIdsT = Aws::String>
  Endpoint& AddSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds.emplace_back(std::forward<SubnetIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Additional details of the endpoint state. </p>
   */
  inline const Aws::String& GetStateDetails() const { return m_stateDetails; }
  inline bool StateDetailsHasBeenSet() const { return m_stateDetailsHasBeenSet; }
  template <typename StateDetailsT = Aws::String>
  void SetStateDetails(StateDetailsT&& value) {
    m_stateDetailsHasBeenSet = true;
    m_stateDetails = std::forward<StateDetailsT>(value);
  }
  template <typename StateDetailsT = Aws::String>
  Endpoint& WithStateDetails(StateDetailsT&& value) {
    SetStateDetails(std::forward<StateDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The reasons why the endpoint has failed. </p>
   */
  inline FailureReason GetFailureReason() const { return m_failureReason; }
  inline bool FailureReasonHasBeenSet() const { return m_failureReasonHasBeenSet; }
  inline void SetFailureReason(FailureReason value) {
    m_failureReasonHasBeenSet = true;
    m_failureReason = value;
  }
  inline Endpoint& WithFailureReason(FailureReason value) {
    SetFailureReason(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags of the endpoint. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Endpoint& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Endpoint& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;
  bool m_idHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::String m_virtualClusterId;
  bool m_virtualClusterIdHasBeenSet = false;

  Aws::String m_type;
  bool m_typeHasBeenSet = false;

  EndpointState m_state{EndpointState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Aws::String m_releaseLabel;
  bool m_releaseLabelHasBeenSet = false;

  Aws::String m_executionRoleArn;
  bool m_executionRoleArnHasBeenSet = false;

  Certificate m_certificateAuthority;
  bool m_certificateAuthorityHasBeenSet = false;

  ConfigurationOverrides m_configurationOverrides;
  bool m_configurationOverridesHasBeenSet = false;

  Aws::String m_serverUrl;
  bool m_serverUrlHasBeenSet = false;

  Aws::Utils::DateTime m_createdAt{};
  bool m_createdAtHasBeenSet = false;

  Aws::String m_securityGroup;
  bool m_securityGroupHasBeenSet = false;

  Aws::Vector<Aws::String> m_subnetIds;
  bool m_subnetIdsHasBeenSet = false;

  Aws::String m_stateDetails;
  bool m_stateDetailsHasBeenSet = false;

  FailureReason m_failureReason{FailureReason::NOT_SET};
  bool m_failureReasonHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMRContainers
}  // namespace Aws
