﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWiseRequest.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IoTSiteWise {
namespace Model {

/**
 */
class DescribeAssetCompositeModelRequest : public IoTSiteWiseRequest {
 public:
  AWS_IOTSITEWISE_API DescribeAssetCompositeModelRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeAssetCompositeModel"; }

  AWS_IOTSITEWISE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the asset. This can be either the actual ID in UUID format, or else
   * <code>externalId:</code> followed by the external ID, if it has one. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetId() const { return m_assetId; }
  inline bool AssetIdHasBeenSet() const { return m_assetIdHasBeenSet; }
  template <typename AssetIdT = Aws::String>
  void SetAssetId(AssetIdT&& value) {
    m_assetIdHasBeenSet = true;
    m_assetId = std::forward<AssetIdT>(value);
  }
  template <typename AssetIdT = Aws::String>
  DescribeAssetCompositeModelRequest& WithAssetId(AssetIdT&& value) {
    SetAssetId(std::forward<AssetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a composite model on this asset. This can be either the actual ID
   * in UUID format, or else <code>externalId:</code> followed by the external ID, if
   * it has one. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetCompositeModelId() const { return m_assetCompositeModelId; }
  inline bool AssetCompositeModelIdHasBeenSet() const { return m_assetCompositeModelIdHasBeenSet; }
  template <typename AssetCompositeModelIdT = Aws::String>
  void SetAssetCompositeModelId(AssetCompositeModelIdT&& value) {
    m_assetCompositeModelIdHasBeenSet = true;
    m_assetCompositeModelId = std::forward<AssetCompositeModelIdT>(value);
  }
  template <typename AssetCompositeModelIdT = Aws::String>
  DescribeAssetCompositeModelRequest& WithAssetCompositeModelId(AssetCompositeModelIdT&& value) {
    SetAssetCompositeModelId(std::forward<AssetCompositeModelIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assetId;
  bool m_assetIdHasBeenSet = false;

  Aws::String m_assetCompositeModelId;
  bool m_assetCompositeModelIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
