﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lexv2-models/LexModelsV2Request.h>
#include <aws/lexv2-models/LexModelsV2_EXPORTS.h>
#include <aws/lexv2-models/model/SlotFilter.h>
#include <aws/lexv2-models/model/SlotSortBy.h>

#include <utility>

namespace Aws {
namespace LexModelsV2 {
namespace Model {

/**
 */
class ListSlotsRequest : public LexModelsV2Request {
 public:
  AWS_LEXMODELSV2_API ListSlotsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSlots"; }

  AWS_LEXMODELSV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the bot that contains the slot.</p>
   */
  inline const Aws::String& GetBotId() const { return m_botId; }
  inline bool BotIdHasBeenSet() const { return m_botIdHasBeenSet; }
  template <typename BotIdT = Aws::String>
  void SetBotId(BotIdT&& value) {
    m_botIdHasBeenSet = true;
    m_botId = std::forward<BotIdT>(value);
  }
  template <typename BotIdT = Aws::String>
  ListSlotsRequest& WithBotId(BotIdT&& value) {
    SetBotId(std::forward<BotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the bot that contains the slot.</p>
   */
  inline const Aws::String& GetBotVersion() const { return m_botVersion; }
  inline bool BotVersionHasBeenSet() const { return m_botVersionHasBeenSet; }
  template <typename BotVersionT = Aws::String>
  void SetBotVersion(BotVersionT&& value) {
    m_botVersionHasBeenSet = true;
    m_botVersion = std::forward<BotVersionT>(value);
  }
  template <typename BotVersionT = Aws::String>
  ListSlotsRequest& WithBotVersion(BotVersionT&& value) {
    SetBotVersion(std::forward<BotVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the language and locale of the slots to list. The string
   * must match one of the supported locales. For more information, see <a
   * href="https://docs.aws.amazon.com/lexv2/latest/dg/how-languages.html">Supported
   * languages</a>.</p>
   */
  inline const Aws::String& GetLocaleId() const { return m_localeId; }
  inline bool LocaleIdHasBeenSet() const { return m_localeIdHasBeenSet; }
  template <typename LocaleIdT = Aws::String>
  void SetLocaleId(LocaleIdT&& value) {
    m_localeIdHasBeenSet = true;
    m_localeId = std::forward<LocaleIdT>(value);
  }
  template <typename LocaleIdT = Aws::String>
  ListSlotsRequest& WithLocaleId(LocaleIdT&& value) {
    SetLocaleId(std::forward<LocaleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the intent that contains the slot.</p>
   */
  inline const Aws::String& GetIntentId() const { return m_intentId; }
  inline bool IntentIdHasBeenSet() const { return m_intentIdHasBeenSet; }
  template <typename IntentIdT = Aws::String>
  void SetIntentId(IntentIdT&& value) {
    m_intentIdHasBeenSet = true;
    m_intentId = std::forward<IntentIdT>(value);
  }
  template <typename IntentIdT = Aws::String>
  ListSlotsRequest& WithIntentId(IntentIdT&& value) {
    SetIntentId(std::forward<IntentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines the sort order for the response from the <code>ListSlots</code>
   * operation. You can choose to sort by the slot name or last updated date in
   * either ascending or descending order.</p>
   */
  inline const SlotSortBy& GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  template <typename SortByT = SlotSortBy>
  void SetSortBy(SortByT&& value) {
    m_sortByHasBeenSet = true;
    m_sortBy = std::forward<SortByT>(value);
  }
  template <typename SortByT = SlotSortBy>
  ListSlotsRequest& WithSortBy(SortByT&& value) {
    SetSortBy(std::forward<SortByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the specification of a filter used to limit the slots in the
   * response to only those that match the filter specification. You can only specify
   * one filter and only one string to filter on.</p>
   */
  inline const Aws::Vector<SlotFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<SlotFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<SlotFilter>>
  ListSlotsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = SlotFilter>
  ListSlotsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of slots to return in each page of results. If there are
   * fewer results than the max page size, only the actual number of results are
   * returned.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSlotsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the response from the <code>ListSlots</code> operation contains more
   * results than specified in the <code>maxResults</code> parameter, a token is
   * returned in the response. Use that token in the <code>nextToken</code> parameter
   * to return the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSlotsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_botId;
  bool m_botIdHasBeenSet = false;

  Aws::String m_botVersion;
  bool m_botVersionHasBeenSet = false;

  Aws::String m_localeId;
  bool m_localeIdHasBeenSet = false;

  Aws::String m_intentId;
  bool m_intentIdHasBeenSet = false;

  SlotSortBy m_sortBy;
  bool m_sortByHasBeenSet = false;

  Aws::Vector<SlotFilter> m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexModelsV2
}  // namespace Aws
