﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/migration-hub-refactor-spaces/MigrationHubRefactorSpaces_EXPORTS.h>
#include <aws/migration-hub-refactor-spaces/model/RouteState.h>
#include <aws/migration-hub-refactor-spaces/model/RouteType.h>
#include <aws/migration-hub-refactor-spaces/model/UriPathRouteInput.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace MigrationHubRefactorSpaces {
namespace Model {
class CreateRouteResult {
 public:
  AWS_MIGRATIONHUBREFACTORSPACES_API CreateRouteResult() = default;
  AWS_MIGRATIONHUBREFACTORSPACES_API CreateRouteResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_MIGRATIONHUBREFACTORSPACES_API CreateRouteResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The ID of the application in which the route is created.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  CreateRouteResult& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the route. The format for this ARN is
   * <code>arn:aws:refactor-spaces:<i>region</i>:<i>account-id</i>:<i>resource-type/resource-id</i>
   * </code>. For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  CreateRouteResult& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID of the route creator.</p>
   */
  inline const Aws::String& GetCreatedByAccountId() const { return m_createdByAccountId; }
  template <typename CreatedByAccountIdT = Aws::String>
  void SetCreatedByAccountId(CreatedByAccountIdT&& value) {
    m_createdByAccountIdHasBeenSet = true;
    m_createdByAccountId = std::forward<CreatedByAccountIdT>(value);
  }
  template <typename CreatedByAccountIdT = Aws::String>
  CreateRouteResult& WithCreatedByAccountId(CreatedByAccountIdT&& value) {
    SetCreatedByAccountId(std::forward<CreatedByAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A timestamp that indicates when the route is created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTime() const { return m_createdTime; }
  template <typename CreatedTimeT = Aws::Utils::DateTime>
  void SetCreatedTime(CreatedTimeT&& value) {
    m_createdTimeHasBeenSet = true;
    m_createdTime = std::forward<CreatedTimeT>(value);
  }
  template <typename CreatedTimeT = Aws::Utils::DateTime>
  CreateRouteResult& WithCreatedTime(CreatedTimeT&& value) {
    SetCreatedTime(std::forward<CreatedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A timestamp that indicates when the route was last updated. </p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedTime() const { return m_lastUpdatedTime; }
  template <typename LastUpdatedTimeT = Aws::Utils::DateTime>
  void SetLastUpdatedTime(LastUpdatedTimeT&& value) {
    m_lastUpdatedTimeHasBeenSet = true;
    m_lastUpdatedTime = std::forward<LastUpdatedTimeT>(value);
  }
  template <typename LastUpdatedTimeT = Aws::Utils::DateTime>
  CreateRouteResult& WithLastUpdatedTime(LastUpdatedTimeT&& value) {
    SetLastUpdatedTime(std::forward<LastUpdatedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID of the route owner.</p>
   */
  inline const Aws::String& GetOwnerAccountId() const { return m_ownerAccountId; }
  template <typename OwnerAccountIdT = Aws::String>
  void SetOwnerAccountId(OwnerAccountIdT&& value) {
    m_ownerAccountIdHasBeenSet = true;
    m_ownerAccountId = std::forward<OwnerAccountIdT>(value);
  }
  template <typename OwnerAccountIdT = Aws::String>
  CreateRouteResult& WithOwnerAccountId(OwnerAccountIdT&& value) {
    SetOwnerAccountId(std::forward<OwnerAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the route.</p>
   */
  inline const Aws::String& GetRouteId() const { return m_routeId; }
  template <typename RouteIdT = Aws::String>
  void SetRouteId(RouteIdT&& value) {
    m_routeIdHasBeenSet = true;
    m_routeId = std::forward<RouteIdT>(value);
  }
  template <typename RouteIdT = Aws::String>
  CreateRouteResult& WithRouteId(RouteIdT&& value) {
    SetRouteId(std::forward<RouteIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The route type of the route.</p>
   */
  inline RouteType GetRouteType() const { return m_routeType; }
  inline void SetRouteType(RouteType value) {
    m_routeTypeHasBeenSet = true;
    m_routeType = value;
  }
  inline CreateRouteResult& WithRouteType(RouteType value) {
    SetRouteType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of service in which the route is created. Traffic that matches this
   * route is forwarded to this service.</p>
   */
  inline const Aws::String& GetServiceId() const { return m_serviceId; }
  template <typename ServiceIdT = Aws::String>
  void SetServiceId(ServiceIdT&& value) {
    m_serviceIdHasBeenSet = true;
    m_serviceId = std::forward<ServiceIdT>(value);
  }
  template <typename ServiceIdT = Aws::String>
  CreateRouteResult& WithServiceId(ServiceIdT&& value) {
    SetServiceId(std::forward<ServiceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current state of the route. Activation state only allows
   * <code>ACTIVE</code> or <code>INACTIVE</code> as user inputs. <code>FAILED</code>
   * is a route state that is system generated.</p>
   */
  inline RouteState GetState() const { return m_state; }
  inline void SetState(RouteState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline CreateRouteResult& WithState(RouteState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags assigned to the created route. A tag is a label that you assign to
   * an Amazon Web Services resource. Each tag consists of a key-value pair. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRouteResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRouteResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration for the URI path route type. </p>
   */
  inline const UriPathRouteInput& GetUriPathRoute() const { return m_uriPathRoute; }
  template <typename UriPathRouteT = UriPathRouteInput>
  void SetUriPathRoute(UriPathRouteT&& value) {
    m_uriPathRouteHasBeenSet = true;
    m_uriPathRoute = std::forward<UriPathRouteT>(value);
  }
  template <typename UriPathRouteT = UriPathRouteInput>
  CreateRouteResult& WithUriPathRoute(UriPathRouteT&& value) {
    SetUriPathRoute(std::forward<UriPathRouteT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateRouteResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;
  bool m_applicationIdHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::String m_createdByAccountId;
  bool m_createdByAccountIdHasBeenSet = false;

  Aws::Utils::DateTime m_createdTime{};
  bool m_createdTimeHasBeenSet = false;

  Aws::Utils::DateTime m_lastUpdatedTime{};
  bool m_lastUpdatedTimeHasBeenSet = false;

  Aws::String m_ownerAccountId;
  bool m_ownerAccountIdHasBeenSet = false;

  Aws::String m_routeId;
  bool m_routeIdHasBeenSet = false;

  RouteType m_routeType{RouteType::NOT_SET};
  bool m_routeTypeHasBeenSet = false;

  Aws::String m_serviceId;
  bool m_serviceIdHasBeenSet = false;

  RouteState m_state{RouteState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  UriPathRouteInput m_uriPathRoute;
  bool m_uriPathRouteHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHubRefactorSpaces
}  // namespace Aws
