﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/Tag.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/CreateHsmClientCertificateMessage">AWS
 * API Reference</a></p>
 */
class CreateHsmClientCertificateRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API CreateHsmClientCertificateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateHsmClientCertificate"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The identifier to be assigned to the new HSM client certificate that the
   * cluster will use to connect to the HSM to use the database encryption keys.</p>
   */
  inline const Aws::String& GetHsmClientCertificateIdentifier() const { return m_hsmClientCertificateIdentifier; }
  inline bool HsmClientCertificateIdentifierHasBeenSet() const { return m_hsmClientCertificateIdentifierHasBeenSet; }
  template <typename HsmClientCertificateIdentifierT = Aws::String>
  void SetHsmClientCertificateIdentifier(HsmClientCertificateIdentifierT&& value) {
    m_hsmClientCertificateIdentifierHasBeenSet = true;
    m_hsmClientCertificateIdentifier = std::forward<HsmClientCertificateIdentifierT>(value);
  }
  template <typename HsmClientCertificateIdentifierT = Aws::String>
  CreateHsmClientCertificateRequest& WithHsmClientCertificateIdentifier(HsmClientCertificateIdentifierT&& value) {
    SetHsmClientCertificateIdentifier(std::forward<HsmClientCertificateIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tag instances.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateHsmClientCertificateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateHsmClientCertificateRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_hsmClientCertificateIdentifier;
  bool m_hsmClientCertificateIdentifierHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
