﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/secretsmanager/SecretsManagerRequest.h>
#include <aws/secretsmanager/SecretsManager_EXPORTS.h>
#include <aws/secretsmanager/model/Filter.h>
#include <aws/secretsmanager/model/SortOrderType.h>

#include <utility>

namespace Aws {
namespace SecretsManager {
namespace Model {

/**
 */
class ListSecretsRequest : public SecretsManagerRequest {
 public:
  AWS_SECRETSMANAGER_API ListSecretsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSecrets"; }

  AWS_SECRETSMANAGER_API Aws::String SerializePayload() const override;

  AWS_SECRETSMANAGER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies whether to include secrets scheduled for deletion. By default,
   * secrets scheduled for deletion aren't included.</p>
   */
  inline bool GetIncludePlannedDeletion() const { return m_includePlannedDeletion; }
  inline bool IncludePlannedDeletionHasBeenSet() const { return m_includePlannedDeletionHasBeenSet; }
  inline void SetIncludePlannedDeletion(bool value) {
    m_includePlannedDeletionHasBeenSet = true;
    m_includePlannedDeletion = value;
  }
  inline ListSecretsRequest& WithIncludePlannedDeletion(bool value) {
    SetIncludePlannedDeletion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of results to include in the response.</p> <p>If there are more
   * results available, in the response, Secrets Manager includes
   * <code>NextToken</code>. To get the next results, call <code>ListSecrets</code>
   * again with the value from <code>NextToken</code>.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSecretsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that indicates where the output should continue from, if a previous
   * call did not show all results. To get the next results, call
   * <code>ListSecrets</code> again with this value.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSecretsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filters to apply to the list of secrets.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  ListSecretsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  ListSecretsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Secrets are listed by <code>CreatedDate</code>. </p>
   */
  inline SortOrderType GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrderType value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListSecretsRequest& WithSortOrder(SortOrderType value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}
 private:
  bool m_includePlannedDeletion{false};
  bool m_includePlannedDeletionHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  SortOrderType m_sortOrder{SortOrderType::NOT_SET};
  bool m_sortOrderHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecretsManager
}  // namespace Aws
