﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm-sap/SsmSapRequest.h>
#include <aws/ssm-sap/SsmSap_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SsmSap {
namespace Model {

/**
 */
class GetApplicationRequest : public SsmSapRequest {
 public:
  AWS_SSMSAP_API GetApplicationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetApplication"; }

  AWS_SSMSAP_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the application.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  GetApplicationRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the application. </p>
   */
  inline const Aws::String& GetApplicationArn() const { return m_applicationArn; }
  inline bool ApplicationArnHasBeenSet() const { return m_applicationArnHasBeenSet; }
  template <typename ApplicationArnT = Aws::String>
  void SetApplicationArn(ApplicationArnT&& value) {
    m_applicationArnHasBeenSet = true;
    m_applicationArn = std::forward<ApplicationArnT>(value);
  }
  template <typename ApplicationArnT = Aws::String>
  GetApplicationRequest& WithApplicationArn(ApplicationArnT&& value) {
    SetApplicationArn(std::forward<ApplicationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the application registry.</p>
   */
  inline const Aws::String& GetAppRegistryArn() const { return m_appRegistryArn; }
  inline bool AppRegistryArnHasBeenSet() const { return m_appRegistryArnHasBeenSet; }
  template <typename AppRegistryArnT = Aws::String>
  void SetAppRegistryArn(AppRegistryArnT&& value) {
    m_appRegistryArnHasBeenSet = true;
    m_appRegistryArn = std::forward<AppRegistryArnT>(value);
  }
  template <typename AppRegistryArnT = Aws::String>
  GetApplicationRequest& WithAppRegistryArn(AppRegistryArnT&& value) {
    SetAppRegistryArn(std::forward<AppRegistryArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;
  bool m_applicationIdHasBeenSet = false;

  Aws::String m_applicationArn;
  bool m_applicationArnHasBeenSet = false;

  Aws::String m_appRegistryArn;
  bool m_appRegistryArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace SsmSap
}  // namespace Aws
