﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class CreateRouteRequest : public EC2Request {
 public:
  AWS_EC2_API CreateRouteRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRoute"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of a prefix list used for the destination match.</p>
   */
  inline const Aws::String& GetDestinationPrefixListId() const { return m_destinationPrefixListId; }
  inline bool DestinationPrefixListIdHasBeenSet() const { return m_destinationPrefixListIdHasBeenSet; }
  template <typename DestinationPrefixListIdT = Aws::String>
  void SetDestinationPrefixListId(DestinationPrefixListIdT&& value) {
    m_destinationPrefixListIdHasBeenSet = true;
    m_destinationPrefixListId = std::forward<DestinationPrefixListIdT>(value);
  }
  template <typename DestinationPrefixListIdT = Aws::String>
  CreateRouteRequest& WithDestinationPrefixListId(DestinationPrefixListIdT&& value) {
    SetDestinationPrefixListId(std::forward<DestinationPrefixListIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a VPC endpoint. Supported for Gateway Load Balancer endpoints
   * only.</p>
   */
  inline const Aws::String& GetVpcEndpointId() const { return m_vpcEndpointId; }
  inline bool VpcEndpointIdHasBeenSet() const { return m_vpcEndpointIdHasBeenSet; }
  template <typename VpcEndpointIdT = Aws::String>
  void SetVpcEndpointId(VpcEndpointIdT&& value) {
    m_vpcEndpointIdHasBeenSet = true;
    m_vpcEndpointId = std::forward<VpcEndpointIdT>(value);
  }
  template <typename VpcEndpointIdT = Aws::String>
  CreateRouteRequest& WithVpcEndpointId(VpcEndpointIdT&& value) {
    SetVpcEndpointId(std::forward<VpcEndpointIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a transit gateway.</p>
   */
  inline const Aws::String& GetTransitGatewayId() const { return m_transitGatewayId; }
  inline bool TransitGatewayIdHasBeenSet() const { return m_transitGatewayIdHasBeenSet; }
  template <typename TransitGatewayIdT = Aws::String>
  void SetTransitGatewayId(TransitGatewayIdT&& value) {
    m_transitGatewayIdHasBeenSet = true;
    m_transitGatewayId = std::forward<TransitGatewayIdT>(value);
  }
  template <typename TransitGatewayIdT = Aws::String>
  CreateRouteRequest& WithTransitGatewayId(TransitGatewayIdT&& value) {
    SetTransitGatewayId(std::forward<TransitGatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the local gateway.</p>
   */
  inline const Aws::String& GetLocalGatewayId() const { return m_localGatewayId; }
  inline bool LocalGatewayIdHasBeenSet() const { return m_localGatewayIdHasBeenSet; }
  template <typename LocalGatewayIdT = Aws::String>
  void SetLocalGatewayId(LocalGatewayIdT&& value) {
    m_localGatewayIdHasBeenSet = true;
    m_localGatewayId = std::forward<LocalGatewayIdT>(value);
  }
  template <typename LocalGatewayIdT = Aws::String>
  CreateRouteRequest& WithLocalGatewayId(LocalGatewayIdT&& value) {
    SetLocalGatewayId(std::forward<LocalGatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the carrier gateway.</p> <p>You can only use this option when the
   * VPC contains a subnet which is associated with a Wavelength Zone.</p>
   */
  inline const Aws::String& GetCarrierGatewayId() const { return m_carrierGatewayId; }
  inline bool CarrierGatewayIdHasBeenSet() const { return m_carrierGatewayIdHasBeenSet; }
  template <typename CarrierGatewayIdT = Aws::String>
  void SetCarrierGatewayId(CarrierGatewayIdT&& value) {
    m_carrierGatewayIdHasBeenSet = true;
    m_carrierGatewayId = std::forward<CarrierGatewayIdT>(value);
  }
  template <typename CarrierGatewayIdT = Aws::String>
  CreateRouteRequest& WithCarrierGatewayId(CarrierGatewayIdT&& value) {
    SetCarrierGatewayId(std::forward<CarrierGatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the core network.</p>
   */
  inline const Aws::String& GetCoreNetworkArn() const { return m_coreNetworkArn; }
  inline bool CoreNetworkArnHasBeenSet() const { return m_coreNetworkArnHasBeenSet; }
  template <typename CoreNetworkArnT = Aws::String>
  void SetCoreNetworkArn(CoreNetworkArnT&& value) {
    m_coreNetworkArnHasBeenSet = true;
    m_coreNetworkArn = std::forward<CoreNetworkArnT>(value);
  }
  template <typename CoreNetworkArnT = Aws::String>
  CreateRouteRequest& WithCoreNetworkArn(CoreNetworkArnT&& value) {
    SetCoreNetworkArn(std::forward<CoreNetworkArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the ODB network.</p>
   */
  inline const Aws::String& GetOdbNetworkArn() const { return m_odbNetworkArn; }
  inline bool OdbNetworkArnHasBeenSet() const { return m_odbNetworkArnHasBeenSet; }
  template <typename OdbNetworkArnT = Aws::String>
  void SetOdbNetworkArn(OdbNetworkArnT&& value) {
    m_odbNetworkArnHasBeenSet = true;
    m_odbNetworkArn = std::forward<OdbNetworkArnT>(value);
  }
  template <typename OdbNetworkArnT = Aws::String>
  CreateRouteRequest& WithOdbNetworkArn(OdbNetworkArnT&& value) {
    SetOdbNetworkArn(std::forward<OdbNetworkArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateRouteRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the route table for the route.</p>
   */
  inline const Aws::String& GetRouteTableId() const { return m_routeTableId; }
  inline bool RouteTableIdHasBeenSet() const { return m_routeTableIdHasBeenSet; }
  template <typename RouteTableIdT = Aws::String>
  void SetRouteTableId(RouteTableIdT&& value) {
    m_routeTableIdHasBeenSet = true;
    m_routeTableId = std::forward<RouteTableIdT>(value);
  }
  template <typename RouteTableIdT = Aws::String>
  CreateRouteRequest& WithRouteTableId(RouteTableIdT&& value) {
    SetRouteTableId(std::forward<RouteTableIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv4 CIDR address block used for the destination match. Routing decisions
   * are based on the most specific match. We modify the specified CIDR block to its
   * canonical form; for example, if you specify <code>100.68.0.18/18</code>, we
   * modify it to <code>100.68.0.0/18</code>.</p>
   */
  inline const Aws::String& GetDestinationCidrBlock() const { return m_destinationCidrBlock; }
  inline bool DestinationCidrBlockHasBeenSet() const { return m_destinationCidrBlockHasBeenSet; }
  template <typename DestinationCidrBlockT = Aws::String>
  void SetDestinationCidrBlock(DestinationCidrBlockT&& value) {
    m_destinationCidrBlockHasBeenSet = true;
    m_destinationCidrBlock = std::forward<DestinationCidrBlockT>(value);
  }
  template <typename DestinationCidrBlockT = Aws::String>
  CreateRouteRequest& WithDestinationCidrBlock(DestinationCidrBlockT&& value) {
    SetDestinationCidrBlock(std::forward<DestinationCidrBlockT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of an internet gateway or virtual private gateway attached to your
   * VPC.</p>
   */
  inline const Aws::String& GetGatewayId() const { return m_gatewayId; }
  inline bool GatewayIdHasBeenSet() const { return m_gatewayIdHasBeenSet; }
  template <typename GatewayIdT = Aws::String>
  void SetGatewayId(GatewayIdT&& value) {
    m_gatewayIdHasBeenSet = true;
    m_gatewayId = std::forward<GatewayIdT>(value);
  }
  template <typename GatewayIdT = Aws::String>
  CreateRouteRequest& WithGatewayId(GatewayIdT&& value) {
    SetGatewayId(std::forward<GatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv6 CIDR block used for the destination match. Routing decisions are
   * based on the most specific match.</p>
   */
  inline const Aws::String& GetDestinationIpv6CidrBlock() const { return m_destinationIpv6CidrBlock; }
  inline bool DestinationIpv6CidrBlockHasBeenSet() const { return m_destinationIpv6CidrBlockHasBeenSet; }
  template <typename DestinationIpv6CidrBlockT = Aws::String>
  void SetDestinationIpv6CidrBlock(DestinationIpv6CidrBlockT&& value) {
    m_destinationIpv6CidrBlockHasBeenSet = true;
    m_destinationIpv6CidrBlock = std::forward<DestinationIpv6CidrBlockT>(value);
  }
  template <typename DestinationIpv6CidrBlockT = Aws::String>
  CreateRouteRequest& WithDestinationIpv6CidrBlock(DestinationIpv6CidrBlockT&& value) {
    SetDestinationIpv6CidrBlock(std::forward<DestinationIpv6CidrBlockT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>[IPv6 traffic only] The ID of an egress-only internet gateway.</p>
   */
  inline const Aws::String& GetEgressOnlyInternetGatewayId() const { return m_egressOnlyInternetGatewayId; }
  inline bool EgressOnlyInternetGatewayIdHasBeenSet() const { return m_egressOnlyInternetGatewayIdHasBeenSet; }
  template <typename EgressOnlyInternetGatewayIdT = Aws::String>
  void SetEgressOnlyInternetGatewayId(EgressOnlyInternetGatewayIdT&& value) {
    m_egressOnlyInternetGatewayIdHasBeenSet = true;
    m_egressOnlyInternetGatewayId = std::forward<EgressOnlyInternetGatewayIdT>(value);
  }
  template <typename EgressOnlyInternetGatewayIdT = Aws::String>
  CreateRouteRequest& WithEgressOnlyInternetGatewayId(EgressOnlyInternetGatewayIdT&& value) {
    SetEgressOnlyInternetGatewayId(std::forward<EgressOnlyInternetGatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a NAT instance in your VPC. The operation fails if you specify an
   * instance ID unless exactly one network interface is attached.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  CreateRouteRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a network interface.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  CreateRouteRequest& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a VPC peering connection.</p>
   */
  inline const Aws::String& GetVpcPeeringConnectionId() const { return m_vpcPeeringConnectionId; }
  inline bool VpcPeeringConnectionIdHasBeenSet() const { return m_vpcPeeringConnectionIdHasBeenSet; }
  template <typename VpcPeeringConnectionIdT = Aws::String>
  void SetVpcPeeringConnectionId(VpcPeeringConnectionIdT&& value) {
    m_vpcPeeringConnectionIdHasBeenSet = true;
    m_vpcPeeringConnectionId = std::forward<VpcPeeringConnectionIdT>(value);
  }
  template <typename VpcPeeringConnectionIdT = Aws::String>
  CreateRouteRequest& WithVpcPeeringConnectionId(VpcPeeringConnectionIdT&& value) {
    SetVpcPeeringConnectionId(std::forward<VpcPeeringConnectionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>[IPv4 traffic only] The ID of a NAT gateway.</p>
   */
  inline const Aws::String& GetNatGatewayId() const { return m_natGatewayId; }
  inline bool NatGatewayIdHasBeenSet() const { return m_natGatewayIdHasBeenSet; }
  template <typename NatGatewayIdT = Aws::String>
  void SetNatGatewayId(NatGatewayIdT&& value) {
    m_natGatewayIdHasBeenSet = true;
    m_natGatewayId = std::forward<NatGatewayIdT>(value);
  }
  template <typename NatGatewayIdT = Aws::String>
  CreateRouteRequest& WithNatGatewayId(NatGatewayIdT&& value) {
    SetNatGatewayId(std::forward<NatGatewayIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_destinationPrefixListId;
  bool m_destinationPrefixListIdHasBeenSet = false;

  Aws::String m_vpcEndpointId;
  bool m_vpcEndpointIdHasBeenSet = false;

  Aws::String m_transitGatewayId;
  bool m_transitGatewayIdHasBeenSet = false;

  Aws::String m_localGatewayId;
  bool m_localGatewayIdHasBeenSet = false;

  Aws::String m_carrierGatewayId;
  bool m_carrierGatewayIdHasBeenSet = false;

  Aws::String m_coreNetworkArn;
  bool m_coreNetworkArnHasBeenSet = false;

  Aws::String m_odbNetworkArn;
  bool m_odbNetworkArnHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::String m_routeTableId;
  bool m_routeTableIdHasBeenSet = false;

  Aws::String m_destinationCidrBlock;
  bool m_destinationCidrBlockHasBeenSet = false;

  Aws::String m_gatewayId;
  bool m_gatewayIdHasBeenSet = false;

  Aws::String m_destinationIpv6CidrBlock;
  bool m_destinationIpv6CidrBlockHasBeenSet = false;

  Aws::String m_egressOnlyInternetGatewayId;
  bool m_egressOnlyInternetGatewayIdHasBeenSet = false;

  Aws::String m_instanceId;
  bool m_instanceIdHasBeenSet = false;

  Aws::String m_networkInterfaceId;
  bool m_networkInterfaceIdHasBeenSet = false;

  Aws::String m_vpcPeeringConnectionId;
  bool m_vpcPeeringConnectionIdHasBeenSet = false;

  Aws::String m_natGatewayId;
  bool m_natGatewayIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
