﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/Filter.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DescribePublicIpv4PoolsRequest : public EC2Request {
 public:
  AWS_EC2_API DescribePublicIpv4PoolsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribePublicIpv4Pools"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The IDs of the address pools.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPoolIds() const { return m_poolIds; }
  inline bool PoolIdsHasBeenSet() const { return m_poolIdsHasBeenSet; }
  template <typename PoolIdsT = Aws::Vector<Aws::String>>
  void SetPoolIds(PoolIdsT&& value) {
    m_poolIdsHasBeenSet = true;
    m_poolIds = std::forward<PoolIdsT>(value);
  }
  template <typename PoolIdsT = Aws::Vector<Aws::String>>
  DescribePublicIpv4PoolsRequest& WithPoolIds(PoolIdsT&& value) {
    SetPoolIds(std::forward<PoolIdsT>(value));
    return *this;
  }
  template <typename PoolIdsT = Aws::String>
  DescribePublicIpv4PoolsRequest& AddPoolIds(PoolIdsT&& value) {
    m_poolIdsHasBeenSet = true;
    m_poolIds.emplace_back(std::forward<PoolIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribePublicIpv4PoolsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with a single call. To retrieve the
   * remaining results, make another call with the returned <code>nextToken</code>
   * value.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribePublicIpv4PoolsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more filters.</p> <ul> <li> <p> <code>tag</code>:&lt;key&gt; - The
   * key/value combination of a tag assigned to the resource. Use the tag key in the
   * filter name and the tag value as the filter value. For example, to find all
   * resources that have a tag with the key <code>Owner</code> and the value
   * <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and
   * <code>TeamA</code> for the filter value.</p> </li> <li> <p> <code>tag-key</code>
   * - The key of a tag assigned to the resource. Use this filter to find all
   * resources assigned a tag with a specific key, regardless of the tag value.</p>
   * </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribePublicIpv4PoolsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribePublicIpv4PoolsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_poolIds;
  bool m_poolIdsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
