﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/PropagatingVgw.h>
#include <aws/ec2/model/Route.h>
#include <aws/ec2/model/RouteTableAssociation.h>
#include <aws/ec2/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes a route table.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/RouteTable">AWS API
 * Reference</a></p>
 */
class RouteTable {
 public:
  AWS_EC2_API RouteTable() = default;
  AWS_EC2_API RouteTable(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API RouteTable& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The associations between the route table and your subnets or gateways.</p>
   */
  inline const Aws::Vector<RouteTableAssociation>& GetAssociations() const { return m_associations; }
  inline bool AssociationsHasBeenSet() const { return m_associationsHasBeenSet; }
  template <typename AssociationsT = Aws::Vector<RouteTableAssociation>>
  void SetAssociations(AssociationsT&& value) {
    m_associationsHasBeenSet = true;
    m_associations = std::forward<AssociationsT>(value);
  }
  template <typename AssociationsT = Aws::Vector<RouteTableAssociation>>
  RouteTable& WithAssociations(AssociationsT&& value) {
    SetAssociations(std::forward<AssociationsT>(value));
    return *this;
  }
  template <typename AssociationsT = RouteTableAssociation>
  RouteTable& AddAssociations(AssociationsT&& value) {
    m_associationsHasBeenSet = true;
    m_associations.emplace_back(std::forward<AssociationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any virtual private gateway (VGW) propagating routes.</p>
   */
  inline const Aws::Vector<PropagatingVgw>& GetPropagatingVgws() const { return m_propagatingVgws; }
  inline bool PropagatingVgwsHasBeenSet() const { return m_propagatingVgwsHasBeenSet; }
  template <typename PropagatingVgwsT = Aws::Vector<PropagatingVgw>>
  void SetPropagatingVgws(PropagatingVgwsT&& value) {
    m_propagatingVgwsHasBeenSet = true;
    m_propagatingVgws = std::forward<PropagatingVgwsT>(value);
  }
  template <typename PropagatingVgwsT = Aws::Vector<PropagatingVgw>>
  RouteTable& WithPropagatingVgws(PropagatingVgwsT&& value) {
    SetPropagatingVgws(std::forward<PropagatingVgwsT>(value));
    return *this;
  }
  template <typename PropagatingVgwsT = PropagatingVgw>
  RouteTable& AddPropagatingVgws(PropagatingVgwsT&& value) {
    m_propagatingVgwsHasBeenSet = true;
    m_propagatingVgws.emplace_back(std::forward<PropagatingVgwsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the route table.</p>
   */
  inline const Aws::String& GetRouteTableId() const { return m_routeTableId; }
  inline bool RouteTableIdHasBeenSet() const { return m_routeTableIdHasBeenSet; }
  template <typename RouteTableIdT = Aws::String>
  void SetRouteTableId(RouteTableIdT&& value) {
    m_routeTableIdHasBeenSet = true;
    m_routeTableId = std::forward<RouteTableIdT>(value);
  }
  template <typename RouteTableIdT = Aws::String>
  RouteTable& WithRouteTableId(RouteTableIdT&& value) {
    SetRouteTableId(std::forward<RouteTableIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The routes in the route table.</p>
   */
  inline const Aws::Vector<Route>& GetRoutes() const { return m_routes; }
  inline bool RoutesHasBeenSet() const { return m_routesHasBeenSet; }
  template <typename RoutesT = Aws::Vector<Route>>
  void SetRoutes(RoutesT&& value) {
    m_routesHasBeenSet = true;
    m_routes = std::forward<RoutesT>(value);
  }
  template <typename RoutesT = Aws::Vector<Route>>
  RouteTable& WithRoutes(RoutesT&& value) {
    SetRoutes(std::forward<RoutesT>(value));
    return *this;
  }
  template <typename RoutesT = Route>
  RouteTable& AddRoutes(RoutesT&& value) {
    m_routesHasBeenSet = true;
    m_routes.emplace_back(std::forward<RoutesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any tags assigned to the route table.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  RouteTable& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  RouteTable& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the VPC.</p>
   */
  inline const Aws::String& GetVpcId() const { return m_vpcId; }
  inline bool VpcIdHasBeenSet() const { return m_vpcIdHasBeenSet; }
  template <typename VpcIdT = Aws::String>
  void SetVpcId(VpcIdT&& value) {
    m_vpcIdHasBeenSet = true;
    m_vpcId = std::forward<VpcIdT>(value);
  }
  template <typename VpcIdT = Aws::String>
  RouteTable& WithVpcId(VpcIdT&& value) {
    SetVpcId(std::forward<VpcIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that owns the route table.</p>
   */
  inline const Aws::String& GetOwnerId() const { return m_ownerId; }
  inline bool OwnerIdHasBeenSet() const { return m_ownerIdHasBeenSet; }
  template <typename OwnerIdT = Aws::String>
  void SetOwnerId(OwnerIdT&& value) {
    m_ownerIdHasBeenSet = true;
    m_ownerId = std::forward<OwnerIdT>(value);
  }
  template <typename OwnerIdT = Aws::String>
  RouteTable& WithOwnerId(OwnerIdT&& value) {
    SetOwnerId(std::forward<OwnerIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<RouteTableAssociation> m_associations;
  bool m_associationsHasBeenSet = false;

  Aws::Vector<PropagatingVgw> m_propagatingVgws;
  bool m_propagatingVgwsHasBeenSet = false;

  Aws::String m_routeTableId;
  bool m_routeTableIdHasBeenSet = false;

  Aws::Vector<Route> m_routes;
  bool m_routesHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_vpcId;
  bool m_vpcIdHasBeenSet = false;

  Aws::String m_ownerId;
  bool m_ownerIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
