﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesis/KinesisRequest.h>
#include <aws/kinesis/Kinesis_EXPORTS.h>
#include <aws/kinesis/model/StreamModeDetails.h>

#include <utility>

namespace Aws {
namespace Kinesis {
namespace Model {

/**
 */
class UpdateStreamModeRequest : public KinesisRequest {
 public:
  AWS_KINESIS_API UpdateStreamModeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateStreamMode"; }

  AWS_KINESIS_API Aws::String SerializePayload() const override;

  AWS_KINESIS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_KINESIS_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p> Specifies the ARN of the data stream whose capacity mode you want to update.
   * </p>
   */
  inline const Aws::String& GetStreamARN() const { return m_streamARN; }
  inline bool StreamARNHasBeenSet() const { return m_streamARNHasBeenSet; }
  template <typename StreamARNT = Aws::String>
  void SetStreamARN(StreamARNT&& value) {
    m_streamARNHasBeenSet = true;
    m_streamARN = std::forward<StreamARNT>(value);
  }
  template <typename StreamARNT = Aws::String>
  UpdateStreamModeRequest& WithStreamARN(StreamARNT&& value) {
    SetStreamARN(std::forward<StreamARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Specifies the capacity mode to which you want to set your data stream.
   * Currently, in Kinesis Data Streams, you can choose between an <b>on-demand</b>
   * capacity mode and a <b>provisioned</b> capacity mode for your data streams. </p>
   */
  inline const StreamModeDetails& GetStreamModeDetails() const { return m_streamModeDetails; }
  inline bool StreamModeDetailsHasBeenSet() const { return m_streamModeDetailsHasBeenSet; }
  template <typename StreamModeDetailsT = StreamModeDetails>
  void SetStreamModeDetails(StreamModeDetailsT&& value) {
    m_streamModeDetailsHasBeenSet = true;
    m_streamModeDetails = std::forward<StreamModeDetailsT>(value);
  }
  template <typename StreamModeDetailsT = StreamModeDetails>
  UpdateStreamModeRequest& WithStreamModeDetails(StreamModeDetailsT&& value) {
    SetStreamModeDetails(std::forward<StreamModeDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The target warm throughput in MB/s that the stream should be scaled to
   * handle. This represents the throughput capacity that will be immediately
   * available for write operations. This field is only valid when the stream mode is
   * being updated to on-demand.</p>
   */
  inline int GetWarmThroughputMiBps() const { return m_warmThroughputMiBps; }
  inline bool WarmThroughputMiBpsHasBeenSet() const { return m_warmThroughputMiBpsHasBeenSet; }
  inline void SetWarmThroughputMiBps(int value) {
    m_warmThroughputMiBpsHasBeenSet = true;
    m_warmThroughputMiBps = value;
  }
  inline UpdateStreamModeRequest& WithWarmThroughputMiBps(int value) {
    SetWarmThroughputMiBps(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_streamARN;
  bool m_streamARNHasBeenSet = false;

  StreamModeDetails m_streamModeDetails;
  bool m_streamModeDetailsHasBeenSet = false;

  int m_warmThroughputMiBps{0};
  bool m_warmThroughputMiBpsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kinesis
}  // namespace Aws
