﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/transcribe/TranscribeServiceRequest.h>
#include <aws/transcribe/TranscribeService_EXPORTS.h>
#include <aws/transcribe/model/LanguageCode.h>
#include <aws/transcribe/model/Tag.h>

#include <utility>

namespace Aws {
namespace TranscribeService {
namespace Model {

/**
 */
class CreateMedicalVocabularyRequest : public TranscribeServiceRequest {
 public:
  AWS_TRANSCRIBESERVICE_API CreateMedicalVocabularyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateMedicalVocabulary"; }

  AWS_TRANSCRIBESERVICE_API Aws::String SerializePayload() const override;

  AWS_TRANSCRIBESERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique name, chosen by you, for your new custom medical vocabulary.</p>
   * <p>This name is case sensitive, cannot contain spaces, and must be unique within
   * an Amazon Web Services account. If you try to create a new custom medical
   * vocabulary with the same name as an existing custom medical vocabulary, you get
   * a <code>ConflictException</code> error.</p>
   */
  inline const Aws::String& GetVocabularyName() const { return m_vocabularyName; }
  inline bool VocabularyNameHasBeenSet() const { return m_vocabularyNameHasBeenSet; }
  template <typename VocabularyNameT = Aws::String>
  void SetVocabularyName(VocabularyNameT&& value) {
    m_vocabularyNameHasBeenSet = true;
    m_vocabularyName = std::forward<VocabularyNameT>(value);
  }
  template <typename VocabularyNameT = Aws::String>
  CreateMedicalVocabularyRequest& WithVocabularyName(VocabularyNameT&& value) {
    SetVocabularyName(std::forward<VocabularyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The language code that represents the language of the entries in your custom
   * vocabulary. US English (<code>en-US</code>) is the only language supported with
   * Amazon Transcribe Medical.</p>
   */
  inline LanguageCode GetLanguageCode() const { return m_languageCode; }
  inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
  inline void SetLanguageCode(LanguageCode value) {
    m_languageCodeHasBeenSet = true;
    m_languageCode = value;
  }
  inline CreateMedicalVocabularyRequest& WithLanguageCode(LanguageCode value) {
    SetLanguageCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 location (URI) of the text file that contains your custom
   * medical vocabulary. The URI must be in the same Amazon Web Services Region as
   * the resource you're calling.</p> <p>Here's an example URI path:
   * <code>s3://DOC-EXAMPLE-BUCKET/my-vocab-file.txt</code> </p>
   */
  inline const Aws::String& GetVocabularyFileUri() const { return m_vocabularyFileUri; }
  inline bool VocabularyFileUriHasBeenSet() const { return m_vocabularyFileUriHasBeenSet; }
  template <typename VocabularyFileUriT = Aws::String>
  void SetVocabularyFileUri(VocabularyFileUriT&& value) {
    m_vocabularyFileUriHasBeenSet = true;
    m_vocabularyFileUri = std::forward<VocabularyFileUriT>(value);
  }
  template <typename VocabularyFileUriT = Aws::String>
  CreateMedicalVocabularyRequest& WithVocabularyFileUri(VocabularyFileUriT&& value) {
    SetVocabularyFileUri(std::forward<VocabularyFileUriT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Adds one or more custom tags, each in the form of a key:value pair, to a new
   * custom medical vocabulary at the time you create this new custom vocabulary.</p>
   * <p>To learn more about using tags with Amazon Transcribe, refer to <a
   * href="https://docs.aws.amazon.com/transcribe/latest/dg/tagging.html">Tagging
   * resources</a>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateMedicalVocabularyRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateMedicalVocabularyRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_vocabularyName;

  LanguageCode m_languageCode{LanguageCode::NOT_SET};

  Aws::String m_vocabularyFileUri;

  Aws::Vector<Tag> m_tags;
  bool m_vocabularyNameHasBeenSet = false;
  bool m_languageCodeHasBeenSet = false;
  bool m_vocabularyFileUriHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace TranscribeService
}  // namespace Aws
