/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.plugintool;

import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainObject;
import ghidra.framework.options.SaveState;
import ghidra.framework.plugintool.Plugin;
import ghidra.framework.plugintool.PluginEvent;
import ghidra.framework.plugintool.PluginTool;
import ghidra.framework.plugintool.PluginsConfiguration;
import ghidra.framework.plugintool.ServiceInterfaceImplementationPair;
import ghidra.framework.plugintool.mgr.ServiceManager;
import ghidra.framework.plugintool.util.PluginDescription;
import ghidra.framework.plugintool.util.PluginException;
import ghidra.framework.plugintool.util.PluginPackage;
import ghidra.framework.plugintool.util.PluginUtils;
import ghidra.framework.plugintool.util.TransientToolState;
import ghidra.framework.plugintool.util.UndoRedoToolState;
import ghidra.util.Msg;
import ghidra.util.classfinder.ClassSearcher;
import ghidra.util.exception.MultipleCauses;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jdom.Content;
import org.jdom.Element;

class PluginManager {
    private static final Logger log = LogManager.getLogger(PluginManager.class);
    private PluginsConfiguration pluginsConfiguration;
    private List<Plugin> pluginList = new ArrayList<Plugin>();
    private PluginTool tool;
    private ServiceManager serviceMgr;

    PluginManager(PluginTool tool, ServiceManager serviceMgr, PluginsConfiguration pluginsConfiguration) {
        this.tool = tool;
        this.serviceMgr = serviceMgr;
        this.pluginsConfiguration = pluginsConfiguration;
    }

    void installUtilityPlugins() throws PluginException {
        PluginPackage utilityPackage = PluginPackage.getPluginPackage("Utility");
        List<PluginDescription> descriptions = this.pluginsConfiguration.getPluginDescriptions(utilityPackage);
        if (descriptions == null) {
            return;
        }
        HashSet<String> classNames = new HashSet<String>();
        for (PluginDescription description : descriptions) {
            String pluginClass = description.getPluginClass().getName();
            classNames.add(pluginClass);
        }
        this.addPlugins(classNames);
    }

    PluginsConfiguration getPluginsConfiguration() {
        return this.pluginsConfiguration;
    }

    boolean acceptData(DomainFile[] data) {
        for (Plugin p : this.pluginList) {
            if (!p.acceptData(data)) continue;
            this.tool.getWindowManager().getMainWindow().toFront();
            return true;
        }
        return false;
    }

    boolean accept(URL url) {
        for (Plugin p : this.pluginList) {
            if (!p.accept(url)) continue;
            this.tool.getWindowManager().getMainWindow().toFront();
            return true;
        }
        return false;
    }

    void dispose() {
        Iterator<Plugin> it = this.pluginList.iterator();
        while (it.hasNext()) {
            Plugin plugin = it.next();
            plugin.cleanup();
            it.remove();
        }
    }

    DomainFile[] getData() {
        ArrayList<DomainFile> list = new ArrayList<DomainFile>();
        for (Plugin plugin : this.pluginList) {
            for (DomainFile file : plugin.getData()) {
                list.add(file);
            }
        }
        DomainFile[] data = new DomainFile[list.size()];
        return list.toArray(data);
    }

    Class<?>[] getSupportedDataTypes() {
        HashSet set = new HashSet();
        for (Plugin plugin : this.pluginList) {
            for (Class<?> element : plugin.getSupportedDataTypes()) {
                set.add(element);
            }
        }
        Class[] cl = new Class[set.size()];
        return set.toArray(cl);
    }

    void addPlugin(Plugin plugin) throws PluginException {
        this.addPlugins(new Plugin[]{plugin});
    }

    void addPlugins(Collection<String> classNames) throws PluginException {
        PluginException pe = null;
        ArrayList<Plugin> list = new ArrayList<Plugin>(classNames.size());
        ArrayList<String> badList = new ArrayList<String>();
        for (String className : classNames) {
            try {
                Class<? extends Plugin> pluginClass = PluginUtils.forName(className);
                if (this.getLoadedPlugin(pluginClass) != null) continue;
                PluginUtils.assertUniquePluginName(pluginClass);
                Plugin p = PluginUtils.instantiatePlugin(pluginClass, this.tool);
                list.add(p);
            }
            catch (PluginException e) {
                pe = e.getPluginException(pe);
                badList.add(className);
            }
        }
        Plugin[] pluginArray = list.toArray(new Plugin[list.size()]);
        try {
            this.addPlugins(pluginArray);
        }
        catch (PluginException e) {
            pe = e.getPluginException(pe);
        }
        if (badList.size() > 0) {
            for (String className : badList) {
                this.tool.removeEventListener(className);
            }
        }
        if (pe != null) {
            throw pe;
        }
    }

    private <T extends Plugin> T getLoadedPlugin(Class<T> pluginClass) {
        for (Plugin p : this.pluginList) {
            if (p.getClass() != pluginClass) continue;
            return (T)((Plugin)pluginClass.cast(p));
        }
        return null;
    }

    private void addPlugins(Plugin[] plugs) throws PluginException {
        this.serviceMgr.setServiceAddedNotificationsOn(false);
        List<ServiceInterfaceImplementationPair> previousServices = this.serviceMgr.getAllServices();
        StringBuilder errMsg = new StringBuilder();
        MultipleCauses report = new MultipleCauses();
        int numOldPlugins = this.pluginList.size();
        for (Plugin newPluginToAdd : plugs) {
            this.pluginList.add(newPluginToAdd);
            newPluginToAdd.initServices();
        }
        HashMap dependencyProblemResults = new HashMap();
        Set<PluginDependency> unresolvedDependencySet = this.resolveDependencies(dependencyProblemResults);
        if (!unresolvedDependencySet.isEmpty()) {
            for (PluginDependency pd : unresolvedDependencySet) {
                Class<?> dependency = pd.dependency();
                PluginException cause = (PluginException)((Object)dependencyProblemResults.get(dependency));
                String dependencyName = dependency.getName();
                String dependentName = pd.dependant().getName();
                String message = "Unresolved dependency: %s.  Used by plugin: %s\n".formatted(dependencyName, dependentName);
                errMsg.append(message).append('\n');
                if (cause != null) {
                    errMsg.append("Reason: ").append(cause.getMessage()).append('\n');
                }
                report.addCause((Throwable)((Object)new PluginException(message, (Throwable)((Object)cause))));
            }
            this.cleanupPluginsWithUnresolvedDependencies();
        }
        PluginEvent[] lastEvents = this.tool.getLastEvents();
        List<Plugin> sortedPlugins = this.getPluginsByServiceOrder(numOldPlugins);
        ArrayList<Plugin> badList = new ArrayList<Plugin>();
        Iterator<Plugin> it = sortedPlugins.iterator();
        while (it.hasNext()) {
            Plugin p = it.next();
            try {
                p.init();
            }
            catch (Throwable t) {
                Msg.error((Object)this, (Object)("Unexpected Exception: " + t.getMessage()), (Throwable)t);
                errMsg.append("Initializing " + p.getName() + " failed: " + String.valueOf(t) + "\n");
                report.addCause(t);
                badList.add(p);
                it.remove();
            }
        }
        if (badList.size() > 0) {
            try {
                this.removePlugins(badList);
            }
            catch (Throwable t) {
                log.debug("Exception unloading plugin", t);
            }
        }
        this.serviceMgr.setServiceAddedNotificationsOn(true);
        for (Plugin p : sortedPlugins) {
            this.notifyServices(p, previousServices);
        }
        for (Plugin p : sortedPlugins) {
            p.processLastEvents(lastEvents);
        }
        if (errMsg.length() > 0) {
            throw new PluginException(errMsg.toString(), (Throwable)report);
        }
    }

    private void notifyServices(Plugin p, List<ServiceInterfaceImplementationPair> previousServices) {
        for (ServiceInterfaceImplementationPair service : previousServices) {
            p.serviceAdded(service.interfaceClass, service.provider);
        }
    }

    List<Plugin> getPlugins() {
        return new ArrayList<Plugin>(this.pluginList);
    }

    void removePlugins(List<Plugin> plugins) {
        for (Plugin plugin : plugins) {
            this.unregisterPlugin(plugin);
        }
        this.cleanupPluginsWithUnresolvedDependencies();
    }

    void saveToXml(Element root, boolean includeConfigState) {
        this.pluginsConfiguration.savePluginsToXml(root, this.pluginList);
        if (!includeConfigState) {
            return;
        }
        SaveState saveState = new SaveState("PLUGIN_STATE");
        for (Plugin p : this.pluginList) {
            p.writeConfigState(saveState);
            if (saveState.isEmpty()) continue;
            Element pluginElem = saveState.saveToXml();
            pluginElem.setAttribute("CLASS", p.getClass().getName());
            root.addContent((Content)pluginElem);
            saveState = new SaveState("PLUGIN_STATE");
        }
    }

    void restorePluginsFromXml(Element root) throws PluginException {
        boolean isOld = this.isOldToolConfig(root);
        Set<String> classNames = isOld ? this.getPluginClassNamesFromOldXml(root) : this.pluginsConfiguration.getPluginClassNames(root);
        Map<String, SaveState> map = isOld ? this.getPluginSavedStates(root, "PLUGIN") : this.getPluginSavedStates(root, "PLUGIN_STATE");
        PluginException pe = null;
        try {
            this.addPlugins(classNames);
        }
        catch (PluginException e) {
            pe = e;
        }
        try {
            this.initConfigStates(map);
        }
        catch (PluginException e) {
            pe = e.getPluginException(pe);
        }
        if (pe != null) {
            throw pe;
        }
    }

    private Map<String, SaveState> getPluginSavedStates(Element root, String elementName) {
        HashMap<String, SaveState> map = new HashMap<String, SaveState>();
        List children = root.getChildren(elementName);
        for (Object object : children) {
            Element child = (Element)object;
            String className = child.getAttributeValue("CLASS");
            map.put(className, new SaveState(child));
        }
        return map;
    }

    private Set<String> getPluginClassNamesFromOldXml(Element root) {
        ArrayList<String> classNames = new ArrayList<String>();
        List pluginElementList = root.getChildren("PLUGIN");
        for (Element elem : pluginElementList) {
            String className = elem.getAttributeValue("CLASS");
            classNames.add(className);
        }
        return this.pluginsConfiguration.getPluginNamesByCurrentPackage(classNames);
    }

    private boolean isOldToolConfig(Element root) {
        return root.getChild("PLUGIN") != null;
    }

    void restoreDataStateFromXml(Element root) {
        HashMap<String, SaveState> map = new HashMap<String, SaveState>();
        for (Element elem : root.getChildren("PLUGIN")) {
            String pluginName = elem.getAttributeValue("NAME");
            SaveState saveState = new SaveState(elem);
            map.put(pluginName, saveState);
        }
        LinkedHashMap<String, Exception> badMap = new LinkedHashMap<String, Exception>();
        List<Plugin> plugins = this.getPluginsByServiceOrder(0);
        for (Plugin p : plugins) {
            SaveState saveState = (SaveState)map.get(p.getName());
            if (saveState == null) continue;
            try {
                p.readDataState(saveState);
            }
            catch (Exception e) {
                badMap.put(p.getName(), e);
            }
        }
        if (badMap.size() > 0) {
            log.error("*** Errors in Plugin Data States  ***");
            log.error("The data states for following plugins could not be restored:");
            Set entrySet = badMap.entrySet();
            for (Map.Entry entry : entrySet) {
                String pluginName = (String)entry.getKey();
                Exception exception = (Exception)entry.getValue();
                log.error("     " + pluginName, (Throwable)exception);
            }
            log.error("*** (finished) Errors in Plugin Data States  ***");
            Msg.showError((Object)this, null, (String)"Data State Error", (Object)"Errors in plugin data states - check console for details");
        }
        plugins.forEach(Plugin::dataStateRestoreCompleted);
    }

    Element saveDataStateToXml(boolean savingProject) {
        Element root = new Element("DATA_STATE");
        for (Plugin p : this.pluginList) {
            SaveState ss = new SaveState("PLUGIN");
            p.writeDataState(ss);
            if (ss.isEmpty()) continue;
            Element e = ss.saveToXml();
            e.setAttribute("NAME", p.getName());
            root.addContent((Content)e);
        }
        return root;
    }

    private void unregisterPlugin(Plugin plugin) {
        if (this.pluginList.remove(plugin)) {
            plugin.cleanup();
        }
    }

    private void cleanupPluginsWithUnresolvedDependencies() {
        Plugin p;
        while ((p = this.findPluginWithUnresolvedDependencies()) != null) {
            this.unregisterPlugin(p);
        }
    }

    private Plugin findPluginWithUnresolvedDependencies() {
        for (Plugin plugin : this.pluginList) {
            if (!plugin.hasMissingRequiredService()) continue;
            return plugin;
        }
        return null;
    }

    private Set<PluginDependency> resolveDependencies(Map<Class<?>, PluginException> dependencyProblemResults) {
        HashSet<PluginDependency> dependencies = new HashSet<PluginDependency>();
        do {
            this.getUnresolvedDependencies(dependencies);
        } while (this.addDependencies(dependencies, dependencyProblemResults));
        return dependencies;
    }

    private void getUnresolvedDependencies(Set<PluginDependency> dependencies) {
        dependencies.clear();
        for (Plugin p : this.pluginList) {
            List<Class<?>> missingServices = p.getMissingRequiredServices();
            for (Class<?> missingService : missingServices) {
                dependencies.add(new PluginDependency(p.getClass(), missingService));
            }
        }
    }

    private boolean addDependencies(Set<PluginDependency> dependencies, Map<Class<?>, PluginException> dependencyProblemResults) {
        boolean fixedDependency = false;
        for (PluginDependency pd : dependencies) {
            Class<?> dependency = pd.dependency();
            fixedDependency |= this.fixDependency(dependency, dependencyProblemResults);
        }
        return fixedDependency;
    }

    private Class<? extends Plugin> discoverServiceProvider(Class<?> dependency) {
        Class<? extends Plugin> pluginClass = PluginUtils.getDefaultProviderForServiceClass(dependency);
        if (pluginClass != null) {
            return pluginClass;
        }
        List plugins = ClassSearcher.getClasses(Plugin.class).stream().filter(this.pluginsConfiguration::accepts).collect(Collectors.toList());
        ArrayList<Class> serviceProviders = new ArrayList<Class>();
        for (Class pc : plugins) {
            if (dependency.isAssignableFrom(pc)) {
                serviceProviders.add(pc);
                continue;
            }
            PluginDescription pd = PluginDescription.getPluginDescription(pc);
            List<Class<?>> servicesProvided = pd.getServicesProvided();
            for (Class<?> service : servicesProvided) {
                if (!dependency.isAssignableFrom(service)) continue;
                serviceProviders.add(pc);
            }
        }
        if (serviceProviders.isEmpty()) {
            return null;
        }
        Class choice = (Class)serviceProviders.get(0);
        if (serviceProviders.size() != 1) {
            Msg.warn((Object)this, (Object)"Unable to find the preferred service provider implementation for %s.\nPicking %s\n".formatted(dependency.getName(), choice.getName()));
        }
        return choice;
    }

    private boolean fixDependency(Class<?> dependency, Map<Class<?>, PluginException> dependencyProblemResults) {
        Class<? extends Plugin> pluginClass = this.discoverServiceProvider(dependency);
        if (pluginClass == null) {
            return false;
        }
        if (this.getLoadedPlugin(pluginClass) != null) {
            return true;
        }
        try {
            PluginUtils.assertUniquePluginName(pluginClass);
            Plugin p = PluginUtils.instantiatePlugin(pluginClass, this.tool);
            p.initServices();
            this.pluginList.add(p);
            return true;
        }
        catch (PluginException e) {
            dependencyProblemResults.put(dependency, e);
            return false;
        }
    }

    private void initConfigStates(Map<String, SaveState> map) throws PluginException {
        StringBuilder errMsg = new StringBuilder();
        for (Plugin p : this.pluginList) {
            this.readSaveState(p, map, errMsg);
        }
        if (errMsg.length() > 0) {
            throw new PluginException(errMsg.toString());
        }
    }

    private void readSaveState(Plugin p, Map<String, SaveState> map, StringBuilder errMsg) {
        SaveState ss = map.get(p.getClass().getName());
        if (ss == null) {
            return;
        }
        try {
            p.readConfigState(ss);
        }
        catch (Exception e) {
            errMsg.append("Problem restoring plugin state for: " + p.getName()).append("\n\n");
            errMsg.append(e.getClass().getName()).append(": ").append(e.getMessage()).append('\n');
            StackTraceElement[] st = e.getStackTrace();
            int depth = Math.min(5, st.length);
            for (int j = 0; j < depth; ++j) {
                errMsg.append("    ").append(st[j].toString()).append('\n');
            }
            errMsg.append('\n');
        }
    }

    private List<Plugin> getPluginsByServiceOrder(int startIndex) {
        ArrayList<Plugin> plugins = new ArrayList<Plugin>(this.pluginList.subList(startIndex, this.pluginList.size()));
        ArrayList<Plugin> orderedList = new ArrayList<Plugin>(plugins.size());
        while (plugins.size() > 0) {
            int n = plugins.size();
            Iterator it = plugins.iterator();
            while (it.hasNext()) {
                Plugin p = (Plugin)it.next();
                if (!this.checkServices(p, plugins)) continue;
                orderedList.add(p);
                it.remove();
            }
            if (n != plugins.size()) continue;
            this.showWarning(plugins);
            orderedList.addAll(plugins);
            plugins.clear();
        }
        return orderedList;
    }

    private boolean checkServices(Plugin usingPlugin, List<Plugin> serviceProvidingPlugins) {
        for (Class<?> usedService : usingPlugin.getServicesRequired()) {
            for (Plugin providingPlugin : serviceProvidingPlugins) {
                if (!providingPlugin.providesService(usedService)) continue;
                return false;
            }
        }
        return true;
    }

    private void showWarning(List<Plugin> plugins) {
        Msg.warn((Object)this, (Object)"The correct order for initializing the following plugins can't be\ndetermined because of circular use of services (check log)");
        for (Plugin plugin : plugins) {
            Msg.info((Object)this, (Object)("Plugin: " + plugin.getClass().getName()));
            for (Class<?> service : plugin.getServiceClasses()) {
                Msg.info((Object)this, (Object)("    provides: " + String.valueOf(service)));
            }
            for (Class clazz : plugin.getServicesRequired()) {
                Msg.info((Object)this, (Object)("    uses: " + String.valueOf(clazz)));
            }
        }
    }

    boolean canClose() {
        for (Plugin p : this.pluginList) {
            if (p.canClose()) continue;
            return false;
        }
        return true;
    }

    boolean canCloseDomainObject(DomainObject domainObject) {
        for (Plugin p : this.pluginList) {
            if (p.canCloseDomainObject(domainObject)) continue;
            return false;
        }
        return true;
    }

    boolean saveData() {
        for (Plugin p : this.pluginList) {
            if (p.saveData()) continue;
            return false;
        }
        return true;
    }

    boolean hasUnsavedData() {
        for (Plugin p : this.pluginList) {
            if (!p.hasUnsaveData()) continue;
            return true;
        }
        return false;
    }

    void close() {
        for (Plugin p : this.pluginList) {
            p.close();
        }
    }

    public TransientToolState getTransientState() {
        return new TransientToolState(new ArrayList<Plugin>(this.pluginList));
    }

    public UndoRedoToolState getUndoRedoToolState(DomainObject domainObject) {
        return new UndoRedoToolState(new ArrayList<Plugin>(this.pluginList), domainObject);
    }

    void prepareToSave(DomainObject domainObject) {
        for (Plugin p : this.pluginList) {
            p.prepareToSave(domainObject);
        }
    }

    private record PluginDependency(Class<?> dependant, Class<?> dependency) {
    }
}

