"""faker.providers.person.nl_BE - 8-9 Jan 2023.

Last names and male and female first names for locale 'nl_BE' (Dutch-speaking Belgium).
Source: Statbel (Directorate-general Statistics - Statistics Belgium), https://statbel.fgov.be/en/about-statbel, 2022.
"""

from collections import Counter, OrderedDict

from .. import Provider as PersonProvider


class Provider(PersonProvider):
    """Provider of `Person` data (male and female first and last names) for locale 'nl_BE' (Belgium Dutch, Flemish)."""

    # As per https://nl.wikipedia.org/wiki/Familienaam#Belgi%C3%AB, from 1 Jun 2014 a child can get
    # the family name of either parent, or both parents' family names separated by a space.
    # As per https://nl.wikipedia.org/wiki/Geboortenaam#Situatie_in_Belgi%C3%AB, marriage does NOT affect family names.
    #
    # Below reflects these with an assumed 10% doubled name (for both genders).
    formats_male = OrderedDict(
        (("{{first_name_male}} {{last_name}}", 90), ("{{first_name_male}} {{last_name}} {{last_name}}", 10))
    )
    formats_female = OrderedDict(
        (("{{first_name_female}} {{last_name}}", 90), ("{{first_name_female}} {{last_name}} {{last_name}}", 10))
    )
    formats = OrderedDict(Counter(formats_male) + Counter(formats_female))

    # 500 most common male first names in Flanders in 2022 as per StatBel
    # note this (simplistically) models 'nl_BE' as identical to Flanders (and ignores Brussels)
    #
    # Data sourced from sheet 'Men' columns E-F ('Flanders') rows 3..502 (top-500 names incl. frequency)
    # of Excel workbook at https://statbel.fgov.be/sites/default/files/files/documents/bevolking/
    # 5.10%20Namen%20en%20voornamen/5.10.2%20Voornamen%20van%20vrouwen%20en%20mannen/First%20names_Population_2022.xlsx
    # and then sorted by name
    first_names_male = OrderedDict(
        (
            ("Aaron", 2889),
            ("Abdul", 1215),
            ("Abdullah", 1130),
            ("Achiel", 1175),
            ("Adam", 4837),
            ("Adriaan", 1168),
            ("Adrian", 1357),
            ("Ahmad", 1532),
            ("Ahmed", 5233),
            ("Ahmet", 1538),
            ("Alain", 8280),
            ("Albert", 7205),
            ("Alex", 5059),
            ("Alexander", 9289),
            ("Alexandre", 1923),
            ("Alexis", 1191),
            ("Alfons", 6341),
            ("Alfred", 1476),
            ("Ali", 5761),
            ("Amir", 1669),
            ("Anas", 1312),
            ("Andreas", 3733),
            ("André", 16580),
            ("Andy", 6567),
            ("Angelo", 1831),
            ("Anthony", 3620),
            ("Antoine", 3850),
            ("Anton", 1565),
            ("Antonio", 1884),
            ("Antonius", 1139),
            ("Antoon", 2329),
            ("Armand", 1933),
            ("Arnaud", 2111),
            ("Arne", 6368),
            ("Arno", 3833),
            ("Arnold", 1110),
            ("Arthur", 7857),
            ("August", 2576),
            ("Axel", 3177),
            ("Ayman", 1008),
            ("Ayoub", 1653),
            ("Bart", 28608),
            ("Bas", 1938),
            ("Ben", 5033),
            ("Benjamin", 5464),
            ("Benny", 5423),
            ("Benoit", 1450),
            ("Bernard", 4627),
            ("Bert", 8695),
            ("Bilal", 1922),
            ("Bjorn", 3795),
            ("Björn", 3946),
            ("Bob", 1651),
            ("Boris", 1250),
            ("Brahim", 1144),
            ("Bram", 8657),
            ("Brecht", 4008),
            ("Brent", 4164),
            ("Brian", 1733),
            ("Briek", 985),
            ("Bruno", 6801),
            ("Bryan", 1893),
            ("Carl", 2689),
            ("Carlo", 2039),
            ("Carlos", 2616),
            ("Cas", 1945),
            ("Casper", 1362),
            ("Cedric", 2647),
            ("Charles", 3721),
            ("Chris", 5309),
            ("Christiaan", 5304),
            ("Christian", 6553),
            ("Christof", 1477),
            ("Christoph", 1322),
            ("Christophe", 9500),
            ("Claude", 1852),
            ("Constant", 1375),
            ("Cornelis", 1071),
            ("Cyriel", 1705),
            ("Cédric", 3306),
            ("Daan", 6062),
            ("Daniel", 11950),
            ("Daniël", 5704),
            ("Danny", 14014),
            ("Dany", 2101),
            ("Dario", 1380),
            ("Dave", 2526),
            ("David", 16804),
            ("Davy", 7650),
            ("Dean", 1094),
            ("Denis", 2612),
            ("Dennis", 3712),
            ("Didier", 4702),
            ("Diederik", 1392),
            ("Diego", 1646),
            ("Dieter", 6396),
            ("Dimitri", 5964),
            ("Dirk", 31219),
            ("Dominique", 3983),
            ("Dries", 7557),
            ("Dylan", 4151),
            ("Eddy", 16551),
            ("Edgard", 1297),
            ("Edmond", 1413),
            ("Edouard", 1216),
            ("Eduard", 3451),
            ("Edward", 1834),
            ("Edwin", 1900),
            ("El", 1546),
            ("Elias", 4557),
            ("Emiel", 5975),
            ("Emile", 3736),
            ("Emmanuel", 2820),
            ("Eric", 15680),
            ("Erik", 11733),
            ("Erwin", 6314),
            ("Etienne", 7068),
            ("Fabian", 1327),
            ("Felix", 3453),
            ("Ferdinand", 1599),
            ("Fernand", 2367),
            ("Ferre", 3013),
            ("Filip", 12468),
            ("Finn", 3614),
            ("Flor", 1262),
            ("Florent", 1250),
            ("Florian", 1754),
            ("Floris", 1592),
            ("Francis", 4947),
            ("Franciscus", 4620),
            ("Frank", 14439),
            ("Franky", 4773),
            ("Frans", 11347),
            ("François", 10144),
            ("Freddy", 10632),
            ("Frederic", 1538),
            ("Frederick", 1009),
            ("Frederik", 7154),
            ("Frédéric", 2620),
            ("Gabriel", 2770),
            ("Gabriël", 1041),
            ("Gaston", 2482),
            ("Geert", 21142),
            ("Geoffrey", 1661),
            ("George", 1256),
            ("Georges", 7694),
            ("Gerard", 3417),
            ("Gerardus", 1106),
            ("Gerrit", 1956),
            ("Gerry", 1040),
            ("Gert", 6171),
            ("Ghislain", 1419),
            ("Gianni", 1608),
            ("Gilbert", 6635),
            ("Gilles", 4438),
            ("Gino", 4390),
            ("Giovanni", 2129),
            ("Giuseppe", 1250),
            ("Glenn", 6482),
            ("Gregory", 3007),
            ("Guido", 13295),
            ("Guillaume", 3037),
            ("Gunter", 1894),
            ("Gunther", 3181),
            ("Gust", 1755),
            ("Gustaaf", 4352),
            ("Guy", 11610),
            ("Hamza", 2370),
            ("Hannes", 1732),
            ("Hans", 9693),
            ("Harry", 1291),
            ("Hasan", 1560),
            ("Hassan", 1825),
            ("Hendrik", 6990),
            ("Henk", 1499),
            ("Henri", 8318),
            ("Henricus", 1528),
            ("Herman", 9236),
            ("Hubert", 4285),
            ("Hugo", 8287),
            ("Ian", 1954),
            ("Ibrahim", 3190),
            ("Ignace", 1783),
            ("Igor", 1052),
            ("Ilias", 2146),
            ("Ilyas", 1616),
            ("Imran", 1313),
            ("Ismail", 1821),
            ("Ivan", 6349),
            ("Ivo", 3879),
            ("Jaak", 1576),
            ("Jack", 2255),
            ("Jacob", 1159),
            ("Jacobus", 1852),
            ("Jacques", 7795),
            ("Jamal", 1125),
            ("James", 1671),
            ("Jan", 38107),
            ("Jari", 1111),
            ("Jarne", 3277),
            ("Jarno", 1610),
            ("Jason", 2377),
            ("Jasper", 6304),
            ("Jayden", 1595),
            ("Jean", 23651),
            ("Jean-Claude", 983),
            ("Jean-Marie", 1498),
            ("Jean-Paul", 1485),
            ("Jean-Pierre", 3944),
            ("Jef", 3956),
            ("Jeffrey", 2160),
            ("Jelle", 6886),
            ("Jens", 8330),
            ("Jeremy", 1045),
            ("Jeroen", 10446),
            ("Jesse", 2025),
            ("Jimmy", 3458),
            ("Jo", 3086),
            ("Joachim", 2548),
            ("Joannes", 4239),
            ("Jochen", 1826),
            ("Joeri", 4374),
            ("Johan", 29693),
            ("Johannes", 4491),
            ("John", 4592),
            ("Johnny", 4371),
            ("Johny", 2421),
            ("Jonas", 10866),
            ("Jonathan", 5528),
            ("Joost", 2193),
            ("Joppe", 1286),
            ("Joran", 1232),
            ("Jordi", 1308),
            ("Jordy", 3753),
            ("Joren", 2126),
            ("Joris", 7332),
            ("Jos", 2712),
            ("Joseph", 12355),
            ("Josephus", 3324),
            ("Joshua", 1110),
            ("Josse", 1054),
            ("José", 3827),
            ("Jozef", 21690),
            ("Joël", 1687),
            ("Juan", 1069),
            ("Jules", 5985),
            ("Julian", 1110),
            ("Julien", 6241),
            ("Jurgen", 5800),
            ("Juul", 1420),
            ("Jürgen", 3070),
            ("Kamiel", 1791),
            ("Karel", 8560),
            ("Karim", 1405),
            ("Karl", 1604),
            ("Kasper", 1792),
            ("Ken", 1966),
            ("Kenneth", 3834),
            ("Kenny", 4889),
            ("Kevin", 16042),
            ("Khalid", 1378),
            ("Kim", 2390),
            ("Kjell", 1139),
            ("Klaas", 1557),
            ("Kobe", 5551),
            ("Koen", 15022),
            ("Koenraad", 3456),
            ("Kris", 11238),
            ("Kristiaan", 1492),
            ("Kristof", 11645),
            ("Kurt", 10713),
            ("Lander", 3719),
            ("Lars", 6583),
            ("Laurens", 3704),
            ("Laurent", 3577),
            ("Leander", 1207),
            ("Lenn", 1396),
            ("Lennert", 2967),
            ("Leo", 5196),
            ("Leon", 6226),
            ("Leonard", 1093),
            ("Leopold", 2051),
            ("Levi", 1759),
            ("Lewis", 2070),
            ("Lex", 1073),
            ("Liam", 5622),
            ("Lieven", 4449),
            ("Lionel", 1116),
            ("Lode", 1814),
            ("Lodewijk", 2443),
            ("Lorenzo", 2320),
            ("Lou", 1780),
            ("Louis", 12930),
            ("Lowie", 3004),
            ("Loïc", 2094),
            ("Luc", 38634),
            ("Luca", 3120),
            ("Lucas", 8648),
            ("Lucien", 5158),
            ("Ludo", 5546),
            ("Ludovic", 1261),
            ("Ludovicus", 3950),
            ("Ludwig", 1671),
            ("Luka", 1250),
            ("Lukas", 3649),
            ("Léon", 1982),
            ("Maarten", 9051),
            ("Manuel", 1979),
            ("Marc", 43643),
            ("Marcel", 13967),
            ("Marco", 2452),
            ("Marcus", 1574),
            ("Mario", 7223),
            ("Mark", 3928),
            ("Marnix", 3209),
            ("Martijn", 1880),
            ("Martin", 4883),
            ("Mathias", 6267),
            ("Mathieu", 5061),
            ("Mathijs", 1477),
            ("Mathis", 4298),
            ("Mats", 3527),
            ("Matteo", 3595),
            ("Matthew", 1080),
            ("Matthias", 6831),
            ("Maurice", 5761),
            ("Maurits", 1498),
            ("Mauro", 2942),
            ("Max", 1913),
            ("Maxim", 5387),
            ("Maxime", 4105),
            ("Mehdi", 1105),
            ("Mehmet", 2458),
            ("Michael", 6907),
            ("Michaël", 5428),
            ("Michel", 13866),
            ("Michiel", 6741),
            ("Miel", 1529),
            ("Miguel", 1820),
            ("Mike", 3337),
            ("Milan", 5998),
            ("Milo", 27),
            ("Mohamad", 1154),
            ("Mohamed", 15295),
            ("Mohammad", 2663),
            ("Mohammed", 5837),
            ("Muhammed", 1339),
            ("Murat", 1161),
            ("Mustafa", 2946),
            ("Mustapha", 1096),
            ("Nand", 1350),
            ("Nathan", 4867),
            ("Nick", 8413),
            ("Nicky", 1155),
            ("Nico", 7393),
            ("Nicolas", 7088),
            ("Niels", 8462),
            ("Nils", 1702),
            ("Noah", 6152),
            ("Norbert", 2442),
            ("Noël", 3889),
            ("Oliver", 1394),
            ("Olivier", 7684),
            ("Omar", 2601),
            ("Omer", 1725),
            ("Oscar", 2771),
            ("Otis", 1173),
            ("Owen", 1167),
            ("Pascal", 8451),
            ("Patrick", 33673),
            ("Patrik", 998),
            ("Paul", 24369),
            ("Pedro", 1559),
            ("Peter", 31045),
            ("Petrus", 3264),
            ("Philip", 5372),
            ("Philippe", 10899),
            ("Pierre", 8633),
            ("Piet", 2415),
            ("Pieter", 14721),
            ("Pieter-Jan", 1875),
            ("Piotr", 983),
            ("Pol", 1080),
            ("Quinten", 4016),
            ("Rachid", 1901),
            ("Raf", 3681),
            ("Rafael", 1077),
            ("Rafaël", 1475),
            ("Raoul", 1454),
            ("Raphaël", 3121),
            ("Rayan", 2440),
            ("Raymond", 4361),
            ("Remi", 2370),
            ("Renaat", 1164),
            ("René", 8351),
            ("Richard", 2874),
            ("Rik", 5378),
            ("Rob", 2129),
            ("Robbe", 7095),
            ("Robby", 2174),
            ("Robert", 14189),
            ("Robin", 8624),
            ("Roel", 2794),
            ("Roeland", 1134),
            ("Roger", 16764),
            ("Roland", 6869),
            ("Romain", 2511),
            ("Ronald", 4742),
            ("Ronny", 10230),
            ("Rony", 2161),
            ("Roy", 1297),
            ("Ruben", 8565),
            ("Rudi", 8602),
            ("Rudolf", 2272),
            ("Rudy", 8506),
            ("Rune", 3126),
            ("Ryan", 1443),
            ("Said", 1810),
            ("Sam", 7843),
            ("Sami", 1089),
            ("Samir", 1268),
            ("Sammy", 1139),
            ("Samuel", 3007),
            ("Sander", 5928),
            ("Sebastiaan", 1599),
            ("Sebastian", 1685),
            ("Sem", 1937),
            ("Senne", 5283),
            ("Seppe", 6145),
            ("Serge", 4070),
            ("Siebe", 3031),
            ("Simon", 8202),
            ("Stan", 4075),
            ("Stef", 3647),
            ("Stefaan", 8349),
            ("Stefan", 8748),
            ("Stephan", 2496),
            ("Steve", 5305),
            ("Steven", 16397),
            ("Stijn", 13070),
            ("Stéphane", 1544),
            ("Sven", 9508),
            ("Sébastien", 1768),
            ("Theo", 2656),
            ("Theodoor", 986),
            ("Thibault", 2022),
            ("Thibaut", 1777),
            ("Thibo", 1716),
            ("Thierry", 4738),
            ("Thijs", 2452),
            ("Thomas", 17956),
            ("Thor", 1055),
            ("Tibe", 981),
            ("Tibo", 3054),
            ("Tijl", 1113),
            ("Tijs", 1174),
            ("Tim", 11570),
            ("Timo", 1107),
            ("Timothy", 2354),
            ("Tobias", 1007),
            ("Tom", 24659),
            ("Tomas", 1127),
            ("Tommy", 1707),
            ("Tony", 4293),
            ("Toon", 3210),
            ("Tristan", 2418),
            ("Tuur", 3209),
            ("Urbain", 1007),
            ("Valère", 1068),
            ("Vic", 3169),
            ("Victor", 8762),
            ("Viktor", 2836),
            ("Vince", 3617),
            ("Vincent", 8083),
            ("Walter", 9833),
            ("Wannes", 2817),
            ("Ward", 3667),
            ("Warre", 3098),
            ("Werner", 5470),
            ("Wesley", 4163),
            ("Wilfried", 5182),
            ("Wilhelmus", 986),
            ("Willem", 6532),
            ("William", 4555),
            ("Willy", 22434),
            ("Wim", 17900),
            ("Wout", 6567),
            ("Wouter", 8743),
            ("Xander", 3256),
            ("Xavier", 2971),
            ("Yannick", 4764),
            ("Yassin", 1078),
            ("Yassine", 1671),
            ("Younes", 1758),
            ("Youssef", 2678),
            ("Yusuf", 1559),
            ("Yvan", 2279),
            ("Yves", 9192),
            ("Zakaria", 1427),
            ("Ömer", 1032),
        )
    )

    # 500 most common female first names in Flanders in 2022 as per StatBel (with the same simplification as above)
    #
    # Data sourced from sheet 'Women' columns E-F ('Flanders') rows 3..502 (top-500 names incl. frequency)
    # of Excel workbook at https://statbel.fgov.be/sites/default/files/files/documents/bevolking/
    # 5.10%20Namen%20en%20voornamen/5.10.2%20Voornamen%20van%20vrouwen%20en%20mannen/First%20names_Population_2022.xlsx
    # and then sorted by name
    first_names_female = OrderedDict(
        (
            ("Adriana", 1763),
            ("Agnes", 7830),
            ("Agnès", 2084),
            ("Aicha", 1247),
            ("Alexandra", 3095),
            ("Alexia", 1248),
            ("Alice", 5331),
            ("Alicia", 1750),
            ("Alina", 1583),
            ("Aline", 3281),
            ("Amber", 7480),
            ("Amelia", 1371),
            ("Amelie", 1761),
            ("Amina", 2237),
            ("Amira", 1322),
            ("Amy", 1935),
            ("Amélie", 4038),
            ("An", 9756),
            ("Ana", 2305),
            ("Anaïs", 2451),
            ("Andrea", 3150),
            ("Angela", 2139),
            ("Angelina", 1360),
            ("Angèle", 1519),
            ("Anita", 9194),
            ("Anja", 8755),
            ("Anke", 3470),
            ("Ann", 24685),
            ("Anna", 20785),
            ("Anne", 14326),
            ("Anne-Marie", 5699),
            ("Anneke", 1526),
            ("Anneleen", 2312),
            ("Annelies", 8475),
            ("Annemie", 2219),
            ("Annette", 1258),
            ("Annick", 8828),
            ("Annie", 14154),
            ("Annita", 1896),
            ("Anny", 5227),
            ("Anouk", 3255),
            ("Antoinette", 1539),
            ("Ariane", 1271),
            ("Arlette", 4633),
            ("Astrid", 4271),
            ("Audrey", 1385),
            ("Augusta", 1394),
            ("Aurélie", 2379),
            ("Axelle", 3442),
            ("Aya", 1788),
            ("Ayse", 1422),
            ("Barbara", 4555),
            ("Beatrice", 1810),
            ("Beatrijs", 1718),
            ("Bernadette", 3418),
            ("Bertha", 1915),
            ("Betty", 2384),
            ("Bianca", 2994),
            ("Bieke", 1586),
            ("Birgit", 1375),
            ("Bo", 2784),
            ("Brenda", 2366),
            ("Brigitta", 1467),
            ("Brigitte", 8901),
            ("Britt", 4019),
            ("Camille", 3694),
            ("Carina", 4095),
            ("Carine", 15540),
            ("Carla", 3208),
            ("Carmen", 2203),
            ("Carolien", 1345),
            ("Carolina", 1417),
            ("Caroline", 11730),
            ("Catharina", 3593),
            ("Catherine", 4021),
            ("Cathy", 1612),
            ("Cecile", 2412),
            ("Cecilia", 2103),
            ("Celine", 1989),
            ("Chantal", 9244),
            ("Charlotte", 12818),
            ("Chiara", 1272),
            ("Chloé", 2171),
            ("Chloë", 1999),
            ("Christa", 3016),
            ("Christel", 10648),
            ("Christelle", 1545),
            ("Christiana", 4731),
            ("Christiane", 17117),
            ("Christianne", 1556),
            ("Christina", 3669),
            ("Christine", 14532),
            ("Cindy", 11454),
            ("Claire", 1818),
            ("Clara", 3004),
            ("Claudia", 2946),
            ("Claudine", 4178),
            ("Colette", 1394),
            ("Conny", 3428),
            ("Cornelia", 1800),
            ("Cynthia", 2343),
            ("Cécile", 1661),
            ("Céline", 3152),
            ("Daisy", 3936),
            ("Danielle", 3802),
            ("Daphne", 1210),
            ("Debbie", 1301),
            ("Debby", 1535),
            ("Deborah", 2125),
            ("Delphine", 3258),
            ("Denise", 8634),
            ("Diana", 5793),
            ("Diane", 5547),
            ("Dina", 2511),
            ("Dominique", 4974),
            ("Dorien", 3745),
            ("Dorine", 2365),
            ("Edith", 2445),
            ("Elena", 5449),
            ("Eliane", 2191),
            ("Elien", 4073),
            ("Elif", 1440),
            ("Eline", 7637),
            ("Elisa", 4405),
            ("Elisabeth", 9021),
            ("Elise", 8341),
            ("Elke", 8317),
            ("Ella", 5463),
            ("Ellen", 11498),
            ("Ellie", 1194),
            ("Elly", 1777),
            ("Els", 17283),
            ("Elsa", 1207),
            ("Elza", 4100),
            ("Emilia", 1925),
            ("Emilie", 2938),
            ("Emily", 2839),
            ("Emma", 11748),
            ("Erika", 2154),
            ("Erna", 4355),
            ("Estelle", 1365),
            ("Esther", 3304),
            ("Eva", 8072),
            ("Evelien", 5078),
            ("Eveline", 3692),
            ("Evelyn", 1556),
            ("Evelyne", 2313),
            ("Evi", 3578),
            ("Evy", 3710),
            ("Fabienne", 3801),
            ("Fanny", 1798),
            ("Farah", 1278),
            ("Fatiha", 1265),
            ("Fatima", 6713),
            ("Fatma", 1947),
            ("Febe", 3097),
            ("Femke", 5575),
            ("Fien", 5583),
            ("Fleur", 3977),
            ("Floor", 1788),
            ("Flore", 1622),
            ("Florence", 1884),
            ("Fran", 1865),
            ("Francine", 7139),
            ("Françoise", 2716),
            ("Freya", 1269),
            ("Frida", 1196),
            ("Frieda", 4009),
            ("Gabriella", 1245),
            ("Gabrielle", 2035),
            ("Gaëlle", 1234),
            ("Georgette", 4707),
            ("Gerarda", 1936),
            ("Gerda", 7709),
            ("Germaine", 2002),
            ("Ghislaine", 1827),
            ("Gilberte", 2703),
            ("Gina", 1618),
            ("Ginette", 1485),
            ("Gisèle", 1417),
            ("Gitte", 1719),
            ("Godelieva", 2176),
            ("Godelieve", 17791),
            ("Greet", 3041),
            ("Greta", 9149),
            ("Griet", 3493),
            ("Hanna", 1469),
            ("Hannah", 2750),
            ("Hanne", 7995),
            ("Hannelore", 2613),
            ("Heidi", 8882),
            ("Heleen", 1290),
            ("Helena", 7025),
            ("Helga", 1504),
            ("Henriette", 1968),
            ("Hilda", 7960),
            ("Hilde", 17845),
            ("Huguette", 1859),
            ("Hélène", 2283),
            ("Ida", 1608),
            ("Ilona", 1400),
            ("Ilse", 11320),
            ("Imane", 1459),
            ("Ine", 3595),
            ("Ines", 2874),
            ("Inge", 11783),
            ("Ingeborg", 1526),
            ("Ingrid", 17534),
            ("Irena", 1954),
            ("Iris", 4203),
            ("Irma", 2148),
            ("Irène", 3886),
            ("Isabel", 3258),
            ("Isabella", 1195),
            ("Isabelle", 9228),
            ("Ivonne", 1323),
            ("Jacqueline", 10639),
            ("Jade", 2951),
            ("Jana", 5996),
            ("Janne", 3022),
            ("Jasmine", 1230),
            ("Jeanine", 3445),
            ("Jeanne", 4925),
            ("Jeannette", 1395),
            ("Jeannine", 11588),
            ("Jennifer", 2280),
            ("Jenny", 6116),
            ("Jessica", 4869),
            ("Jessie", 1266),
            ("Jessy", 1261),
            ("Jill", 3436),
            ("Joanna", 5733),
            ("Johanna", 3543),
            ("Joke", 5645),
            ("Jolien", 8045),
            ("Josepha", 1804),
            ("Josephina", 3042),
            ("Josephine", 2406),
            ("Josiane", 2631),
            ("José", 1919),
            ("Josée", 3325),
            ("Joyce", 3143),
            ("Joëlle", 1354),
            ("Judith", 1892),
            ("Julia", 5958),
            ("Juliana", 1319),
            ("Julie", 14016),
            ("Julienne", 1955),
            ("Juliette", 4718),
            ("Justine", 2430),
            ("Kaat", 3341),
            ("Karen", 6671),
            ("Karima", 1399),
            ("Karin", 6450),
            ("Karina", 2522),
            ("Karine", 4250),
            ("Karolien", 2745),
            ("Katelijne", 1466),
            ("Kathleen", 8619),
            ("Kathy", 2864),
            ("Katia", 2251),
            ("Katja", 1249),
            ("Katleen", 3872),
            ("Kato", 3511),
            ("Katrien", 15525),
            ("Katrijn", 1680),
            ("Katty", 1392),
            ("Kelly", 9166),
            ("Khadija", 2140),
            ("Kiara", 1193),
            ("Kim", 9550),
            ("Kimberley", 1469),
            ("Kimberly", 3711),
            ("Kirsten", 2252),
            ("Krista", 1537),
            ("Kristel", 5525),
            ("Kristien", 3727),
            ("Kristina", 1337),
            ("Kristine", 1727),
            ("Lana", 1593),
            ("Lara", 3630),
            ("Laura", 13947),
            ("Laure", 2630),
            ("Lauren", 2295),
            ("Laurence", 2219),
            ("Lea", 5471),
            ("Leen", 6357),
            ("Lena", 6069),
            ("Leona", 1884),
            ("Leonie", 2239),
            ("Lien", 6602),
            ("Lies", 3903),
            ("Liesbet", 2741),
            ("Liesbeth", 6527),
            ("Lieselot", 1628),
            ("Lieve", 5027),
            ("Lieze", 1471),
            ("Liliana", 1328),
            ("Liliane", 11030),
            ("Lily", 2389),
            ("Lina", 4137),
            ("Linda", 17990),
            ("Linde", 2148),
            ("Lindsay", 1966),
            ("Lindsey", 1437),
            ("Lisa", 8518),
            ("Lise", 2585),
            ("Lisette", 5589),
            ("Liv", 2040),
            ("Lize", 1738),
            ("Loes", 1439),
            ("Lore", 7851),
            ("Lotte", 8633),
            ("Louisa", 3360),
            ("Louise", 8126),
            ("Lucia", 2420),
            ("Lucie", 2899),
            ("Lucienne", 3846),
            ("Luna", 4523),
            ("Lutgarde", 5667),
            ("Lutgardis", 1728),
            ("Lutgart", 1238),
            ("Lydia", 5032),
            ("Lydie", 1463),
            ("Lynn", 3638),
            ("Maaike", 2042),
            ("Madeleine", 2624),
            ("Magali", 1375),
            ("Magda", 8093),
            ("Magdalena", 5259),
            ("Malika", 1690),
            ("Manon", 3793),
            ("Marcella", 2144),
            ("Margareta", 2835),
            ("Margaretha", 2933),
            ("Margaux", 2149),
            ("Margo", 1333),
            ("Margot", 3965),
            ("Marguerite", 1891),
            ("Maria", 84005),
            ("Mariam", 1455),
            ("Marianne", 5561),
            ("Marie", 42082),
            ("Marie-Christine", 1877),
            ("Marie-Claire", 1642),
            ("Marie-Jeanne", 2256),
            ("Marie-José", 1621),
            ("Marie-Louise", 2805),
            ("Marie-Rose", 1742),
            ("Marie-Thérèse", 1490),
            ("Marieke", 2194),
            ("Mariette", 6488),
            ("Marijke", 7878),
            ("Marina", 9824),
            ("Marjan", 1185),
            ("Marleen", 22451),
            ("Marlies", 1873),
            ("Martha", 3796),
            ("Marthe", 2578),
            ("Martina", 1976),
            ("Martine", 23567),
            ("Maryam", 1486),
            ("Mathilde", 1237),
            ("Maud", 1765),
            ("Maya", 2065),
            ("Melanie", 1716),
            ("Melissa", 6335),
            ("Merel", 2546),
            ("Mia", 4243),
            ("Micheline", 2854),
            ("Michelle", 2995),
            ("Michèle", 2609),
            ("Mieke", 6414),
            ("Mila", 3734),
            ("Mira", 1381),
            ("Miranda", 1849),
            ("Mireille", 4106),
            ("Mona", 2133),
            ("Monica", 2508),
            ("Monika", 1845),
            ("Monique", 21567),
            ("Myriam", 8905),
            ("Nadia", 6318),
            ("Nadine", 10186),
            ("Nancy", 15781),
            ("Naomi", 2656),
            ("Natacha", 1501),
            ("Natalia", 1478),
            ("Natalie", 2209),
            ("Natascha", 1600),
            ("Natasja", 1546),
            ("Nathalie", 16550),
            ("Nele", 8801),
            ("Nelly", 3714),
            ("Nicole", 17332),
            ("Nina", 5383),
            ("Noa", 2399),
            ("Noor", 5253),
            ("Nora", 4463),
            ("Nore", 1887),
            ("Noëlla", 1906),
            ("Odette", 3077),
            ("Olga", 2195),
            ("Olivia", 4685),
            ("Oona", 1511),
            ("Pascale", 5080),
            ("Patricia", 11556),
            ("Paula", 9017),
            ("Paulien", 2158),
            ("Pauline", 3256),
            ("Peggy", 5095),
            ("Petra", 5704),
            ("Rachel", 2620),
            ("Rani", 1535),
            ("Rania", 1310),
            ("Rebecca", 1910),
            ("Regina", 2099),
            ("Reinhilde", 1270),
            ("Renée", 3546),
            ("Ria", 4661),
            ("Rita", 29651),
            ("Romy", 1764),
            ("Roos", 2194),
            ("Rosa", 7245),
            ("Rosalia", 1421),
            ("Rosalie", 1595),
            ("Rose", 1284),
            ("Rosette", 3094),
            ("Rosita", 1345),
            ("Ruth", 2455),
            ("Sabine", 9869),
            ("Sabrina", 4489),
            ("Sam", 1809),
            ("Samira", 2072),
            ("Sandra", 12403),
            ("Sandy", 2141),
            ("Sanne", 2921),
            ("Sara", 10099),
            ("Sarah", 15389),
            ("Saskia", 3904),
            ("Shana", 2539),
            ("Sharon", 2167),
            ("Sien", 2112),
            ("Sigrid", 2235),
            ("Silke", 4799),
            ("Silvia", 1265),
            ("Simone", 2938),
            ("Simonne", 6519),
            ("Sofia", 2446),
            ("Sofie", 16942),
            ("Sonia", 4859),
            ("Sonja", 9209),
            ("Sophia", 1491),
            ("Sophie", 8008),
            ("Stefanie", 6235),
            ("Steffi", 1527),
            ("Stella", 1401),
            ("Stephanie", 6152),
            ("Stien", 1413),
            ("Stéphanie", 2684),
            ("Suzanna", 2178),
            ("Suzanne", 4367),
            ("Suzy", 1245),
            ("Sylvia", 3160),
            ("Sylvie", 4659),
            ("Tamara", 4386),
            ("Tania", 3564),
            ("Tanja", 2048),
            ("Tatiana", 1474),
            ("Tess", 1848),
            ("Tessa", 3083),
            ("Theresia", 1250),
            ("Tina", 2437),
            ("Tine", 4832),
            ("Tineke", 1530),
            ("Tinne", 1398),
            ("Valentina", 1375),
            ("Valerie", 4085),
            ("Valérie", 3114),
            ("Vanessa", 6425),
            ("Veerle", 12048),
            ("Vera", 6550),
            ("Veronique", 4845),
            ("Vicky", 4617),
            ("Victoria", 2024),
            ("Virginie", 2389),
            ("Viviane", 7507),
            ("Véronique", 4908),
            ("Wendy", 11011),
            ("Yana", 2642),
            ("Yara", 1321),
            ("Yasmina", 1486),
            ("Yasmine", 2681),
            ("Yolanda", 1641),
            ("Yolande", 2231),
            ("Yvette", 4226),
            ("Yvonne", 4767),
            ("Zeynep", 1240),
            ("Zita", 1245),
            ("Zoë", 4381),
        )
    )

    # combination of *almost* 1000 common first names (not necesarilly *the most* common)
    # with totalled frequencies (for duplicate 'unisex'/gender-neutral first names, which cause < 1000 total names)
    first_names = OrderedDict(Counter(first_names_male) + Counter(first_names_female))

    # 500 most common last names in Flanders in 2022 as per StatBel (with the same simplification as above)
    #
    # Data sourced from sheet 'top10000_2022' columns E-F ('Flemish Region') rows 2..501 (top-500 names incl. frequency)
    # of Excel workbook at https://statbel.fgov.be/sites/default/files/files/documents/bevolking/
    # 5.10%20Namen%20en%20voornamen/5.10.1%20Familienamen/Family_names_2022.xlsx
    # and then sorted by name
    last_names = OrderedDict(
        (
            ("Adams", 1819),
            ("Adriaensen", 1794),
            ("Adriaenssens", 1504),
            ("Aerts", 11048),
            ("Ahmed", 1828),
            ("Ali", 1709),
            ("Andries", 3617),
            ("Baele", 1465),
            ("Baert", 5608),
            ("Baeten", 2798),
            ("Baetens", 2477),
            ("Baeyens", 3418),
            ("Bauwens", 4545),
            ("Beckers", 3518),
            ("Beeckman", 1618),
            ("Beernaert", 1727),
            ("Bekaert", 2894),
            ("Belmans", 1784),
            ("Berghmans", 1606),
            ("Bertels", 2106),
            ("Bervoets", 1449),
            ("Beyens", 1708),
            ("Blommaert", 1622),
            ("Blomme", 2701),
            ("Blondeel", 1704),
            ("Boeckx", 1741),
            ("Bogaert", 4869),
            ("Bogaerts", 4692),
            ("Bollen", 2750),
            ("Bonte", 2313),
            ("Boon", 1673),
            ("Boone", 2679),
            ("Boonen", 2804),
            ("Borremans", 1922),
            ("Bosmans", 4959),
            ("Bossuyt", 1809),
            ("Bouckaert", 1620),
            ("Bracke", 2832),
            ("Braem", 1728),
            ("Breugelmans", 1656),
            ("Broeckx", 1602),
            ("Brouwers", 1520),
            ("Bruggeman", 2025),
            ("Bruneel", 1769),
            ("Bruyninckx", 1902),
            ("Buelens", 2153),
            ("Buyse", 1865),
            ("Buysse", 2141),
            ("Callebaut", 2096),
            ("Callens", 3750),
            ("Callewaert", 2561),
            ("Carlier", 1649),
            ("Casier", 2001),
            ("Casteleyn", 1480),
            ("Celis", 1915),
            ("Ceulemans", 4170),
            ("Ceuppens", 1578),
            ("Christiaens", 4621),
            ("Claes", 13525),
            ("Claesen", 1617),
            ("Claessens", 4062),
            ("Claeys", 8904),
            ("Claus", 2974),
            ("Coene", 1633),
            ("Coenen", 2965),
            ("Colman", 1500),
            ("Colpaert", 1818),
            ("Cools", 6568),
            ("Coppens", 6811),
            ("Cornelis", 5113),
            ("Cox", 2056),
            ("Cuypers", 3783),
            ("Cuyvers", 1497),
            ("D'Hondt", 2261),
            ("D'hondt", 2348),
            ("Daems", 3788),
            ("Daniels", 2186),
            ("David", 1933),
            ("De Backer", 7534),
            ("De Baere", 1576),
            ("De Baets", 2439),
            ("De Bie", 2541),
            ("De Block", 2599),
            ("De Bock", 3582),
            ("De Boeck", 3266),
            ("De Bondt", 1938),
            ("De Bruycker", 1864),
            ("De Bruyn", 3613),
            ("De Bruyne", 3567),
            ("De Clerck", 3353),
            ("De Clercq", 8606),
            ("De Cock", 6486),
            ("De Coninck", 3926),
            ("De Coster", 3426),
            ("De Cuyper", 1980),
            ("De Decker", 3401),
            ("De Graeve", 2152),
            ("De Greef", 2161),
            ("De Groof", 1519),
            ("De Groote", 4200),
            ("De Jonghe", 2651),
            ("De Keyser", 3242),
            ("De Laet", 2589),
            ("De Leeuw", 1540),
            ("De Maeyer", 2447),
            ("De Meester", 1969),
            ("De Mey", 1789),
            ("De Meyer", 5698),
            ("De Moor", 2746),
            ("De Mulder", 1747),
            ("De Muynck", 1568),
            ("De Neve", 2219),
            ("De Paepe", 2664),
            ("De Pauw", 5349),
            ("De Ridder", 5214),
            ("De Roeck", 2468),
            ("De Roo", 1485),
            ("De Roover", 1628),
            ("De Ryck", 1897),
            ("De Rycke", 1923),
            ("De Schepper", 2894),
            ("De Schrijver", 1671),
            ("De Schryver", 1661),
            ("De Schutter", 1909),
            ("De Smedt", 6067),
            ("De Smet", 12094),
            ("De Sutter", 2295),
            ("De Vos", 9015),
            ("De Vuyst", 2116),
            ("De Wachter", 1990),
            ("De Waele", 3373),
            ("De Weerdt", 1761),
            ("De Wilde", 5511),
            ("De Winter", 3748),
            ("De Wit", 3129),
            ("De Witte", 3673),
            ("De Wolf", 3439),
            ("De Wulf", 1715),
            ("Debruyne", 2637),
            ("Deceuninck", 1681),
            ("Deckers", 3144),
            ("Declerck", 3534),
            ("Declercq", 5175),
            ("Decock", 2484),
            ("Deconinck", 1677),
            ("Decoster", 2351),
            ("Degroote", 1584),
            ("Degryse", 1840),
            ("Dejonghe", 1825),
            ("Dekeyser", 1693),
            ("Delaere", 1717),
            ("Deleu", 1719),
            ("Deman", 1443),
            ("Demeulenaere", 1596),
            ("Demeyer", 1538),
            ("Demeyere", 1696),
            ("Demuynck", 2011),
            ("Denys", 1891),
            ("Depoorter", 2084),
            ("Deprez", 3081),
            ("Depuydt", 1863),
            ("Derycke", 1466),
            ("Desmedt", 2222),
            ("Desmet", 7897),
            ("Devolder", 1613),
            ("Devos", 6499),
            ("Devriendt", 1699),
            ("Dewaele", 2548),
            ("Dewilde", 1463),
            ("Dewitte", 1877),
            ("Dewulf", 3221),
            ("Dhondt", 3055),
            ("Diallo", 2213),
            ("Dierckx", 4202),
            ("Dierick", 1605),
            ("Dierickx", 3156),
            ("Dillen", 2152),
            ("Dobbelaere", 2337),
            ("Dockx", 1644),
            ("Driesen", 2675),
            ("Dubois", 2344),
            ("Dupont", 3008),
            ("Eeckhout", 2874),
            ("Engelen", 3298),
            ("Engels", 2820),
            ("Everaert", 2537),
            ("Faes", 1904),
            ("Feys", 1941),
            ("Fierens", 1799),
            ("Fransen", 1852),
            ("François", 1921),
            ("Galle", 1526),
            ("Geens", 3337),
            ("Geeraerts", 2198),
            ("Geerts", 5208),
            ("Geudens", 1695),
            ("Geuens", 1451),
            ("Gevaert", 1941),
            ("Gielen", 3915),
            ("Gillis", 1520),
            ("Goethals", 4344),
            ("Goossens", 13062),
            ("Goovaerts", 1985),
            ("Goris", 3278),
            ("Govaerts", 2402),
            ("Goyvaerts", 1473),
            ("Hanssens", 1496),
            ("Hellemans", 2037),
            ("Helsen", 2867),
            ("Hendrickx", 8190),
            ("Hens", 1539),
            ("Heremans", 1890),
            ("Herman", 1790),
            ("Hermans", 10690),
            ("Heylen", 4692),
            ("Heymans", 2285),
            ("Heyvaert", 2032),
            ("Holvoet", 1710),
            ("Horemans", 2063),
            ("Hoste", 2300),
            ("Houben", 2883),
            ("Huybrechts", 3294),
            ("Huyghe", 2873),
            ("Huys", 1778),
            ("Huysmans", 3609),
            ("Jacobs", 16329),
            ("Jans", 3185),
            ("Jansen", 4650),
            ("Janssen", 7920),
            ("Janssens", 24733),
            ("Jonckheere", 2364),
            ("Joos", 1897),
            ("Joris", 2046),
            ("Kaur", 1465),
            ("Kaya", 1577),
            ("Kenis", 1694),
            ("Kerkhofs", 1771),
            ("Knockaert", 1772),
            ("Laenen", 2198),
            ("Lambert", 2205),
            ("Lambrecht", 2771),
            ("Lambrechts", 4989),
            ("Lammens", 2061),
            ("Laureys", 2065),
            ("Lauwers", 5422),
            ("Leemans", 3176),
            ("Lemmens", 7243),
            ("Lenaerts", 5136),
            ("Leroy", 1716),
            ("Leys", 2584),
            ("Leysen", 2439),
            ("Liekens", 1965),
            ("Lievens", 4245),
            ("Lippens", 2267),
            ("Loos", 1950),
            ("Luyckx", 2825),
            ("Luyten", 3962),
            ("Maertens", 2967),
            ("Maes", 21410),
            ("Mahieu", 2207),
            ("Malfait", 1535),
            ("Marien", 1706),
            ("Maris", 1506),
            ("Mariën", 3169),
            ("Martens", 9695),
            ("Matthijs", 1577),
            ("Matthys", 3658),
            ("Meersman", 1769),
            ("Meert", 2844),
            ("Mees", 1815),
            ("Meeus", 3197),
            ("Melis", 2127),
            ("Merckx", 2353),
            ("Mertens", 14720),
            ("Mestdagh", 2133),
            ("Meyers", 1437),
            ("Michiels", 9056),
            ("Michielsen", 2244),
            ("Moens", 4953),
            ("Moerman", 2608),
            ("Moonen", 1518),
            ("Moons", 3226),
            ("Moors", 2207),
            ("Moreels", 1598),
            ("Mortier", 3241),
            ("Naessens", 2265),
            ("Neirynck", 1653),
            ("Nelissen", 1513),
            ("Neven", 1501),
            ("Neyens", 1715),
            ("Nijs", 4206),
            ("Nuyts", 2984),
            ("Nys", 3433),
            ("Ooms", 2822),
            ("Op de Beeck", 2157),
            ("Parmentier", 2019),
            ("Pattyn", 2299),
            ("Pauwels", 11234),
            ("Peeters", 27625),
            ("Peters", 1978),
            ("Philips", 2285),
            ("Pieters", 4405),
            ("Poelmans", 1634),
            ("Polfliet", 1442),
            ("Pollet", 1811),
            ("Poppe", 2412),
            ("Praet", 1443),
            ("Proost", 2043),
            ("Put", 2001),
            ("Raes", 3810),
            ("Raeymaekers", 1843),
            ("Ramaekers", 1787),
            ("Renders", 2215),
            ("Reynaert", 1805),
            ("Reynders", 2948),
            ("Roelandt", 2111),
            ("Roels", 3982),
            ("Rogiers", 1978),
            ("Rombaut", 1756),
            ("Rombouts", 1850),
            ("Roose", 2039),
            ("Roosen", 1812),
            ("Rosseel", 2022),
            ("Rottiers", 1472),
            ("Rutten", 2007),
            ("Sabbe", 1842),
            ("Saelens", 1487),
            ("Samyn", 1875),
            ("Sanders", 1835),
            ("Schepens", 2550),
            ("Schepers", 3019),
            ("Schoofs", 1907),
            ("Segers", 7482),
            ("Seghers", 1784),
            ("Sels", 1913),
            ("Simoens", 1551),
            ("Simons", 4259),
            ("Singh", 3338),
            ("Slegers", 1619),
            ("Smeets", 3356),
            ("Smet", 6103),
            ("Smets", 8517),
            ("Smits", 3291),
            ("Smolders", 2118),
            ("Snauwaert", 1467),
            ("Somers", 3832),
            ("Staelens", 2138),
            ("Standaert", 1654),
            ("Stas", 1464),
            ("Sterckx", 2923),
            ("Stevens", 7609),
            ("Steyaert", 2016),
            ("Stroobants", 1829),
            ("Swennen", 1490),
            ("Swinnen", 3550),
            ("Tack", 2361),
            ("Tanghe", 2436),
            ("Temmerman", 3033),
            ("Theunis", 1766),
            ("Theys", 1794),
            ("Thijs", 4532),
            ("Thomas", 2242),
            ("Thys", 5018),
            ("Tielemans", 1496),
            ("Timmerman", 1909),
            ("Timmermans", 4855),
            ("Torfs", 2719),
            ("Troch", 1524),
            ("Vaes", 2814),
            ("Van Acker", 4075),
            ("Van Assche", 2446),
            ("Van Baelen", 1574),
            ("Van Belle", 2339),
            ("Van Bogaert", 1455),
            ("Van Camp", 3020),
            ("Van Campenhout", 1796),
            ("Van Daele", 3406),
            ("Van Damme", 8481),
            ("Van Dessel", 2283),
            ("Van Dijck", 2138),
            ("Van Dooren", 1528),
            ("Van Driessche", 2524),
            ("Van Dyck", 4872),
            ("Van Genechten", 1711),
            ("Van Gestel", 2018),
            ("Van Goethem", 3348),
            ("Van Gorp", 2203),
            ("Van Hecke", 4109),
            ("Van Herck", 2701),
            ("Van Hoecke", 1722),
            ("Van Hoof", 4179),
            ("Van Hove", 3609),
            ("Van Hulle", 1529),
            ("Van Impe", 1864),
            ("Van Laer", 1981),
            ("Van Laere", 2410),
            ("Van Landeghem", 1728),
            ("Van Lierde", 1613),
            ("Van Loo", 2240),
            ("Van Loock", 1665),
            ("Van Loon", 1937),
            ("Van Looveren", 2207),
            ("Van Looy", 2892),
            ("Van Nuffel", 1719),
            ("Van Parys", 1453),
            ("Van Poucke", 1583),
            ("Van Praet", 1585),
            ("Van Raemdonck", 1482),
            ("Van Rompaey", 2437),
            ("Van Roy", 2441),
            ("Van de Velde", 7761),
            ("Van de Voorde", 2658),
            ("Van de Walle", 2445),
            ("Van den Bergh", 3421),
            ("Van den Berghe", 2788),
            ("Van den Bosch", 2429),
            ("Van den Bossche", 4470),
            ("Van den Brande", 2969),
            ("Van den Broeck", 6977),
            ("Van den Eynde", 3352),
            ("Van der Auwera", 1931),
            ("Van der Linden", 1484),
            ("Vanbrabant", 1465),
            ("Vandaele", 2747),
            ("Vandamme", 3788),
            ("Vandecasteele", 2198),
            ("Vandekerckhove", 2540),
            ("Vandenberghe", 5021),
            ("Vandenbroucke", 2912),
            ("Vandenbussche", 3018),
            ("Vandendriessche", 2965),
            ("Vandeputte", 3010),
            ("Vanderheyden", 1781),
            ("Vanderlinden", 1672),
            ("Vanderstraeten", 2213),
            ("Vandevelde", 3340),
            ("Vandewalle", 3435),
            ("Vanhee", 1884),
            ("Vanhoutte", 3206),
            ("Vanhove", 3263),
            ("Vanneste", 3563),
            ("Vansteenkiste", 2351),
            ("Verbeeck", 3620),
            ("Verbeke", 5173),
            ("Verbiest", 1767),
            ("Verbist", 2413),
            ("Verboven", 2295),
            ("Verbruggen", 3835),
            ("Verbrugghe", 1491),
            ("Vercammen", 4018),
            ("Vercauteren", 3750),
            ("Vercruysse", 3532),
            ("Verdonck", 3722),
            ("Vereecke", 1832),
            ("Vereecken", 2011),
            ("Verelst", 2490),
            ("Verfaillie", 1755),
            ("Vergauwen", 1682),
            ("Verhaegen", 3890),
            ("Verhaeghe", 4339),
            ("Verhelst", 4056),
            ("Verheyden", 3138),
            ("Verheyen", 5656),
            ("Verhoeven", 6452),
            ("Verhulst", 3410),
            ("Verlinden", 4955),
            ("Vermeersch", 4688),
            ("Vermeir", 1914),
            ("Vermeire", 2870),
            ("Vermeiren", 5030),
            ("Vermeulen", 11300),
            ("Vermeylen", 1624),
            ("Verschueren", 4870),
            ("Verstappen", 1545),
            ("Verstraete", 5731),
            ("Verstraeten", 4484),
            ("Vervaeke", 1575),
            ("Vervaet", 1716),
            ("Vervoort", 3912),
            ("Verwimp", 1676),
            ("Viaene", 2290),
            ("Vinck", 1661),
            ("Vissers", 1852),
            ("Vlaeminck", 1525),
            ("Vleugels", 1475),
            ("Voet", 1635),
            ("Vos", 3337),
            ("Vrancken", 2096),
            ("Vranckx", 1635),
            ("Vranken", 2538),
            ("Wauters", 5713),
            ("Wellens", 2311),
            ("Willaert", 2140),
            ("Willekens", 1889),
            ("Willems", 14709),
            ("Wouters", 13352),
            ("Wuyts", 4945),
            ("Wynants", 1604),
            ("Yildirim", 1560),
            ("Yilmaz", 2779),
        )
    )
