﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appsync/AppSyncRequest.h>
#include <aws/appsync/AppSync_EXPORTS.h>
#include <aws/appsync/model/EventConfig.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace AppSync {
namespace Model {

/**
 */
class CreateApiRequest : public AppSyncRequest {
 public:
  AWS_APPSYNC_API CreateApiRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateApi"; }

  AWS_APPSYNC_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name for the <code>Api</code>.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateApiRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The owner contact information for the <code>Api</code>.</p>
   */
  inline const Aws::String& GetOwnerContact() const { return m_ownerContact; }
  inline bool OwnerContactHasBeenSet() const { return m_ownerContactHasBeenSet; }
  template <typename OwnerContactT = Aws::String>
  void SetOwnerContact(OwnerContactT&& value) {
    m_ownerContactHasBeenSet = true;
    m_ownerContact = std::forward<OwnerContactT>(value);
  }
  template <typename OwnerContactT = Aws::String>
  CreateApiRequest& WithOwnerContact(OwnerContactT&& value) {
    SetOwnerContact(std::forward<OwnerContactT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateApiRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateApiRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Event API configuration. This includes the default authorization
   * configuration for connecting, publishing, and subscribing to an Event API.</p>
   */
  inline const EventConfig& GetEventConfig() const { return m_eventConfig; }
  inline bool EventConfigHasBeenSet() const { return m_eventConfigHasBeenSet; }
  template <typename EventConfigT = EventConfig>
  void SetEventConfig(EventConfigT&& value) {
    m_eventConfigHasBeenSet = true;
    m_eventConfig = std::forward<EventConfigT>(value);
  }
  template <typename EventConfigT = EventConfig>
  CreateApiRequest& WithEventConfig(EventConfigT&& value) {
    SetEventConfig(std::forward<EventConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_ownerContact;

  Aws::Map<Aws::String, Aws::String> m_tags;

  EventConfig m_eventConfig;
  bool m_nameHasBeenSet = false;
  bool m_ownerContactHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_eventConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace AppSync
}  // namespace Aws
