export type JSONOkParseResult = OkParseResult<DocumentNode>;
export type JSONParseResult = ParseResult<DocumentNode>;
export type JSONLanguageOptions = {
    /**
     * Whether to allow trailing commas in JSONC mode.
     */
    allowTrailingCommas?: boolean;
};
export type NoDuplicateKeysMessageIds = "duplicateKey";
export type NoDuplicateKeysRuleDefinition = JSONRuleDefinition<{
    MessageIds: NoDuplicateKeysMessageIds;
}>;
export type NoEmptyKeysMessageIds = "emptyKey";
export type NoEmptyKeysRuleDefinition = JSONRuleDefinition<{
    MessageIds: NoEmptyKeysMessageIds;
}>;
export type NoUnnormalizedKeysMessageIds = "unnormalizedKey";
export type NoUnnormalizedKeysOptions = {
    form: string;
};
export type NoUnnormalizedKeysRuleDefinition = JSONRuleDefinition<{
    RuleOptions: [NoUnnormalizedKeysOptions];
    MessageIds: NoUnnormalizedKeysMessageIds;
}>;
export type NoUnsafeValuesMessageIds = "unsafeNumber" | "unsafeInteger" | "unsafeZero" | "subnormal" | "loneSurrogate";
export type NoUnsafeValuesRuleDefinition = JSONRuleDefinition<{
    MessageIds: NoUnsafeValuesMessageIds;
}>;
export type SortOptions = {
    caseSensitive: boolean;
    natural: boolean;
    minKeys: number;
    allowLineSeparatedGroups: boolean;
};
export type SortKeysMessageIds = "sortKeys";
export type SortDirection = "asc" | "desc";
export type SortKeysRuleOptions = [SortDirection, SortOptions];
export type SortKeysRuleDefinition = JSONRuleDefinition<{
    RuleOptions: SortKeysRuleOptions;
    MessageIds: SortKeysMessageIds;
}>;
export type Comparator = (a: string, b: string) => boolean;
export type TopLevelInteropMessageIds = "topLevel";
export type TopLevelInteropRuleDefinition = JSONRuleDefinition<{
    MessageIds: TopLevelInteropMessageIds;
}>;
/**
 * @fileoverview The JSONLanguage class.
 * @author Nicholas C. Zakas
 */
/**
 *
 * @typedef {OkParseResult<DocumentNode>} JSONOkParseResult
 * @typedef {ParseResult<DocumentNode>} JSONParseResult
 *
 * @typedef {Object} JSONLanguageOptions
 * @property {boolean} [allowTrailingCommas] Whether to allow trailing commas in JSONC mode.
 */
/**
 * JSON Language Object
 * @implements {Language<{ LangOptions: JSONLanguageOptions; Code: JSONSourceCode; RootNode: DocumentNode; Node: AnyNode }>}
 */
export class JSONLanguage implements Language {
    /**
     * Creates a new instance.
     * @param {Object} options The options to use for this instance.
     * @param {"json"|"jsonc"|"json5"} options.mode The parser mode to use.
     */
    constructor({ mode }: {
        mode: "json" | "jsonc" | "json5";
    });
    /**
     * The type of file to read.
     * @type {"text"}
     */
    fileType: "text";
    /**
     * The line number at which the parser starts counting.
     * @type {0|1}
     */
    lineStart: 0 | 1;
    /**
     * The column number at which the parser starts counting.
     * @type {0|1}
     */
    columnStart: 0 | 1;
    /**
     * The name of the key that holds the type of the node.
     * @type {string}
     */
    nodeTypeKey: string;
    /**
     * The visitor keys.
     * @type {Record<string, string[]>}
     */
    visitorKeys: Record<string, string[]>;
    /**
     * Validates the language options.
     * @param {JSONLanguageOptions} languageOptions The language options to validate.
     * @returns {void}
     * @throws {Error} When the language options are invalid.
     */
    validateLanguageOptions(languageOptions: JSONLanguageOptions): void;
    /**
     * Parses the given file into an AST.
     * @param {File} file The virtual file to parse.
     * @param {{languageOptions: JSONLanguageOptions}} context The options to use for parsing.
     * @returns {JSONParseResult} The result of parsing.
     */
    parse(file: File, context: {
        languageOptions: JSONLanguageOptions;
    }): JSONParseResult;
    /**
     * Creates a new `JSONSourceCode` object from the given information.
     * @param {File} file The virtual file to create a `JSONSourceCode` object from.
     * @param {JSONOkParseResult} parseResult The result returned from `parse()`.
     * @returns {JSONSourceCode} The new `JSONSourceCode` object.
     */
    createSourceCode(file: File, parseResult: JSONOkParseResult): JSONSourceCode;
    #private;
}
/**
 * JSON Source Code Object
 * @extends {TextSourceCodeBase<{LangOptions: JSONLanguageOptions, RootNode: DocumentNode, SyntaxElementWithLoc: JSONSyntaxElement, ConfigNode: Token}>}
 */
export class JSONSourceCode extends TextSourceCodeBase<{
    LangOptions: JSONLanguageOptions;
    RootNode: DocumentNode;
    SyntaxElementWithLoc: JSONSyntaxElement;
    ConfigNode: Token;
}> {
    /**
     * Creates a new instance.
     * @param {Object} options The options for the instance.
     * @param {string} options.text The source code text.
     * @param {DocumentNode} options.ast The root AST node.
     */
    constructor({ text, ast }: {
        text: string;
        ast: DocumentNode;
    });
    /**
     * The comment tokens in the source code.
     * @type {Array<Token>|undefined}
     */
    comments: Array<Token> | undefined;
    /**
     * Returns an array of all inline configuration nodes found in the
     * source code.
     * @returns {Array<Token>} An array of all inline configuration nodes.
     */
    getInlineConfigNodes(): Array<Token>;
    /**
     * Returns directives that enable or disable rules along with any problems
     * encountered while parsing the directives.
     * @returns {{problems:Array<FileProblem>,directives:Array<Directive>}} Information
     *      that ESLint needs to further process the directives.
     */
    getDisableDirectives(): {
        problems: Array<FileProblem>;
        directives: Array<Directive>;
    };
    /**
     * Returns inline rule configurations along with any problems
     * encountered while parsing the configurations.
     * @returns {{problems:Array<FileProblem>,configs:Array<{config:{rules:RulesConfig},loc:SourceLocation}>}} Information
     *      that ESLint needs to further process the rule configurations.
     */
    applyInlineConfig(): {
        problems: Array<FileProblem>;
        configs: Array<{
            config: {
                rules: RulesConfig;
            };
            loc: SourceLocation;
        }>;
    };
    /**
     * Returns the parent of the given node.
     * @param {AnyNode} node The node to get the parent of.
     * @returns {AnyNode|undefined} The parent of the node.
     */
    getParent(node: AnyNode): AnyNode | undefined;
    /**
     * Traverse the source code and return the steps that were taken.
     * @returns {Iterable<JSONTraversalStep>} The steps that were taken while traversing the source code.
     */
    traverse(): Iterable<JSONTraversalStep>;
    /**
     * Gets the token before the given node or token, optionally including comments.
     * @param {AnyNode|Token} nodeOrToken The node or token to get the previous token for.
     * @param {Object} [options] Options object.
     * @param {boolean} [options.includeComments] If true, return comments when they are present.
     * @returns {Token|null} The previous token or comment, or null if there is none.
     */
    getTokenBefore(nodeOrToken: AnyNode | Token, { includeComments }?: {
        includeComments?: boolean;
    }): Token | null;
    /**
     * Gets the token after the given node or token, skipping any comments unless includeComments is true.
     * @param {AnyNode|Token} nodeOrToken The node or token to get the next token for.
     * @param {Object} [options] Options object.
     * @param {boolean} [options.includeComments=false] If true, return comments when they are present.
     * @returns {Token|null} The next token or comment, or null if there is none.
     */
    getTokenAfter(nodeOrToken: AnyNode | Token, { includeComments }?: {
        includeComments?: boolean;
    }): Token | null;
    #private;
}
declare namespace plugin {
    export namespace meta {
        let name: string;
        let version: string;
    }
    export namespace languages {
        let json: JSONLanguage;
        let jsonc: JSONLanguage;
        let json5: JSONLanguage;
    }
    export { rules };
    export namespace configs {
        namespace recommended {
            export let plugins: {};
            export { rules$1 as rules };
        }
    }
}
import type { DocumentNode } from "@humanwhocodes/momoa";
import type { OkParseResult } from "@eslint/core";
import type { ParseResult } from "@eslint/core";
import type { JSONRuleDefinition } from "./types.ts";
import type { Language } from "@eslint/core";
import type { File } from "@eslint/core";
import type { JSONSyntaxElement } from "./types.ts";
import type { Token } from "@humanwhocodes/momoa";
import { TextSourceCodeBase } from '@eslint/plugin-kit';
import type { FileProblem } from "@eslint/core";
import { Directive } from '@eslint/plugin-kit';
import type { RulesConfig } from "@eslint/core";
import type { SourceLocation } from "@eslint/core";
import type { AnyNode } from "@humanwhocodes/momoa";
/**
 * A class to represent a step in the traversal process.
 */
declare class JSONTraversalStep extends VisitNodeStep {
    /**
     * Creates a new instance.
     * @param {Object} options The options for the step.
     * @param {AnyNode} options.target The target of the step.
     * @param {1|2} options.phase The phase of the step.
     * @param {Array<any>} options.args The arguments of the step.
     */
    constructor({ target, phase, args }: {
        target: AnyNode;
        phase: 1 | 2;
        args: Array<any>;
    });
    /**
     * The target of the step.
     * @type {AnyNode}
     */
    target: AnyNode;
}
declare var rules: {
    "no-duplicate-keys": NoDuplicateKeysRuleDefinition;
    "no-empty-keys": NoEmptyKeysRuleDefinition;
    "no-unnormalized-keys": NoUnnormalizedKeysRuleDefinition;
    "no-unsafe-values": NoUnsafeValuesRuleDefinition;
    "sort-keys": SortKeysRuleDefinition;
    "top-level-interop": TopLevelInteropRuleDefinition;
};
declare const rules$1: {
    readonly "json/no-duplicate-keys": "error";
    readonly "json/no-empty-keys": "error";
    readonly "json/no-unnormalized-keys": "error";
    readonly "json/no-unsafe-values": "error";
};
import { VisitNodeStep } from '@eslint/plugin-kit';
export { plugin as default };
